﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace ParseExtension.RegularExpressions
{
    class ParserRegexMatch
    {
        private ParseToken[][] matchedTokens;
        private Dictionary<string, ParseToken[]> nameTokens;

        public bool IsSuccess
        {
            get;
            private set;
        }

        public ParserRegex Regex
        {
            get;
            private set;
        }

        public ParseToken[] Tokens
        {
            get;
            private set;
        }

        public ParseToken[] this[int index]
        {
            get
            {
                return matchedTokens[index];
            }
        }

        public ParseToken[] this[string name]
        {
            get
            {
                return nameTokens[name];
            }
        }

        public int StartIndex
        {
            get;
            private set;
        }

        public int Length
        {
            get;
            private set;
        }

        public ParserRegexMatch(ParserRegex regex, ParseToken[] tokens, ParserRegexBackTrack backTrack)
        {
            Regex = regex;
            Tokens = tokens;
            IsSuccess = backTrack != null;
            if (IsSuccess)
            {
                Initialize(backTrack);
            }
        }

        private void Initialize(ParserRegexBackTrack backTrack)
        {
            Dictionary<ParserRegexExpression, List<int>> usedIndexes =
                new Dictionary<ParserRegexExpression, List<int>>();
            ParserRegexBackTrack current = backTrack;
            int min = int.MaxValue, max = int.MinValue;
            while (true)
            {
                if (current.BackTrack == null)
                {
                    break;
                }

                if (current.Expression != null)
                {
                    List<int> indexes;
                    if (!usedIndexes.TryGetValue(current.Expression, out indexes))
                    {
                        indexes = new List<int>();
                        usedIndexes.Add(current.Expression, indexes);
                    }
                    indexes.Add(current.UsedIndex);
                    if (current.UsedIndex < min)
                    {
                        min = current.UsedIndex;
                    }
                    if (max < current.UsedIndex)
                    {
                        max = current.UsedIndex;
                    }
                }

                current = current.BackTrack;
            }
            StartIndex = min;
            Length = max - min + 1;

            foreach (List<int> usedIndex in usedIndexes.Values)
            {
                usedIndex.Reverse();
            }

            nameTokens = new Dictionary<string, ParseToken[]>();
            List<ParseToken[]> tempTokens = new List<ParseToken[]>();
            foreach (ParserRegexExpression expression in Regex.Expressions)
            {
                List<int> indexes;
                ParseToken[] subToken;
                if (!usedIndexes.TryGetValue(expression, out indexes))
                {
                    subToken = new ParseToken[0];
                }
                else
                {
                    subToken = new ParseToken[indexes[indexes.Count - 1] - indexes[0] + 1];
                    Array.Copy(Tokens, indexes[0], subToken, 0, subToken.Length);
                }
                tempTokens.Add(subToken);
                if (!String.IsNullOrEmpty(expression.Name) && !nameTokens.ContainsKey(expression.Name))
                {
                    nameTokens.Add(expression.Name, subToken);
                }
            }
            matchedTokens = tempTokens.ToArray();
        }

        public override string ToString()
        {
            if (IsSuccess)
            {
                return Tokens.Join(StartIndex, Length);
            }
            return base.ToString();
        }
    }
}
