﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;

namespace ParseExtension
{
	class TexWordChecker : IWordChecker
	{
		delegate bool Command(NotepadNeueExtension.CheckWordEventArgs e);

		static Regex beginRegex = new Regex("\\\\begin\\{(?<content>[^{}\r\n]+)\\}");
		static Regex endRegex = new Regex("\\\\end\\{(?<content>[^{}\r\n]+)\\}");
		static Regex bibItemRegex = new Regex("\\\\bibitem\\{(?<content>[^{}\r\n]+)\\}");
		static Regex labelRegex = new Regex("\\\\label\\{(?<content>[^{}\r\n]+)\\}");
		static Dictionary<string, Command> commandList = new Dictionary<string, Command>()
		{
			{"end",EndCommand},
			{"cite",CiteCommand},
			{"ref",RefCommand},
			{"pageref",PagerefCommand},
			{"eqref",ErefCommand},
		};

		#region IWordChecker メンバ

		public bool CheckWord(NotepadNeueExtension.CheckWordEventArgs e)
		{
			if (e.CaretIndex == e.Range.Start && e.CaretIndex == e.Range.End && !e.Document.ShouldBeIgnoredGrammatically(e.CaretIndex))
			{
				int startIndex = e.CaretIndex - 1;
				for (int i = startIndex; i >= 0; i--)
				{
					if (Char.IsLetterOrDigit(e.Document[i]))
					{
						continue;
					}
					else
					{
						if (e.Document[i] == '\\' && i < startIndex)
						{
							string str = e.Document.GetTextInRange(i + 1, e.CaretIndex);
							foreach (KeyValuePair<string, Command> command in commandList)
							{
								if (command.Key.StartsWith(str))
								{
									return command.Value(e);
								}
							}
						}
						break;
					}
				}
			}

			return false;
		}

		private static bool EndCommand(NotepadNeueExtension.CheckWordEventArgs e)
		{
			int lineIndex = e.Document.GetLineIndexFromCharIndex(e.CaretIndex);
			int lineHeadIndex = e.Document.GetLineHeadIndex(lineIndex);

			string currentLine = e.Document.GetTextInRange(lineHeadIndex, e.CaretIndex);
			MatchCollection beginMatchColl = beginRegex.Matches(currentLine), endMatchColl = endRegex.Matches(currentLine);

			List<MatchInfo> matches = new List<MatchInfo>();
			foreach (Match match in beginMatchColl)
			{
				matches.Add(new MatchInfo() { Match = match, Regex = beginRegex });
			}
			foreach (Match match in endMatchColl)
			{
				matches.Add(new MatchInfo() { Match = match, Regex = endRegex });
			}
			matches.Sort((match1, match2) => match1.Match.Index - match2.Match.Index);

			int count = 0;
			foreach (MatchInfo matchInfo in matches)
			{
				if (matchInfo.Regex == endRegex)
				{
					count++;
				}
				else
				{
					count--;

					if (count == -1)
					{
						Add("\\end" + "{" + matchInfo.Match.Groups["content"].Value + "}", e);
						return true;
					}
				}
			}

			while (lineIndex > 0)
			{
				lineIndex--;
				matches.Clear();

				currentLine = e.Document.GetLineContent(lineIndex);
				beginMatchColl = beginRegex.Matches(currentLine);
				endMatchColl = endRegex.Matches(currentLine);

				foreach (Match match in beginMatchColl)
				{
					matches.Add(new MatchInfo() { Match = match, Regex = beginRegex });
				}
				foreach (Match match in endMatchColl)
				{
					matches.Add(new MatchInfo() { Match = match, Regex = endRegex });
				}
				matches.Sort((match1, match2) => match1.Match.Index - match2.Match.Index);

				foreach (MatchInfo matchInfo in matches)
				{
					if (matchInfo.Regex == endRegex)
					{
						count++;
					}
					else
					{
						count--;
						if (count == -1)
						{
							Add("\\end" + "{" + matchInfo.Match.Groups["content"].Value + "}", e);
							return true;
						}
					}
				}
			}

			return false;
		}

		private static bool CommandBase(Regex regex, string commandName, NotepadNeueExtension.CheckWordEventArgs e)
		{
			bool add = false;
			foreach (Match match in regex.Matches(e.Text))
			{
				Add(String.Format("\\{0}", commandName) + "{" + match.Groups["content"].Value + "}", e);
				add = true;
			}

			return add;
		}

		private static bool CiteCommand(NotepadNeueExtension.CheckWordEventArgs e)
		{
			return CommandBase(bibItemRegex, "cite", e);
		}

		private static bool RefCommand(NotepadNeueExtension.CheckWordEventArgs e)
		{
			return CommandBase(labelRegex, "ref", e);
		}

		private static bool PagerefCommand(NotepadNeueExtension.CheckWordEventArgs e)
		{
			return CommandBase(labelRegex, "pageref", e);
		}

		private static bool ErefCommand(NotepadNeueExtension.CheckWordEventArgs e)
		{
			return CommandBase(labelRegex, "eqref", e);
		}

		private static void Add(string text, NotepadNeueExtension.CheckWordEventArgs e)
		{
			e.AssistInformations.Add(new NotepadNeueExtension.AssistInformation()
			{
				Text = text,
				HintText = text,
				Summary = "",
				Image = null
			});
			e.SelectIndex = 0;
		}

		#endregion
	}

	class MatchInfo
	{
		public Match Match
		{
			get;
			set;
		}

		public Regex Regex
		{
			get;
			set;
		}
	}
}
