/**
 * @file ntmcmd.c
 * @author Shinichiro Nakamura
 * @brief コマンド解釈層の実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 * Inspired by M, Murakami
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmcmd.h"
#include "ntmuart.h"
#include "ntmconf.h"
#include "ntmopt.h"
#include "ntmlibc.h"
#include "ntmcmd_read.h"
#include "ntmcmd_write.h"
#include "ntmcmd_dump.h"
#include "ntmcmd_copy.h"
#include "ntmcmd_verify.h"
#include "ntmcmd_fill.h"
#include "ntmcmd_scroll.h"

static const char *DISPTEXT_USAGE         = "(Debug commands)\r\n";
static const char *DISPTEXT_USAGE_READ    = " Read    : RB|RW|RL [ADDR]\r\n";
static const char *DISPTEXT_USAGE_WRITE   = " Write   : WB|WW|WL [ADDR] [DATA]\r\n";
static const char *DISPTEXT_USAGE_DUMP    = " Dump    : DB|DW|DL [ADDR] (LEN)\r\n";
static const char *DISPTEXT_USAGE_COPY    = " Copy    : CB|CW|CL [ADDR-SRC] [ADDR-DES] [LEN]\r\n";
static const char *DISPTEXT_USAGE_FILL    = " Fill    : FB|FW|FL [ADDR] [LEN] [DATA|TYPE]  (TYPE: inc, dec, rand, bitwalk)\r\n";
static const char *DISPTEXT_USAGE_VERIFY  = " Verify  : VB|VW|VL [ADDR-SRC] [ADDR-SRC] [LEN]\r\n";
static const char *DISPTEXT_USAGE_SCROLL  = " Scroll  : SB|SW|SL (ADDR)\r\n";

static const char *DISPTEXT_NUMFMT       = "(Number formats)\r\n";
static const char *DISPTEXT_NUMFMT_BIN   = " Binary      : b01010101 or 01010101b\r\n";
static const char *DISPTEXT_NUMFMT_OCT   = " Octal       : o01234567 or 01234567o\r\n";
static const char *DISPTEXT_NUMFMT_DEC   = " Decimal     : d12345678 or 12345678d\r\n";
static const char *DISPTEXT_NUMFMT_HEX   = " Hexadecimal : h1234ABCD or 1234ABCDh\r\n";

/**
 * @brief ntmoptモジュールから呼ばれるコールバック関数。
 * @details
 * ユーザから渡って来たコマンド文字列をntmoptモジュールに渡すと、
 * パースした結果と共にこの関数を呼んでくれるようになっている。
 *
 * @param argc 引数の数。
 * @param argv 引数。
 * @param extobj 外部オブジェクト。
 *
 * @return ntmoptモジュールに返す値。
 */
static int ntmopt_callback(int argc, char **argv, void *extobj)
{
    /*
     * 何の入力もなかった場合は何もしない。
     */
    if (argc == 0) {
        return 0;
    }

    /*
     * コマンドを実行する。
     */
#if (NTMCONF_ENABLE_READ == 1)
    if (ntmcmd_read(argc, argv, extobj)) {
        return 0;
    }
#endif
#if (NTMCONF_ENABLE_WRITE == 1)
    if (ntmcmd_write(argc, argv, extobj)) {
        return 0;
    }
#endif
#if (NTMCONF_ENABLE_DUMP == 1)
    if (ntmcmd_dump(argc, argv, extobj)) {
        return 0;
    }
#endif
#if (NTMCONF_ENABLE_FILL == 1)
    if (ntmcmd_fill(argc, argv, extobj)) {
        return 0;
    }
#endif
#if (NTMCONF_ENABLE_COPY == 1)
    if (ntmcmd_copy(argc, argv, extobj)) {
        return 0;
    }
#endif
#if (NTMCONF_ENABLE_VERIFY == 1)
    if (ntmcmd_verify(argc, argv, extobj)) {
        return 0;
    }
#endif
#if (NTMCONF_ENABLE_SCROLL == 1)
    if (ntmcmd_scroll(argc, argv, extobj)) {
        return 0;
    }
#endif

    /*
     * 対応しているコマンドを提示する。
     */
    ntmuart_write(DISPTEXT_USAGE, ntmlibc_strlen(DISPTEXT_USAGE));
#if (NTMCONF_ENABLE_READ == 1)
    ntmuart_write(DISPTEXT_USAGE_READ, ntmlibc_strlen(DISPTEXT_USAGE_READ));
#endif
#if (NTMCONF_ENABLE_WRITE == 1)
    ntmuart_write(DISPTEXT_USAGE_WRITE, ntmlibc_strlen(DISPTEXT_USAGE_WRITE));
#endif
#if (NTMCONF_ENABLE_DUMP == 1)
    ntmuart_write(DISPTEXT_USAGE_DUMP, ntmlibc_strlen(DISPTEXT_USAGE_DUMP));
#endif
#if (NTMCONF_ENABLE_FILL == 1)
    ntmuart_write(DISPTEXT_USAGE_FILL, ntmlibc_strlen(DISPTEXT_USAGE_FILL));
#endif
#if (NTMCONF_ENABLE_COPY == 1)
    ntmuart_write(DISPTEXT_USAGE_COPY, ntmlibc_strlen(DISPTEXT_USAGE_COPY));
#endif
#if (NTMCONF_ENABLE_VERIFY == 1)
    ntmuart_write(DISPTEXT_USAGE_VERIFY, ntmlibc_strlen(DISPTEXT_USAGE_VERIFY));
#endif
#if (NTMCONF_ENABLE_SCROLL == 1)
    ntmuart_write(DISPTEXT_USAGE_SCROLL, ntmlibc_strlen(DISPTEXT_USAGE_SCROLL));
#endif

    /*
     * 対応している数値フォーマットを提示する。
     */
    ntmuart_write(DISPTEXT_NUMFMT, ntmlibc_strlen(DISPTEXT_NUMFMT));
    ntmuart_write(DISPTEXT_NUMFMT_BIN, ntmlibc_strlen(DISPTEXT_NUMFMT_BIN));
    ntmuart_write(DISPTEXT_NUMFMT_OCT, ntmlibc_strlen(DISPTEXT_NUMFMT_OCT));
    ntmuart_write(DISPTEXT_NUMFMT_DEC, ntmlibc_strlen(DISPTEXT_NUMFMT_DEC));
    ntmuart_write(DISPTEXT_NUMFMT_HEX, ntmlibc_strlen(DISPTEXT_NUMFMT_HEX));

    return 0;
}

/**
 * @brief コマンドを実行する。
 * @details ntshellモジュールのコールバックという位置付けである。
 *
 * @param s ユーザから渡って来たコマンド文字列。
 * @param extobj 外部オブジェクト。
 *
 * @return ntshellモジュールに返す値。
 */
int ntmcmd_execute(const char *s, void *extobj)
{
    return ntmopt_parse(s, ntmopt_callback, extobj);
}

