/**
 * @file ntmcmd_read.c
 * @author Shinichiro Nakamura
 * @brief readコマンドの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 * Inspired by M, Murakami
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmcmd_read.h"
#include "ntmuart.h"
#include "ntmlibc.h"
#include "ntmio.h"
#include "ntmconf.h"
#include "ntmcmd_common.h"

#if (NTMCONF_ENABLE_READ == 1)

static const char *DISPTEXT_USAGE_RB = "RB [ADDR]\r\n";
static const char *DISPTEXT_USAGE_RW = "RW [ADDR]\r\n";
static const char *DISPTEXT_USAGE_RL = "RL [ADDR]\r\n";

static const char *DISPTEXT_HEAD_BIN = "BIN: ";
static const char *DISPTEXT_HEAD_OCT = "OCT: ";
static const char *DISPTEXT_HEAD_DEC = "DEC: ";
static const char *DISPTEXT_HEAD_HEX = "HEX: ";

#define SPACE(N)    do { int i; for (i = 0; i < N; i++) { ntmuart_write(" ", 1); } } while (0)

/**
 * @brief READコマンド。
 *
 * @retval 0 与えられたパラメータはREADコマンドでなかった。
 * @retval 1 与えられたパラメータをREADコマンドとして処理した。
 */
int ntmcmd_read(int argc, char **argv, void *extobj)
{
    NtmAccessMode am = NtmAccessModeDefault;
    uint32_t adr;
    NtmStrType adr_type;
    char string[16];

    /*
     * コマンドからアクセスモードを決定する。
     */
    if ((ntmlibc_strcmp(argv[0], "rb") == 0) || (ntmlibc_strcmp(argv[0], "RB") == 0)) {
        am = NtmAccessModeByte;
    } else if ((ntmlibc_strcmp(argv[0], "rw") == 0) || (ntmlibc_strcmp(argv[0], "RW") == 0)) {
        am = NtmAccessModeWord;
    } else if ((ntmlibc_strcmp(argv[0], "rl") == 0) || (ntmlibc_strcmp(argv[0], "RL") == 0)) {
        am = NtmAccessModeLong;
    } else {
        /*
         * 与えられたコマンドは、このモジュールのコマンドに該当しなかった。
         */
        return 0;
    }

    /*
     * パラメータ数を検証する。
     */
    if (argc != 2) {
        switch (am) {
            case NtmAccessModeByte:
                ntmuart_write(DISPTEXT_USAGE_RB, ntmlibc_strlen(DISPTEXT_USAGE_RB));
                return 1;
            case NtmAccessModeWord:
                ntmuart_write(DISPTEXT_USAGE_RW, ntmlibc_strlen(DISPTEXT_USAGE_RW));
                return 1;
            case NtmAccessModeLong:
                ntmuart_write(DISPTEXT_USAGE_RL, ntmlibc_strlen(DISPTEXT_USAGE_RL));
                return 1;
            case NtmAccessModeDefault:
                ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
                return 1;
            default:
                ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
                return 1;
        }
    }

    /*
     * ユーザの入力を取得する。
     */
    ntmstr_string_to_number(argv[1], &adr, &adr_type, NTMCONF_DEFAULT_STRTYPE);

    /*
     * ユーザからの入力を検証する。
     * 不正な指定が有った場合、エラーメッセージを出力して終了する。
     */
    if (adr_type == NtmStrTypeInvalid) {
        ntmuart_write(NTMCMD_COMMON_TEXT_INVADR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVADR));
        return 1;
    }

    /*
     * アクセスモードに応じて処理を実行する。
     */
    switch (am) {
        case NtmAccessModeByte:
            {
                uint8_t data;
                /*
                 * 指定アドレスのデータを読み込む。
                 */
                ntmio_read_byte(adr, &data);

                /*
                 * Bin, Oct, Dec, Hexで値を表示する。
                 */
                ntmstr_number_to_string(data, &string[0], 8, NtmStrTypeBin);
                ntmuart_write(DISPTEXT_HEAD_BIN, ntmlibc_strlen(DISPTEXT_HEAD_BIN));
                ntmuart_write(string, ntmlibc_strlen(string));
                ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));

                ntmstr_number_to_string(data, &string[0], 3, NtmStrTypeOct);
                ntmuart_write(DISPTEXT_HEAD_OCT, ntmlibc_strlen(DISPTEXT_HEAD_OCT));
                SPACE(5);
                ntmuart_write(string, ntmlibc_strlen(string));
                ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));

                ntmstr_number_to_string(data, &string[0], 3, NtmStrTypeDec);
                ntmuart_write(DISPTEXT_HEAD_DEC, ntmlibc_strlen(DISPTEXT_HEAD_DEC));
                SPACE(5);
                ntmuart_write(string, ntmlibc_strlen(string));
                ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));

                ntmstr_number_to_string(data, &string[0], 2, NtmStrTypeHex);
                ntmuart_write(DISPTEXT_HEAD_HEX, ntmlibc_strlen(DISPTEXT_HEAD_HEX));
                SPACE(6);
                ntmuart_write(string, ntmlibc_strlen(string));
                ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));
            }
            break;
        case NtmAccessModeWord:
            {
                uint16_t data;
                /*
                 * 指定アドレスのデータを読み込む。
                 */
                ntmio_read_word(adr, &data);

                /*
                 * Bin, Oct, Dec, Hexで値を表示する。
                 */
                ntmstr_number_to_string(data, &string[0], 16, NtmStrTypeBin);
                ntmuart_write(DISPTEXT_HEAD_BIN, ntmlibc_strlen(DISPTEXT_HEAD_BIN));
                ntmuart_write(string, ntmlibc_strlen(string));
                ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));

                ntmstr_number_to_string(data, &string[0], 6, NtmStrTypeOct);
                ntmuart_write(DISPTEXT_HEAD_OCT, ntmlibc_strlen(DISPTEXT_HEAD_OCT));
                SPACE(10);
                ntmuart_write(string, ntmlibc_strlen(string));
                ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));

                ntmstr_number_to_string(data, &string[0], 5, NtmStrTypeDec);
                ntmuart_write(DISPTEXT_HEAD_DEC, ntmlibc_strlen(DISPTEXT_HEAD_DEC));
                SPACE(11);
                ntmuart_write(string, ntmlibc_strlen(string));
                ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));

                ntmstr_number_to_string(data, &string[0], 4, NtmStrTypeHex);
                ntmuart_write(DISPTEXT_HEAD_HEX, ntmlibc_strlen(DISPTEXT_HEAD_HEX));
                SPACE(12);
                ntmuart_write(string, ntmlibc_strlen(string));
                ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));
            }
            break;
        case NtmAccessModeLong:
            {
                uint32_t data;
                /*
                 * 指定アドレスのデータを読み込む。
                 */
                ntmio_read_long(adr, &data);

                /*
                 * Bin, Oct, Dec, Hexで値を表示する。
                 */
                ntmstr_number_to_string(data, &string[0], 32, NtmStrTypeBin);
                ntmuart_write(DISPTEXT_HEAD_BIN, ntmlibc_strlen(DISPTEXT_HEAD_BIN));
                ntmuart_write(string, ntmlibc_strlen(string));
                ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));

                ntmstr_number_to_string(data, &string[0], 11, NtmStrTypeOct);
                ntmuart_write(DISPTEXT_HEAD_OCT, ntmlibc_strlen(DISPTEXT_HEAD_OCT));
                SPACE(21);
                ntmuart_write(string, ntmlibc_strlen(string));
                ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));

                ntmstr_number_to_string(data, &string[0], 10, NtmStrTypeDec);
                ntmuart_write(DISPTEXT_HEAD_DEC, ntmlibc_strlen(DISPTEXT_HEAD_DEC));
                SPACE(22);
                ntmuart_write(string, ntmlibc_strlen(string));
                ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));

                ntmstr_number_to_string(data, &string[0], 8, NtmStrTypeHex);
                ntmuart_write(DISPTEXT_HEAD_HEX, ntmlibc_strlen(DISPTEXT_HEAD_HEX));
                SPACE(24);
                ntmuart_write(string, ntmlibc_strlen(string));
                ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));
            }
            break;
        default:
            ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
            break;
    }
    return 1;
}

#endif

