/**
 * @file ntmcmd_scroll.c
 * @author Shinichiro Nakamura
 * @brief SCROLLコマンドの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 * Inspired by M, Murakami
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "vtsend.h"
#include "vtrecv.h"
#include "ntmuart.h"
#include "ntmstr.h"
#include "ntmlibc.h"
#include "ntmio.h"
#include "ntmcmd_scroll.h"
#include "ntmcmd_common.h"
#include "ntmattr.h"

#if (NTMCONF_ENABLE_SCROLL == 1)

#define SCR_WIDTH   80
#define SCR_HEIGHT  40

#define SCRTOP  2
#define SCRBTM  (SCR_HEIGHT - 1)

typedef struct {
    NtmAccessMode mode;
    vtsend_t vtsend;
    vtrecv_t vtrecv;
    uint32_t offset;
    int loop;
} work_t;

static work_t work;

static void cbfunc(struct vtrecv *p, vtrecv_action_t action, unsigned char c);
static void dump_byte(const uint32_t addr);
static void dump_word(const uint32_t addr);
static void dump_long(const uint32_t addr);

static void cbfunc(struct vtrecv *p, vtrecv_action_t action, unsigned char c)
{
    int i;
    switch (action) {
        case VTRECV_ACTION_CSI_DISPATCH:
            if (c == 0x41) {
                /*
                 * 画面をスクロールさせて１行だけ書き換える。
                 */
                if (16 <= work.offset) {
                    work.offset-=16;

                    vtsend_cursor_position(&work.vtsend, 1, SCRTOP);
                    ntmuart_write("\x1b", 1);
                    ntmuart_write("M", 1);

                    vtsend_cursor_position(&work.vtsend, 1, SCRTOP);
                    switch (work.mode) {
                        case NtmAccessModeByte:
                            dump_byte(work.offset);
                            break;
                        case NtmAccessModeWord:
                            dump_word(work.offset);
                            break;
                        case NtmAccessModeLong:
                            dump_long(work.offset);
                            break;
                    }
                }
            }
            if (c == 0x42) {
                /*
                 * 画面をスクロールさせて１行だけ書き換える。
                 */
                if (work.offset <= 0xFFFFFFFF - 16) {
                    work.offset+=16;

                    vtsend_cursor_position(&work.vtsend, 1, SCRBTM);
                    ntmuart_write("\x1b", 1);
                    ntmuart_write("D", 1);

                    vtsend_cursor_position(&work.vtsend, 1, SCRBTM);
                    switch (work.mode) {
                        case NtmAccessModeByte:
                            dump_byte(work.offset + (16 * (SCRBTM - SCRTOP)));
                            break;
                        case NtmAccessModeWord:
                            dump_word(work.offset + (16 * (SCRBTM - SCRTOP)));
                            break;
                        case NtmAccessModeLong:
                            dump_long(work.offset + (16 * (SCRBTM - SCRTOP)));
                            break;
                    }
                }
            }
            if (c == 0x44) {
                /*
                 * 画面全体を書き換えても良いのだが、
                 * スクロールしたい数だけスクロールさせた方が表示更新が早い。
                 */
                int i;
                for (i = 0; i < SCRBTM - SCRTOP + 1; i++) {
                    if (16 <= work.offset) {
                        work.offset-=16;

                        vtsend_cursor_position(&work.vtsend, 1, SCRTOP);
                        ntmuart_write("\x1b", 1);
                        ntmuart_write("M", 1);

                        vtsend_cursor_position(&work.vtsend, 1, SCRTOP);
                        switch (work.mode) {
                            case NtmAccessModeByte:
                                dump_byte(work.offset);
                                break;
                            case NtmAccessModeWord:
                                dump_word(work.offset);
                                break;
                            case NtmAccessModeLong:
                                dump_long(work.offset);
                                break;
                        }
                    }
                }
            }
            if (c == 0x43) {
                /*
                 * 画面全体を書き換えても良いのだが、
                 * スクロールしたい数だけスクロールさせた方が表示更新が早い。
                 */
                int i;
                for (i = 0; i < SCRBTM - SCRTOP + 1; i++) {
                    if (work.offset <= 0xFFFFFFFF - 16) {
                        work.offset+=16;

                        vtsend_cursor_position(&work.vtsend, 1, SCRBTM);
                        ntmuart_write("\x1b", 1);
                        ntmuart_write("D", 1);

                        vtsend_cursor_position(&work.vtsend, 1, SCRBTM);
                        switch (work.mode) {
                            case NtmAccessModeByte:
                                dump_byte(work.offset + (16 * (SCRBTM - SCRTOP)));
                                break;
                            case NtmAccessModeWord:
                                dump_word(work.offset + (16 * (SCRBTM - SCRTOP)));
                                break;
                            case NtmAccessModeLong:
                                dump_long(work.offset + (16 * (SCRBTM - SCRTOP)));
                                break;
                        }
                    }
                }
            }
        case VTRECV_ACTION_PRINT:
            if (c == 0x71) {
                work.loop = 0;
                vtsend_reset(&work.vtsend);
                vtsend_erase_display(&work.vtsend);
            }
            break;
        default:
            {
                /*
                 * ディスプレイを消去してフレームを描画する。
                 */
                vtsend_erase_display(&work.vtsend);
                vtsend_draw_box(&work.vtsend, 1, 1, SCR_WIDTH, SCRBTM + 1);

                /*
                 * スクロール範囲を設定する。
                 */
                vtsend_set_scroll_region(&work.vtsend, SCRTOP, SCRBTM);

                /*
                 * 画面全体を更新する。
                 */
                for (i = 0; i < SCRBTM - SCRTOP + 1; i++) {
                    vtsend_cursor_position(&work.vtsend, 1, SCRTOP + i);
                    switch (work.mode) {
                        case NtmAccessModeByte:
                            dump_byte(work.offset + (i * 16));
                            break;
                        case NtmAccessModeWord:
                            dump_word(work.offset + (i * 16));
                            break;
                        case NtmAccessModeLong:
                            dump_long(work.offset + (i * 16));
                            break;
                    }
                }
            }
            break;
    }
}

static void dump_byte(const uint32_t addr)
{
    uint32_t len = 16;
    uint32_t i;
    uint8_t data;
    char string[16];
    char ascii[24];

    /*
     * メモリ情報を読み込む。
     */
    ntmattr_info_t meminfo;
    ntmattr_get_info(addr, &meminfo);

    /*
     * ヘッダを表示する。
     */
    ntmstr_number_to_string(addr, &string[0], 8, NtmStrTypeHex);

    ntmuart_write(string, ntmlibc_strlen(string));
    ntmuart_write(" | ", 3);

    /*
     * 指定された長さに応じて処理を実行する。
     */
    for (i = 0; i < len; i++) {
        if (meminfo.attr != NTMATTR_ATTR_RSV) {
            /*
             * 指定アドレスのデータを読み込む。
             */
            ntmio_read_byte(addr + i, &data);

            /*
             * ASCII表示用にデータを用意する。
             */
            ascii[i % 16] = ntmstr_isprint(data) ? data : '.';

            /*
             * 値を表示する。
             */
            ntmstr_number_to_string(data, &string[0], 2, NtmStrTypeHex);
            ntmuart_write(string, ntmlibc_strlen(string));
        } else {
            /*
             * ASCII表示用にデータを用意する。
             */
            ascii[i % 16] = ' ';

            /*
             * 値を表示する。
             */
            ntmuart_write("--", 2);
        }
        /*
         * 値と値の間に空白を作る。
         */
        if ((i % 0x10) == 0x7) {
            ntmuart_write(" - ", 3);
        } else {
            ntmuart_write(" ", 1);
        }
    }
    ascii[16] = '\0';

    /*
     * 行末ASCII表示を加えて改行する。
     */
    ntmuart_write("| ", 2);
    ntmuart_write(ascii, ntmlibc_strlen(ascii));
    ntmuart_write(" ", 1);
}

static void dump_word(const uint32_t addr)
{
    uint32_t len = 16;
    uint32_t i;
    uint8_t data;
    char string[16];
    char ascii[24];

    /*
     * メモリ情報を読み込む。
     */
    ntmattr_info_t meminfo;
    ntmattr_get_info(addr, &meminfo);

    /*
     * ヘッダを表示する。
     */
    ntmstr_number_to_string(addr, &string[0], 8, NtmStrTypeHex);

    ntmuart_write(string, ntmlibc_strlen(string));
    ntmuart_write(" | ", 3);

    /*
     * 指定された長さに応じて処理を実行する。
     */
    for (i = 0; i < len / 2; i++) {
        if (meminfo.attr != NTMATTR_ATTR_RSV) {
            /*
             * 指定アドレスのデータを読み込む。
             */
            ntmio_read_byte(addr + (i * 2), &data);

            /*
             * ASCII表示用にデータを用意する。
             */
            ascii[((i * 2) % 16) + 0] = ntmstr_isprint((uint8_t)(data >> 0)) ? ((uint8_t)(data >> 0)) : '.';
            ascii[((i * 2) % 16) + 1] = ntmstr_isprint((uint8_t)(data >> 8)) ? ((uint8_t)(data >> 8)) : '.';

            /*
             * 値を表示する。
             */
            ntmstr_number_to_string(data, &string[0], 4, NtmStrTypeHex);
            ntmuart_write(string, ntmlibc_strlen(string));
        } else {
            /*
             * ASCII表示用にデータを用意する。
             */
            ascii[((i * 2) % 16) + 0] = ' ';
            ascii[((i * 2) % 16) + 1] = ' ';

            /*
             * 値を表示する。
             */
            ntmuart_write("----", 4);
        }
        /*
         * 値と値の間に空白を作る。
         */
        if (((i * 2) % 0x10) == 0x6) {
            ntmuart_write(" - ", 3);
        } else {
            ntmuart_write(" ", 1);
        }
    }
    ascii[16] = '\0';

    /*
     * 行末ASCII表示を加えて改行する。
     */
    ntmuart_write("| ", 2);
    ntmuart_write(ascii, ntmlibc_strlen(ascii));
    ntmuart_write(" ", 1);
}

static void dump_long(const uint32_t addr)
{
    uint32_t len = 16;
    uint32_t i;
    uint8_t data;
    char string[16];
    char ascii[24];

    /*
     * メモリ情報を読み込む。
     */
    ntmattr_info_t meminfo;
    ntmattr_get_info(addr, &meminfo);

    /*
     * ヘッダを表示する。
     */
    ntmstr_number_to_string(addr, &string[0], 8, NtmStrTypeHex);

    ntmuart_write(string, ntmlibc_strlen(string));
    ntmuart_write(" | ", 3);

    /*
     * 指定された長さに応じて処理を実行する。
     */
    for (i = 0; i < len / 4; i++) {
        if (meminfo.attr != NTMATTR_ATTR_RSV) {
            /*
             * 指定アドレスのデータを読み込む。
             */
            ntmio_read_byte(addr + (i * 4), &data);

            /*
             * ASCII表示用にデータを用意する。
             */
            ascii[((i * 4) % 16) + 0] = ntmstr_isprint((uint8_t)(data >>  0)) ? ((uint8_t)(data >>  0)) : '.';
            ascii[((i * 4) % 16) + 1] = ntmstr_isprint((uint8_t)(data >>  8)) ? ((uint8_t)(data >>  8)) : '.';
            ascii[((i * 4) % 16) + 2] = ntmstr_isprint((uint8_t)(data >> 16)) ? ((uint8_t)(data >> 16)) : '.';
            ascii[((i * 4) % 16) + 3] = ntmstr_isprint((uint8_t)(data >> 24)) ? ((uint8_t)(data >> 24)) : '.';

            /*
             * 値を表示する。
             */
            ntmstr_number_to_string(data, &string[0], 8, NtmStrTypeHex);
            ntmuart_write(string, ntmlibc_strlen(string));
        } else {
            /*
             * ASCII表示用にデータを用意する。
             */
            ascii[((i * 4) % 16) + 0] = ' ';
            ascii[((i * 4) % 16) + 1] = ' ';
            ascii[((i * 4) % 16) + 2] = ' ';
            ascii[((i * 4) % 16) + 3] = ' ';

            /*
             * 値を表示する。
             */
            ntmuart_write("--------", 8);
        }
        /*
         * 値と値の間に空白を作る。
         */
        if (((i * 4) % 0x10) == 0x4) {
            ntmuart_write(" - ", 3);
        } else {
            ntmuart_write(" ", 1);
        }
    }
    ascii[16] = '\0';

    /*
     * 行末ASCII表示を加えて改行する。
     */
    ntmuart_write("| ", 2);
    ntmuart_write(ascii, ntmlibc_strlen(ascii));
    ntmuart_write(" ", 1);
}

int ntmcmd_scroll(int argc, char **argv, void *extobj)
{
    NtmAccessMode am = NtmAccessModeDefault;
    uint32_t adr = 0x00000000;
    NtmStrType adr_type = NtmStrTypeInvalid;
    int i;

    /*
     * コマンドからアクセスモードを決定する。
     */
    if ((ntmlibc_strcmp(argv[0], "sb") == 0) || (ntmlibc_strcmp(argv[0], "SB") == 0)) {
        am = NtmAccessModeByte;
    } else if ((ntmlibc_strcmp(argv[0], "sw") == 0) || (ntmlibc_strcmp(argv[0], "SW") == 0)) {
        am = NtmAccessModeWord;
    } else if ((ntmlibc_strcmp(argv[0], "sl") == 0) || (ntmlibc_strcmp(argv[0], "SL") == 0)) {
        am = NtmAccessModeLong;
    } else {
        /*
         * 与えられたコマンドは、このモジュールのコマンドに該当しなかった。
         */
        return 0;
    }

    if (argc == 2) {
        /*
         * ユーザの入力を取得する。
         */
        ntmstr_string_to_number(argv[1], &adr, &adr_type, NTMCONF_DEFAULT_STRTYPE);

        /*
         * ユーザからの入力を検証する。
         * 不正な指定が有った場合、エラーメッセージを出力して終了する。
         */
        if (adr_type == NtmStrTypeInvalid) {
            ntmuart_write(NTMCMD_COMMON_TEXT_INVADR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVADR));
            return 1;
        }
    }

    /*
     * Workの各種フィールドを初期化する。
     */
    vtsend_init(&work.vtsend, ntmuart_write);
    vtrecv_init(&work.vtrecv, cbfunc);
    work.mode = am;
    work.offset = adr;
    work.loop = 1;

    /*
     * ディスプレイを消去してフレームを描画する。
     */
    vtsend_erase_display(&work.vtsend);
    vtsend_draw_box(&work.vtsend, 1, 1, SCR_WIDTH, SCRBTM + 1);

    /*
     * スクロール範囲を設定する。
     */
    vtsend_set_scroll_region(&work.vtsend, SCRTOP, SCRBTM);

    /*
     * 画面全体を更新する。
     */
    for (i = 0; i < SCRBTM - SCRTOP + 1; i++) {
        vtsend_cursor_position(&work.vtsend, 1, SCRTOP + i);
        switch (work.mode) {
            case NtmAccessModeByte:
                dump_byte(work.offset + (i * 16));
                break;
            case NtmAccessModeWord:
                dump_word(work.offset + (i * 16));
                break;
            case NtmAccessModeLong:
                dump_long(work.offset + (i * 16));
                break;
        }
    }

    /*
     * ユーザ入力ループ。
     */
    vtsend_set_cursor(&work.vtsend, 0);
    while (work.loop) {
        char c;
        ntmuart_read(&c, 1);
        vtrecv_execute(&work.vtrecv, &c, 1);
    }
    vtsend_set_cursor(&work.vtsend, 1);

    return 1;
}

#endif

