/**
 * @file ntmcmd_verify.c
 * @author Shinichiro Nakamura
 * @brief verifyコマンドの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 * Inspired by M, Murakami
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmcmd_verify.h"
#include "ntmuart.h"
#include "ntmlibc.h"
#include "ntmio.h"
#include "ntmconf.h"
#include "ntmcmd_common.h"

#if (NTMCONF_ENABLE_VERIFY == 1)

static const char *DISPTEXT_USAGE_VB    = "VB [ADDR-SRC] [ADDR-SRC] [LEN]\r\n";
static const char *DISPTEXT_USAGE_VW    = "VW [ADDR-SRC] [ADDR-SRC] [LEN]\r\n";
static const char *DISPTEXT_USAGE_VL    = "VL [ADDR-SRC] [ADDR-SRC] [LEN]\r\n";

static const char *DISPTEXT_DIFF_HEADER = "Address1\tAddress2\tData1   \tData2   \r\n";

/**
 * @brief VERIFYコマンド。
 *
 * @retval 0 与えられたパラメータはVERIFYコマンドでなかった。
 * @retval 1 与えられたパラメータをVERIFYコマンドとして処理した。
 */
int ntmcmd_verify(int argc, char **argv, void *extobj)
{
    NtmAccessMode am = NtmAccessModeDefault;
    uint32_t adr1;
    uint32_t adr2;
    uint32_t len;
    NtmStrType adr1_type;
    NtmStrType adr2_type;
    NtmStrType len_type;
    uint32_t cnt = 0;
    char string[16];
    uint32_t i;

    /*
     * コマンドからアクセスモードを決定する。
     */
    if ((ntmlibc_strcmp(argv[0], "vb") == 0) || (ntmlibc_strcmp(argv[0], "VB") == 0)) {
        am = NtmAccessModeByte;
    } else if ((ntmlibc_strcmp(argv[0], "vw") == 0) || (ntmlibc_strcmp(argv[0], "VW") == 0)) {
        am = NtmAccessModeWord;
    } else if ((ntmlibc_strcmp(argv[0], "vl") == 0) || (ntmlibc_strcmp(argv[0], "VL") == 0)) {
        am = NtmAccessModeLong;
    } else {
        /*
         * 与えられたコマンドは、このモジュールのコマンドに該当しなかった。
         */
        return 0;
    }

    /*
     * パラメータ数を検証する。
     */
    if (argc != 4) {
        switch (am) {
            case NtmAccessModeByte:
                ntmuart_write(DISPTEXT_USAGE_VB, ntmlibc_strlen(DISPTEXT_USAGE_VB));
                return 1;
            case NtmAccessModeWord:
                ntmuart_write(DISPTEXT_USAGE_VW, ntmlibc_strlen(DISPTEXT_USAGE_VW));
                return 1;
            case NtmAccessModeLong:
                ntmuart_write(DISPTEXT_USAGE_VL, ntmlibc_strlen(DISPTEXT_USAGE_VL));
                return 1;
            case NtmAccessModeDefault:
                ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
                return 1;
            default:
                ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
                return 1;
        }
    }

    /*
     * ユーザの入力を取得する。
     */
    ntmstr_string_to_number(argv[1], &adr1, &adr1_type, NTMCONF_DEFAULT_STRTYPE);
    ntmstr_string_to_number(argv[2], &adr2, &adr2_type, NTMCONF_DEFAULT_STRTYPE);
    ntmstr_string_to_number(argv[3], &len, &len_type, NTMCONF_DEFAULT_STRTYPE);

    /*
     * ユーザからの入力を検証する。
     * 不正な指定が有った場合、エラーメッセージを出力して終了する。
     */
    if (adr1_type == NtmStrTypeInvalid) {
        /*
         * 認識できない型を発見したということは、不正なアドレス指定。
         */
        ntmuart_write(NTMCMD_COMMON_TEXT_INVADR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVADR));
        return 1;
    }
    if (adr2_type == NtmStrTypeInvalid) {
        /*
         * 認識できない型を発見したということは、不正なアドレス指定。
         */
        ntmuart_write(NTMCMD_COMMON_TEXT_INVADR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVADR));
        return 1;
    }
    if (len_type == NtmStrTypeInvalid) {
        /*
         * 認識できない型を発見したということは、不正な長さ指定。
         */
        ntmuart_write(NTMCMD_COMMON_TEXT_INVLEN, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVLEN));
        return 1;
    }

    /*
     * アクセスモードに応じて処理を実行する。
     */
    switch (am) {
        case NtmAccessModeByte:
            {
                uint8_t dat1, dat2;
                for (i = 0; i < len / 1; i++) {
                    /*
                     * 指定アドレスのデータを読み込む。
                     */
                    ntmio_read_byte(adr1 + (i * 1), &dat1);

                    /*
                     * 指定アドレスのデータを読み込む。
                     */
                    ntmio_read_byte(adr2 + (i * 1), &dat2);

                    /*
                     * ベリファイを実行する。
                     */
                    if (dat1 != dat2) {
                        /*
                         * ヘッダを表示する。
                         */
                        if (cnt == 0) {
                            ntmuart_write(DISPTEXT_DIFF_HEADER, ntmlibc_strlen(DISPTEXT_DIFF_HEADER));
                        }
                        /*
                         * アドレスを表示する。
                         */
                        ntmstr_number_to_string(adr1 + (i * 1), &string[0], 8, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write("\t", 1);
                        ntmstr_number_to_string(adr2 + (i * 1), &string[0], 8, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write("\t", 1);
                        /*
                         * データを表示する。
                         */
                        ntmstr_number_to_string(dat1, &string[0], 2, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write("\t", 1);
                        ntmstr_number_to_string(dat2, &string[0], 2, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));

                        cnt++;
                    }
                }
            }
            break;
        case NtmAccessModeWord:
            {
                uint16_t dat1, dat2;
                for (i = 0; i < len / 2; i++) {
                    /*
                     * 指定アドレスのデータを読み込む。
                     */
                    ntmio_read_word(adr1 + (i * 2), &dat1);

                    /*
                     * 指定アドレスのデータを読み込む。
                     */
                    ntmio_read_word(adr2 + (i * 2), &dat2);

                    /*
                     * ベリファイを実行する。
                     */
                    if (dat1 != dat2) {
                        /*
                         * ヘッダを表示する。
                         */
                        if (cnt == 0) {
                            ntmuart_write(DISPTEXT_DIFF_HEADER, ntmlibc_strlen(DISPTEXT_DIFF_HEADER));
                        }
                        /*
                         * アドレスを表示する。
                         */
                        ntmstr_number_to_string(adr1 + (i * 2), &string[0], 8, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write("\t", 1);
                        ntmstr_number_to_string(adr2 + (i * 2), &string[0], 8, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write("\t", 1);
                        /*
                         * データを表示する。
                         */
                        ntmstr_number_to_string(dat1, &string[0], 4, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write("\t", 1);
                        ntmstr_number_to_string(dat2, &string[0], 4, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));

                        cnt++;
                    }
                }
            }
            break;
        case NtmAccessModeLong:
            {
                uint32_t dat1, dat2;
                for (i = 0; i < len / 4; i++) {
                    /*
                     * 指定アドレスのデータを読み込む。
                     */
                    ntmio_read_long(adr1 + (i * 4), &dat1);

                    /*
                     * 指定アドレスのデータを読み込む。
                     */
                    ntmio_read_long(adr2 + (i * 4), &dat2);

                    /*
                     * ベリファイを実行する。
                     */
                    if (dat1 != dat2) {
                        /*
                         * ヘッダを表示する。
                         */
                        if (cnt == 0) {
                            ntmuart_write(DISPTEXT_DIFF_HEADER, ntmlibc_strlen(DISPTEXT_DIFF_HEADER));
                        }
                        /*
                         * アドレスを表示する。
                         */
                        ntmstr_number_to_string(adr1 + (i * 4), &string[0], 8, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write("\t", 1);
                        ntmstr_number_to_string(adr2 + (i * 4), &string[0], 8, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write("\t", 1);
                        /*
                         * データを表示する。
                         */
                        ntmstr_number_to_string(dat1, &string[0], 8, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write("\t", 1);
                        ntmstr_number_to_string(dat2, &string[0], 8, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));

                        cnt++;
                    }
                }
            }
            break;
        default:
            ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
            break;
    }
    return 1;
}

#endif

