/**
 * @file ntmconf.h
 * @author Shinichiro Nakamura
 * @brief NTM設定の定義。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 * Inspired by M, Murakami
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#ifndef NTMCONF_H
#define NTMCONF_H

#include "ntmstr.h"

/**
 * @brief シリアル・ポートのデフォルト・ボーレート。
 */
#define NTMCONF_DEFAULT_BAUDRATE    (57600)

/**
 * @brief 基数指定がない入力に対するデフォルトの解釈。
 * @details
 * 入力を簡便にするために基数指定を省略する事もできるようにしてある。
 * 基数指定がない場合の解釈はユーザによって要求が異なるだろう。
 * そこで、NTMではntmstr.hで示されたNtmStrTypeの値から指定することが
 * できるようにした。値はNtmStrTypeBin、NtmStrTypeOct、NtmStrTypeDec、
 * NtmStrTypeHexから指定する事ができる。
 * NtmStrTypeInvalidを指定すれば、基数指定のない値をエラーとする。
 */
#define NTMCONF_DEFAULT_STRTYPE     NtmStrTypeHex

/**
 * @brief 16進数の表現のデフォルト・スタイル。
 * @details 1で大文字、0で小文字表現となる。それ以外の値は動作未定義である。
 */
#define NTMCONF_DEFAULT_HEXSTYLE    (1)

/**
 * @brief NT-Shellライブラリを使用するかどうかの選択。
 * @details
 * NTMでは、ntshellモジュールの他に、ntoptやntlibcなどを用いている。
 * それらに依存させるかどうかをここで決定する事ができる。
 */
#define NTMCONF_USE_NTSHELL         (0)

/**
 * @brief READコマンド活性化スイッチ。
 * @details 使用する場合には1、使用しない場合には0を指定する。
 */
#define NTMCONF_ENABLE_READ         (1)

/**
 * @brief WRITEコマンド活性化スイッチ。
 * @details 使用する場合には1、使用しない場合には0を指定する。
 */
#define NTMCONF_ENABLE_WRITE        (1)

/**
 * @brief DUMPコマンド活性化スイッチ。
 * @details 使用する場合には1、使用しない場合には0を指定する。
 */
#define NTMCONF_ENABLE_DUMP         (1)

/**
 * @brief FILLコマンド活性化スイッチ。
 * @details 使用する場合には1、使用しない場合には0を指定する。
 */
#define NTMCONF_ENABLE_FILL         (1)

/**
 * @brief COPYコマンド活性化スイッチ。
 * @details 使用する場合には1、使用しない場合には0を指定する。
 */
#define NTMCONF_ENABLE_COPY         (1)

/**
 * @brief VERIFYコマンド活性化スイッチ。
 * @details 使用する場合には1、使用しない場合には0を指定する。
 */
#define NTMCONF_ENABLE_VERIFY       (1)

/**
 * @brief SCROLLコマンド活性化スイッチ。
 * @details 使用する場合には1、使用しない場合には0を指定する。
 */
#define NTMCONF_ENABLE_SCROLL       (0)

/**
 * @brief SCROLLコマンドはNT-Shellのvtsendとvtrecvを必要とする。
 * @details
 * SCROLLコマンドを使用する設定にも関わらず、
 * NT-Shellを使用しないと宣言している場合はエラーとする。
 */
#if ((NTMCONF_ENABLE_SCROLL == 1) && (NTMCONF_USE_NTSHELL == 0))
#   error "Illegal configuration found. NTMCONF_ENABLE_SCROLL needs NTMCONF_USE_NTSHELL."
#endif

/**
 * @brief パース可能な最大文字列長。
 * @details このパラメータは、NT-Shellを使用しない時に参照される。
 */
#define NTMCONF_EDITOR_MAXLEN   (32)

#endif

