/**
 * @file ntmshell.c
 * @author Shinichiro Nakamura
 * @brief NTMシェルシステムの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 * Inspired by M, Murakami
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmshell.h"
#include "ntmlibc.h"

#define NTMSHELL_MAJOR      0
#define NTMSHELL_MINOR      0
#define NTMSHELL_RELEASE    1

#if (NTMCONF_USE_NTSHELL == 1)

/*
 * NT-Shellの実装を使用する。
 * このファイルでは実装を一切提供しない。
 */

#else

/*
 * 独自の実装を使用する。
 * このファイルで実装を提供する。
 */

void ntmshell_version(int *major, int *minor, int *release)
{
    *major = NTMSHELL_MAJOR;
    *minor = NTMSHELL_MINOR;
    *release = NTMSHELL_RELEASE;
}

void ntmshell_execute(
        ntmshell_t *p,
        int (*func_read)(char *buf, int cnt),
        int (*func_write)(const char *buf, int cnt),
        int (*func_callback)(const char *text, void *extobj),
        void *func_callback_extobj)
{
    int i;

    /*
     * ここでの入力系の実装は、
     * NT-Shellの代替となるものであるが、
     * 非常に簡易的なものである。
     * NT-Shellが解決しているような
     * 種々の問題に対しての回答を示していない。
     * 期待通りに動作する入力系が欲しい場合、
     * NT-Shellを使うことを検討して欲しい。
     */

    for (i = 0; i < sizeof(p->buf_curr); i++) {
        p->buf_curr[i] = 0;
    }
    for (i = 0; i < sizeof(p->buf_prev); i++) {
        p->buf_prev[i] = 0;
    }
    p->len = 0;

    func_write(">", 1);
    while (1) {
        char c;
        func_read(&c, 1);
        switch (c) {
            case 0x08:
                /*
                 * BS
                 */
                func_write(&c, 1);
                func_write(" ", 1);
                func_write(&c, 1);
                if (0 < p->len) {
                    p->buf_curr[p->len--] = '\0';
                }
                break;
            case 0x09:
                /*
                 * TAB
                 */
                for (i = 0; i < p->len; i++) {
                    func_write("\x08 \x08", 3);
                }
                ntmlibc_strcpy(p->buf_curr, p->buf_prev);
                p->len = ntmlibc_strlen(p->buf_curr);
                func_write(p->buf_curr, p->len);
                break;
            case 0x0D:
                /*
                 * ENTER
                 */
                func_write("\r\n", 2);
                p->buf_curr[p->len++] = '\0';
                func_callback(p->buf_curr, func_callback_extobj);
                ntmlibc_strcpy(p->buf_prev, p->buf_curr);
                p->len = 0;
                func_write(">", 1);
                break;
        }
        if ((0x20 <= c) && (c <= 0x7E)) {
            if (p->len < (sizeof(p->buf_curr) - 1)) {
                func_write(&c, 1);
                p->buf_curr[p->len++] = c;
            } else {
                func_write("\r\n", 2);
                p->buf_curr[p->len++] = '\0';
                func_callback(p->buf_curr, func_callback_extobj);
                ntmlibc_strcpy(p->buf_prev, p->buf_curr);
                p->len = 0;
                func_write(">", 1);
            }
        }
    }
}

#endif

