/**
 * @file ntmstr.c
 * @author Shinichiro Nakamura
 * @brief NTMが使用する文字列ユティリティの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 * Inspired by M, Murakami
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmstr.h"
#include "ntmconf.h"
#include "ntmlibc.h"

#define TYPECHAR_BIN    'b'
#define TYPECHAR_OCT    'o'
#define TYPECHAR_DEC    'd'
#define TYPECHAR_HEX    'h'

#define BINCHAR_TO_NUM(C)  ((C) - '0')
#define OCTCHAR_TO_NUM(C)  ((C) - '0')
#define DECCHAR_TO_NUM(C)  ((C) - '0')
#define HEXCHAR_TO_NUM(C)  ((('0' <= (C)) && ((C) <= '9')) ? ((C) - '0') : ((('A' <= (C)) && ((C) <= 'F')) ? (10 + ((C) - 'A')) : (10 + ((C) - 'a'))))

/**
 * @brief 値を2進表現文字列に変換する。
 * @details
 * - ここで言う「2進表現文字列」とは、以下のようなものを指す。
 *   - b11010011
 *   - 11010011b
 * 要するに、文字列が0-1で構成され、かつ先頭か後尾にbが付くものである。
 *
 * @param number 値。
 * @param string 2進表現文字列。
 * @param length 文字列幅。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
static int number_to_binstr(
        const uint32_t number,
        char *string,
        const uint32_t length)
{
    const uint8_t base = 2;
    uint32_t work = number;
    uint32_t bnum = 1;
    uint32_t i;

    for (i = 0; i < length - 1; i++) {
        bnum = bnum * base;
    }

    for (i = 0; i < length; i++) {
        uint32_t r = work / bnum;
        string[i] = "01"[r % 2];
        work = work - (r * bnum);
        bnum = bnum / base;
    }
    string[length] = '\0';

    return 0;
}

/**
 * @brief 値を8進表現文字列に変換する。
 * @details
 * - ここで言う「8進表現文字列」とは、以下のようなものを指す。
 *   - o01234567
 *   - 01234567o
 * 要するに、文字列が0-7で構成され、かつ先頭か後尾にoが付くものである。
 *
 * @param number 値。
 * @param string 8進表現文字列。
 * @param length 文字列幅。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
static int number_to_octstr(
        const uint32_t number,
        char *string,
        const uint32_t length)
{
    const uint8_t base = 8;
    uint32_t work = number;
    uint32_t bnum = 1;
    uint32_t i;

    for (i = 0; i < length - 1; i++) {
        bnum = bnum * base;
    }

    for (i = 0; i < length; i++) {
        uint32_t r = work / bnum;
        string[i] = "01234567"[r % 8];
        work = work - (r * bnum);
        bnum = bnum / base;
    }
    string[length] = '\0';

    return 0;
}

/**
 * @brief 値を10進表現文字列に変換する。
 * @details
 * - ここで言う「10進表現文字列」とは、以下のようなものを指す。
 *   - d12345678
 *   - 12345678d
 * 要するに、文字列が0-9で構成され、かつ先頭か後尾にdが付くものである。
 *
 * @param number 値。
 * @param string 10進表現文字列。
 * @param length 文字列幅。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
static int number_to_decstr(
        const uint32_t number,
        char *string,
        const uint32_t length)
{
    const uint8_t base = 10;
    uint32_t work = number;
    uint32_t bnum = 1;
    uint32_t i;

    for (i = 0; i < length - 1; i++) {
        bnum = bnum * base;
    }

    for (i = 0; i < length; i++) {
        uint32_t r = work / bnum;
        string[i] = "0123456789"[r % 10];
        work = work - (r * bnum);
        bnum = bnum / base;
    }
    string[length] = '\0';

    return 0;
}

/**
 * @brief 値を16進表現文字列に変換する。
 * @details
 * - ここで言う「16進表現文字列」とは、以下のようなものを指す。
 *   - h1234abcd
 *   - 1234abcdh
 * 要するに、文字列が0-9,a-fで構成され、かつ先頭か後尾にhが付くものである。
 *
 * @param number 値。
 * @param string 16進表現文字列。
 * @param length 文字列幅。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
static int number_to_hexstr(
        const uint32_t number,
        char *string,
        const uint32_t length)
{
    const uint8_t base = 16;
    uint32_t work = number;
    uint32_t bnum = 1;
    uint32_t i;

    for (i = 0; i < length - 1; i++) {
        bnum = bnum * base;
    }

    for (i = 0; i < length; i++) {
        uint32_t r = work / bnum;
#if (NTMCONF_DEFAULT_HEXSTYLE)
        string[i] = "0123456789ABCDEF"[r % 16];
#else
        string[i] = "0123456789abcdef"[r % 16];
#endif
        work = work - (r * bnum);
        bnum = bnum / base;
    }
    string[length] = '\0';

    return 0;
}

/**
 * @brief 与えられた数値を与えられた条件に従って文字列に変換する。
 * @details
 * 大文字と小文字の選択が可能である。
 * 出力桁数の選択が可能である。
 * 基数の選択が可能である。
 *
 * @param number 値。
 * @param string 文字列。
 * @param length 文字列の整形サイズ。
 * @param type 基数指定。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmstr_number_to_string(
        const uint32_t number,
        char *string,
        const uint32_t length,
        const NtmStrType type)
{
    uint32_t i;
    switch (type) {
        case NtmStrTypeBin:
            return number_to_binstr(number, string, length);
        case NtmStrTypeOct:
            return number_to_octstr(number, string, length);
        case NtmStrTypeDec:
            return number_to_decstr(number, string, length);
        case NtmStrTypeHex:
            return number_to_hexstr(number, string, length);
        case NtmStrTypeInvalid:
            break;
        default:
            break;
    }
    for (i = 0; i < length; i++) {
        string[i] = '0';
    }
    string[length] = '\0';
    return -1;
}

/**
 * @brief 2進表現文字列を値に変換する。
 * @details
 * - ここで言う「2進表現文字列」とは、以下のようなものを指す。
 *   - b01001100
 *   - 01001100b
 * 要するに、文字列が0と1で構成され、かつ先頭か後尾にbが付くものである。
 *
 * @param string 2進表現文字列。
 * @param number 値。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
static int binstr_to_number(const char *string, uint32_t *number)
{
    const uint8_t base = 2;
    const uint32_t len = ntmlibc_strlen(string);
    uint32_t work = 0;
    uint32_t bnum = 1;
    uint32_t i;

    /*
     * 基数に従って文字列の後尾から値に変換する。
     */
    char *p = (char *)&string[len - 1];
    for (i = 0; i < len; i++, p--) {
        char c = *p;
        switch (c) {
            case '0':
            case '1':
                work += bnum * BINCHAR_TO_NUM(c);
                bnum = bnum * base;
                break;
            case TYPECHAR_BIN:
                if ((i != 0) && (i != (len - 1))) {
                    *number = 0;
                    return -1;
                }
                break;
            default:
                *number = 0;
                return -1;
        }
    }
    *number = work;
    return 0;
}

/**
 * @brief 8進表現文字列を値に変換する。
 * @details
 * - ここで言う「8進表現文字列」とは、以下のようなものを指す。
 *   - o01234567
 *   - 01234567o
 * 要するに、文字列が0から7で構成され、かつ先頭か後尾にoが付くものである。
 *
 * @param string 8進表現文字列。
 * @param number 値。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
static int octstr_to_number(const char *string, uint32_t *number)
{
    const uint8_t base = 8;
    const uint32_t len = ntmlibc_strlen(string);
    uint32_t work = 0;
    uint32_t bnum = 1;
    uint32_t i;

    /*
     * 基数に従って文字列の後尾から値に変換する。
     */
    char *p = (char *)&string[len - 1];
    for (i = 0; i < len; i++, p--) {
        char c = *p;
        switch (c) {
            case '0':
            case '1':
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
                work += bnum * OCTCHAR_TO_NUM(c);
                bnum = bnum * base;
                break;
            case TYPECHAR_OCT:
                if ((i != 0) && (i != (len - 1))) {
                    *number = 0;
                    return -1;
                }
                break;
            default:
                *number = 0;
                return -1;
        }
    }
    *number = work;
    return 0;
}

/**
 * @brief 10進表現文字列を値に変換する。
 * @details
 * - ここで言う「10進表現文字列」とは、以下のようなものを指す。
 *   - d12345678
 *   - 12345678d
 * 要するに、文字列が0-9で構成され、かつ先頭か後尾にdが付くものである。
 *
 * @param string 10進表現文字列。
 * @param number 値。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
static int decstr_to_number(const char *string, uint32_t *number)
{
    const uint8_t base = 10;
    const uint32_t len = ntmlibc_strlen(string);
    uint32_t work = 0;
    uint32_t bnum = 1;
    uint32_t i;

    /*
     * 基数に従って文字列の後尾から値に変換する。
     */
    char *p = (char *)&string[len - 1];
    for (i = 0; i < len; i++, p--) {
        char c = *p;
        switch (c) {
            case '0':
            case '1':
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
            case '8':
            case '9':
                work += bnum * DECCHAR_TO_NUM(c);
                bnum = bnum * base;
                break;
            case TYPECHAR_DEC:
                if ((i != 0) && (i != (len - 1))) {
                    *number = 0;
                    return -1;
                }
                break;
            default:
                *number = 0;
                return -1;
        }
    }
    *number = work;
    return 0;
}

/**
 * @brief 16進表現文字列を値に変換する。
 * @details
 * - ここで言う「16進表現文字列」とは、以下のようなものを指す。
 *   - h1234abcd
 *   - 1234abcdh
 * 要するに、文字列が0-9,a-fで構成され、かつ先頭か後尾にhが付くものである。
 *
 * @param string 16進表現文字列。
 * @param number 値。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
static int hexstr_to_number(const char *string, uint32_t *number)
{
    const uint8_t base = 16;
    const uint32_t len = ntmlibc_strlen(string);
    uint32_t work = 0;
    uint32_t bnum = 1;
    uint32_t i;

    /*
     * 基数に従って文字列の後尾から値に変換する。
     */
    char *p = (char *)&string[len - 1];
    for (i = 0; i < len; i++, p--) {
        char c = *p;
        switch (c) {
            case '0':
            case '1':
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
            case '8':
            case '9':
            case 'A':
            case 'B':
            case 'C':
            case 'D':
            case 'E':
            case 'F':
            case 'a':
            case 'b':
            case 'c':
            case 'd':
            case 'e':
            case 'f':
                work += bnum * HEXCHAR_TO_NUM(c);
                bnum = bnum * base;
                break;
            case TYPECHAR_HEX:
                if ((i != 0) && (i != (len - 1))) {
                    *number = 0;
                    return -1;
                }
                break;
            default:
                *number = 0;
                return -1;
        }
    }
    *number = work;
    return 0;
}

/**
 * @brief 与えられた文字列を数値として解釈し変換する。
 * @details
 * この関数は大文字と小文字を区別しない。
 * 書式に従うことで16進、10進、2進と自動判別する。
 * 与えられた文字列に基数指定がない場合の動作は選択可能である。
 *
 * @param string 文字列。
 * @param number 値。
 * @param type 判別した基数。
 * @param defaultType 自動判別できなかった時に採用する値。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmstr_string_to_number(
        const char *string,
        uint32_t *number,
        NtmStrType *type,
        const NtmStrType defaultType)
{
    /*
     * 基数の判定を実行する。
     * 基数の判定は、文字列の先頭か後尾に基数指定があるかどうかで判定する。
     */
    const uint32_t len = ntmlibc_strlen(string);
    char head, tail;
    if (len == 0) {
        *number = 0;
        *type = NtmStrTypeInvalid;
        return -1;
    }
    head = string[0];
    tail = string[len - 1];
    if ((head == TYPECHAR_BIN) || (tail == TYPECHAR_BIN)) {
        /*
         * 与えられた文字列は2進表現である。
         */
        if (binstr_to_number(string, number) == 0) {
            *type = NtmStrTypeBin;
            return 0;
        } else {
            *type = NtmStrTypeInvalid;
            return -1;
        }
    }
    if ((head == TYPECHAR_OCT) || (tail == TYPECHAR_OCT)) {
        /*
         * 与えられた文字列は8進表現である。
         */
        if (octstr_to_number(string, number) == 0) {
            *type = NtmStrTypeOct;
            return 0;
        } else {
            *type = NtmStrTypeInvalid;
            return -1;
        }
    }
    if ((head == TYPECHAR_DEC) || (tail == TYPECHAR_DEC)) {
        /*
         * 与えられた文字列は10進表現である。
         */
        if (decstr_to_number(string, number) == 0) {
            *type = NtmStrTypeDec;
            return 0;
        } else {
            *type = NtmStrTypeInvalid;
            return -1;
        }
    }
    if ((head == TYPECHAR_HEX) || (tail == TYPECHAR_HEX)) {
        /*
         * 与えられた文字列は16進表現である。
         */
        if (hexstr_to_number(string, number) == 0) {
            *type = NtmStrTypeHex;
            return 0;
        } else {
            *type = NtmStrTypeInvalid;
            return -1;
        }
    }
    /*
     * 与えられた文字列をデフォルトの基数表現であると見做して処理する。
     */
    switch (defaultType) {
        case NtmStrTypeBin:
            if (binstr_to_number(string, number) == 0) {
                *type = NtmStrTypeBin;
                return 0;
            } else {
                *type = NtmStrTypeInvalid;
                return -1;
            }
        case NtmStrTypeOct:
            if (octstr_to_number(string, number) == 0) {
                *type = NtmStrTypeOct;
                return 0;
            } else {
                *type = NtmStrTypeInvalid;
                return -1;
            }
        case NtmStrTypeDec:
            if (decstr_to_number(string, number) == 0) {
                *type = NtmStrTypeDec;
                return 0;
            } else {
                *type = NtmStrTypeInvalid;
                return -1;
            }
        case NtmStrTypeHex:
            if (hexstr_to_number(string, number) == 0) {
                *type = NtmStrTypeHex;
                return 0;
            } else {
                *type = NtmStrTypeInvalid;
                return -1;
            }
        case NtmStrTypeInvalid:
            break;
    }
    *type = NtmStrTypeInvalid;
    *number = 0;
    return -1;
}

/**
 * @brief 与えられた文字が表示可能かどうかを返す。
 *
 * @param c 文字。
 *
 * @retval !0 表示可能である。
 * @retval 0 表示可能でない。
 */
int ntmstr_isprint(int c)
{
    if ((0x20 <= c) && (c <= 0x7E)) {
        return 1;
    }
    return 0;
}

