/*
 *  UART Driver
 */

#include "mcu.h"
#include "Uart.h"
#include "Uart_FM3.h"
#include <stdlib.h>
#include "clock_def.h"
#include "core_cm3.h"



/*
 * UART Device Configuration Structure
 */
typedef struct {
    int32_t BaudRate;     /* Baudrate    */
    int32_t DataBits:5;   /* Data bits   */
    int32_t StopBits:3;   /* Stop bits   */
    int32_t Parity:3;     /* Parity      */
    int32_t BitOrder:2;   /* BitOrder    */
    int32_t Reserved:19;
} UartDev_CFG;

/*
 *  Driver Information Control Struct
 */
typedef struct {
    int32_t      State;     /* State            */
#ifdef USE_CH03
    FM3_MFS03_UART_TypeDef *pReg;   /* Register Struct  */
#else
    FM3_MFS47_UART_TypeDef *pReg;   /* Register Struct  */
#endif
    UartDev_CFG   Cfg;    /* UART Device Configuration Structure */
} Uart_INFO ;

static Uart_INFO UartInfo = {0};


/*
 *  Internal Function
 */
static int32_t  uart_Init_Internal(void);
static void  uart_UnInit_Internal(void);


/*
 *  Uart_Init
 */
int32_t  Uart_Init(void)
{
    int32_t ret;
    
    UartInfo.State = UART_UNINITIALIZED;
    UartInfo.pReg = UART_REG_ADR;
    UartInfo.Cfg.BaudRate = UART_BAUDRATE;
    UartInfo.Cfg.DataBits = UART_DATABITS;
    UartInfo.Cfg.StopBits = UART_STOPBITS;
    UartInfo.Cfg.Parity = UART_PARITY;
    UartInfo.Cfg.BitOrder = UART_BITODER;
    UartInfo.Cfg.Reserved = 0;
    
        
    /* initialize hardware */
    ret = uart_Init_Internal();
    
    if (ret == SUCCESS) {
        /* change state */
        UartInfo.State = UART_INITIALIZED;
        
        ret = SUCCESS;
    } else {
        ret = ERROR; /* uart_Init error */
    }
    
    return ret;
}


/*
 *  Uart_UnInit
 */
int32_t  Uart_UnInit(void)
{
    int32_t ret;
    
    
    if (UartInfo.State != UART_INITIALIZED) {
        ret = ERROR; /* not initialized */
        
    } else {
        /* change state */
        UartInfo.State = UART_UNINITIALIZED;
        
        /* uninitialize */
        uart_UnInit_Internal();
        
        ret = SUCCESS;
    }
        
    
    return ret;
}


/*
 *  Uart_BufTx
 */
int32_t  Uart_BufTx(void *pData, int32_t *pSize)
{
    int32_t ret;
    int32_t count;
    int32_t retry_count;
    uint8_t *p_char;
    uint8_t  reg_ssr;
    
    /* check NULL pointer */
    if ((pData == NULL) || (pSize == NULL)) {
        return ERROR;
    }
    
    /* check parameter */
    if (*pSize <= 0) {
        return ERROR;
    }
    
    
    if (UartInfo.State != UART_INITIALIZED) {
        ret = ERROR; /* not initialized */
        
    } else {
        p_char = (uint8_t *)pData;

        for (count=0;count<*pSize;count++) {

            retry_count = 100000;
            do {
                reg_ssr  = UartInfo.pReg->SSR;
                retry_count--;
            } while ((retry_count != 0) && ((reg_ssr & (SSR_TDRE)) == 0));

            if (retry_count != 0) {
                UartInfo.pReg->TDR = (uint16_t)*p_char;
                p_char++;
            } else {
                /* break if timeout */
                break;
            }
        }
        
        /* return size */
        *pSize = count;
        
        ret = SUCCESS;
    }
    
    return ret;
}


/*
 *  Uart_BufRx
 */
int32_t  Uart_BufRx(void *pData, int32_t *pSize)
{
    int32_t ret;
    int32_t count;
    uint8_t *p_char;
    uint8_t  reg_ssr;
    
    /* check NULL pointer */
    if ((pData == NULL) || (pSize == NULL)) {
        return ERROR;
    }
    
    /* check parameter */
    if (*pSize <= 0) {
        return ERROR;
    }
    
    
    if (UartInfo.State != UART_INITIALIZED) {
        ret = ERROR; /* not initialized */
    } else {
        count  = 0;
        p_char = (uint8_t *)pData;
        
        reg_ssr  = UartInfo.pReg->SSR;
        if ((reg_ssr & (SSR_RDRF)) != 0) {
            *p_char = (uint8_t)UartInfo.pReg->RDR;
            count = 1;
        }
        
        /* return size */
        *pSize = count;

        ret = SUCCESS;
    }
    
    return ret;
}



/*
 *  uart_Init (Internal Function)
 */
static int32_t  uart_Init_Internal(void)
{
    int32_t ret;
    uint16_t reg_bgr;
    uint8_t reg_smr    = 0;
    uint8_t reg_escr   = 0;
    uint8_t reg_scr    = 0;
    
    /*
     *  Set value of SMR register
     */
    /* Mode = UART,  serial output enable */
    reg_smr |= (SMR_MD_UART | SMR_SOE);
    
    /* BitOrder */
    if (UartInfo.Cfg.BitOrder != UartDev_BITORDER_LSB) {
        /* MSB first */
        reg_smr |= (SMR_BDS);
        
    } /* Do nothing if LSB first */
    
    /* StopBits */
    if (UartInfo.Cfg.StopBits == UartDev_STOPBITS_2) {
        /* StopBits 2 */
        reg_smr |= (SMR_SBL);
    } /* Do nothing if StopBits 1 */
    
    
    /*
     *  Set value of BGR register
     */
    /* BaudRate : round off reg_bgr to the whole number */
    reg_bgr = (uint16_t)(((SysFrePCLK2 + ((uint32_t)UartInfo.Cfg.BaudRate / 2)) / (uint32_t)UartInfo.Cfg.BaudRate) - 1);
    
    /*
     *  Set value of ESCR register
     */
    /* DataBits */
    switch (UartInfo.Cfg.DataBits) {
    case UartDev_DATABITS_6:
        reg_escr |= (ESCR_DATABITS_6);
        break;
    case UartDev_DATABITS_7:
        reg_escr |= (ESCR_DATABITS_7);
        break;
    default: /* UartDev_DATABITS_8 */
        reg_escr |= (ESCR_DATABITS_8);
    }
    
    /* Parity */
    switch (UartInfo.Cfg.Parity) {
    case UartDev_PARITY_ODD:
        reg_escr |= (ESCR_PEN);
        break;
    case UartDev_PARITY_EVEN:
        reg_escr |= (ESCR_PEN | ESCR_P);
        break;
    default: /* UartDev_PARITY_NONE */
        /* Do nothing if Parity none */
        reg_escr |= 0; //DMY
    }
    
    
    /*
     *  Set value of SCR register
     */
    /* rx enable, tx enable */
    reg_scr |= (SCR_RXE | SCR_TXE);
    
    
    /*
     *  Register writing
     */
    /* set mode */
    UartInfo.pReg->SMR    = reg_smr;
    
    /* All disable before other registers are set. */
    UartInfo.pReg->SCR    = 0;
    
    UartInfo.pReg->BGR    = reg_bgr;
    UartInfo.pReg->ESCR   = reg_escr;
    
    UartInfo.pReg->SCR = reg_scr;
    
    ret = SUCCESS;
    
    
    return ret;
}

/*
 *  uart_UnInit (Internal Function)
 */
static void  uart_UnInit_Internal(void)
{
    uint8_t reg_scr;

    /* rx disable, tx disable */
    reg_scr = UartInfo.pReg->SCR & ~(SCR_RXE | SCR_TXE);
    UartInfo.pReg->SCR = reg_scr;
    
    /* software rest */
    reg_scr = UartInfo.pReg->SCR | (SCR_UPGL);
    UartInfo.pReg->SCR = reg_scr;
    
    return;
}


