/**
 * @file ntmattr.c
 * @author Shinichiro Nakamura
 * @brief メモリ情報の実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmattr.h"
#include "ntmlibc.h"

static const ntmattr_info_t infolist[] = {
    /* 12345678901234567890123456789012 */
    { "Onchip flash ROM",               0x00000000, 0x000fffff, NTMATTR_ATTR_RO,    },
    { "Security/CR Trim",               0x00100000, 0x00101fff, NTMATTR_ATTR_RO,    },
    { "Reserved",                       0x00102000, 0x1ffeffff, NTMATTR_ATTR_RSV,   },
    { "SRAM0",                          0x1fff0000, 0x1fffffff, NTMATTR_ATTR_RW,    },
    { "SRAM1",                          0x20000000, 0x2007ffff, NTMATTR_ATTR_RW,    },
    { "Reserved",                       0x20080000, 0x21ffffff, NTMATTR_ATTR_RSV,   },
    { "32MB Bit Band Alias",            0x22000000, 0x23ffffff, NTMATTR_ATTR_RO,    },
    { "Resreved",                       0x24000000, 0x3fffffff, NTMATTR_ATTR_RSV,   },
    { "Peripheral",                     0x40000000, 0x41ffffff, NTMATTR_ATTR_RW,    },
    { "32MB Bit Band Alias",            0x42000000, 0x43ffffff, NTMATTR_ATTR_RO,    },
    { "Reserved",                       0x44000000, 0x5fffffff, NTMATTR_ATTR_RSV,   },
    { "External Device Area",           0x60000000, 0xefffffff, NTMATTR_ATTR_RW,    },
    { "Cortex-M3 Private Peripheral",   0xf0000000, 0xf00fffff, NTMATTR_ATTR_RW,    },
    { "Reserved",                       0xf0100000, 0xffffffff, NTMATTR_ATTR_RSV,   },
};
static const int LISTSIZE = sizeof(infolist) / sizeof(infolist[0]);

/**
 * @brief メモリ情報を取得する。
 *
 * @param addr アドレス。
 * @param meminfo メモリ情報構造体。
 *
 * @retval 0 指定アドレスのメモリ情報が存在し取得した。
 * @retval !0 指定アドレスのメモリ情報が存在しなかった。
 */
int ntmattr_get_info(uint32_t addr, ntmattr_info_t *meminfo)
{
    int i;
    ntmattr_info_t *src = (ntmattr_info_t *)&infolist[0];
    for (i = 0; i < LISTSIZE; i++, src++) {
        if ((src->addr_start <= addr) && (addr <= src->addr_end)) {
            ntmlibc_strcpy(meminfo->name, src->name);
            meminfo->addr_start = src->addr_start;
            meminfo->addr_end = src->addr_end;
            meminfo->attr = src->attr;
            return 0;
        }
    }
    ntmlibc_strcpy(meminfo->name, "");
    meminfo->addr_start = 0;
    meminfo->addr_end = 0;
    meminfo->attr = NTMATTR_ATTR_UNKNOWN;
    return -1;
}

