/**
 * @file ntmuart.c
 * @author Shinichiro Nakamura
 * @brief UARTの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmuart.h"
#include "mcu.h"
#include "Uart_FM3.h"

static void InitPort(void)
{
    /*
     * Set I/O Port
     */

    /* ADC */
    FM3_GPIO->ADE_f.AN31 = 0;  // disable AN31(P22/SOT0_0)

    /* Ch0 */
    FM3_GPIO->PFR2_f.P1 = 1;   // P21/SIN0_0
    FM3_GPIO->EPFR07_f.SIN0S1 = 0;
    FM3_GPIO->EPFR07_f.SIN0S0 = 0;
    FM3_GPIO->PFR2_f.P2 = 1;   // P22/SOT0_0
    FM3_GPIO->EPFR07_f.SOT0B1 = 0;
    FM3_GPIO->EPFR07_f.SOT0B0 = 1;

    /* Ch3 */
    FM3_GPIO->PFR4_f.P8 = 1;   // P48/SIN3_2
    FM3_GPIO->EPFR07_f.SIN3S1 = 1;
    FM3_GPIO->EPFR07_f.SIN3S0 = 1;
    FM3_GPIO->PFR4_f.P9 = 1;   // P49/SOT3_2
    FM3_GPIO->EPFR07_f.SOT3B1 = 1;
    FM3_GPIO->EPFR07_f.SOT3B0 = 1;

    /* Ch4 */
    FM3_GPIO->PFR0_f.P5 = 1;   // P05/SIN4_2
    FM3_GPIO->EPFR08_f.SIN4S1 = 1;
    FM3_GPIO->EPFR08_f.SIN4S0 = 1;
    FM3_GPIO->PFR0_f.P6 = 1;   // P06/SOT4_2
    FM3_GPIO->EPFR08_f.SOT4B1 = 1;
    FM3_GPIO->EPFR08_f.SOT4B0 = 1;
}

/**
 * @brief UARTを初期化する。
 *
 * @param baudrate ボーレート。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmuart_init(uint32_t baudrate)
{
    InitPort();
    Uart_Init();
    return 0;
}

/**
 * @brief UARTの読み込みを実行する。
 *
 * @param buf バッファ。
 * @param cnt 読み込みバイト数。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmuart_read(char *buf, const int cnt)
{
    int32_t n = 0;
    do {
        int32_t r = cnt;
        Uart_BufRx((void *)(buf + n), &r);
        n = n + r;
    } while (n < cnt);
    return (n == cnt) ? 0 : -1;
}

/**
 * @brief UARTの書き込みを実行する。
 *
 * @param buf バッファ。
 * @param cnt 書き込みバイト数。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmuart_write(const char *buf, const int cnt)
{
    int32_t n = 0;
    do {
        int32_t w = cnt;
        Uart_BufTx((void *)(buf + n), &w);
        n = n + w;
    } while (n < cnt);
    return (n == cnt) ? 0 : -1;
}

