/**
 * @file ntmuart.c
 * @author Shinichiro Nakamura
 * @brief UARTの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmuart.h"
#include "iodefine.h"

/**
 * @brief UARTを初期化する。
 *
 * @param baudrate ボーレート。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmuart_init(uint32_t baudrate)
{
    MSTP(SCI1) = 0;
    PORT3.DDR.BIT.B0 = 0;
    PORT3.ICR.BIT.B0 = 1;
    PORT2.DDR.BIT.B6 = 1;
    SCI1.SCR.BYTE = 0;
    SCI1.SMR.BYTE = 0x00;
    SCI1.SCR.BIT.RIE = 1;
    SCI1.SCR.BIT.TIE = 1;
    IR(SCI1, RXI1) = 0;
    IR(SCI1, TXI1) = 0;
    IEN(SCI1, RXI1) = 0;
    IEN(SCI1, TXI1) = 0;
    switch (baudrate) {
        case 9600:
            SCI1.BRR = 156;
            break;
        case 19200:
            SCI1.BRR = 78;
            break;
        case 38400:
            SCI1.BRR = 39;
            break;
        case 57600:
            SCI1.BRR = 26;
            break;
        case 115200:
            SCI1.BRR = 13;
            break;
    }
    SCI1.SCR.BYTE |= 0x30;
    return 0;
}

/**
 * @brief UARTの読み込みを実行する。
 *
 * @param buf バッファ。
 * @param cnt 読み込みバイト数。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmuart_read(char *buf, const int cnt)
{
    int i;
    for (i = 0; i < cnt; i++) {
        char c = SCI1.RDR;
        IR(SCI1, RXI1) = 0;
        while(IR(SCI1, RXI1) == 0) {
        }
        buf[i] = SCI1.RDR;
    }
    return 0;
}

/**
 * @brief UARTの書き込みを実行する。
 *
 * @param buf バッファ。
 * @param cnt 書き込みバイト数。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmuart_write(const char *buf, const int cnt)
{
    int i;
    for (i = 0; i < cnt; i++) {
        while(IR(SCI1, TXI1) == 0) {
        }
        IR(SCI1, TXI1) = 0;
        SCI1.TDR = buf[i];
    }
    return 0;
}

