/* ////////// LICENSE INFO ////////////////////

 * Copyright (C) 2013 by NYSOL CORPORATION
 *
 * Unless you have received this program directly from NYSOL pursuant
 * to the terms of a commercial license agreement with NYSOL, then
 * this program is licensed to you under the terms of the GNU Affero General
 * Public License (AGPL) as published by the Free Software Foundation,
 * either version 3 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY EXPRESS OR IMPLIED WARRANTY, INCLUDING THOSE OF 
 * NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Please refer to the AGPL (http://www.gnu.org/licenses/agpl-3.0.txt)
 * for more details.

 ////////// LICENSE INFO ////////////////////*/
// =============================================================================
// kgmvstats.cpp 移動窓の統計量の算出クラス
// =============================================================================
#include <cstdio>
#include <sstream>
#include <vector>
#include <cmath>
#include <cstdlib>
#include <algorithm>
#include <cfloat>
#include <kgmvstats.h>
#include <kgError.h>
#include <kgVal.h>
#include <kgMethod.h>
#include <kgConfig.h>

#include <boost/function.hpp>
#include <boost/lambda/bind.hpp>
#include <boost/lambda/construct.hpp>
#include <boost/bind.hpp>

using namespace std;
using namespace kglib;
using namespace kgmod;
using namespace boost;



// =============================================================================
// kgMvBase 
// =============================================================================
//--------------------------------------------------------------
// データの更新
// _valのデータを更新する_sposが最初に一周したとき_fullをtureにする
// -----------------------------------------------------------------------------
void kgmvstats_::kgMvBase::update(vector<double>& inp)
{
		for(size_t i=0; i<_fsize; i++){
			_val.at(i).at(_spos)=inp.at(i); 
		}
		pos_inc();
		if(_spos==0){ _filled = true; }
}
//--------------------------------------------------------------
// データの初期化(ポジション,fullフラグ、データの初期化)
//--------------------------------------------------------------
void kgmvstats_::kgMvBase::clear()
{
	for(vector<double>::size_type i=0;  i < _val.size(); i++){
		for(vector<double>::size_type j=0;  j < _val[i].size(); j++){
			_val.at(i).at(j)=0;
		}
	}
	_spos=0;
	_filled=false;
}

// =============================================================================
// kgMvSum　集計 
// =============================================================================
//--------------------------------------------------------------
// 計算
//--------------------------------------------------------------
void kgmvstats_::kgMvSum::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	for(vector<double>::size_type i = 0 ;i < _sumary.size() ;i++){
		_sumary.at(i) = _sumary.at(i) - kgMvBase::get(i) + inp.at(i); 
		rls.at(i).r(_sumary.at(i));
	}
	kgMvBase::update(inp);
}
//--------------------------------------------------------------
// データの初期化
//--------------------------------------------------------------
void kgmvstats_::kgMvSum::clear()
{
	for(vector<double>::size_type i = 0 ;i < _sumary.size() ;i++){
		_sumary.at(i) = 0; 
	}
	kgMvBase::clear();
}
// =============================================================================
// kgMvMean 算術平均
// =============================================================================
//--------------------------------------------------------------
// 計算
//--------------------------------------------------------------
void kgmvstats_::kgMvMean::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvSum::calc(rls,inp);
	if(stocksize()<=0){
		for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
			rls.at(i).null(true);		
		}
		return;
	}	

	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		_mean.at(i)= rls.at(i).r()/stocksize();
		rls.at(i).r(_mean.at(i));
	}
}
//--------------------------------------------------------------
// データの初期化
//--------------------------------------------------------------
void kgmvstats_::kgMvMean::clear()
{
	for(vector<double>::size_type i = 0 ;i < _mean.size() ;i++){
		_mean.at(i) = 0; 
	}
	kgMvSum::clear();
}

// =============================================================================
// kgMvDevsq 二乗差
// =============================================================================
//--------------------------------------------------------------
// 計算
//--------------------------------------------------------------
void kgmvstats_::kgMvDevsq::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvMean::calc(rls,inp);
	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		vector<double> v = gets(i);
		double t=0;
		for(vector<double>::size_type j = 0 ;j < stocksize() ;j++){
			t += ( (v[j]-rls.at(i).r())*(v[j]-rls.at(i).r()) );
		}
		rls.at(i).r(t);
	}
}
// =============================================================================
// kgMvVar 分散
// =============================================================================
//--------------------------------------------------------------
// データの分散
//--------------------------------------------------------------
void kgmvstats_::kgMvVar::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvDevsq::calc(rls,inp);
	if(stocksize()<=1){
		for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
			rls.at(i).null(true);		
		}
		return;
	}	

	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		rls.at(i).r(rls.at(i).r()/stocksize());
	}
}
// =============================================================================
// kgMvUvar 分散 (不偏推定値)
// =============================================================================
//--------------------------------------------------------------
// データの分散
//--------------------------------------------------------------
void kgmvstats_::kgMvUvar::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvDevsq::calc(rls,inp);
	if(stocksize()<=1){
		for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
			rls.at(i).null(true);		
		}
		return;
	}	
	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		rls.at(i).r(rls.at(i).r()/(stocksize()-1));
	}
}
// =============================================================================
// kgMvSd 標準偏差
// =============================================================================
//--------------------------------------------------------------
// 計算
//--------------------------------------------------------------
void kgmvstats_::kgMvSd::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvDevsq::calc(rls,inp);
	if(stocksize()<=1){
		for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
			rls.at(i).null(true);		
		}
		return;
	}	
	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		rls.at(i).r(sqrt(rls.at(i).r()/stocksize()));
	}
}
// =============================================================================
// kgMvUsd 標準偏差 (不偏推定値)
// =============================================================================
//--------------------------------------------------------------
// 計算
//--------------------------------------------------------------
void kgmvstats_::kgMvUsd::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvDevsq::calc(rls,inp);
	if(stocksize()<=1){
		for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
			rls.at(i).null(true);		
		}
		return;
	}
	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		rls.at(i).r(sqrt(rls.at(i).r()/(stocksize()-1)));
	}
}

// =============================================================================
// kgMvCv 変動係数
// =============================================================================
//--------------------------------------------------------------
// 計算
//--------------------------------------------------------------
void kgmvstats_::kgMvCv::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvSd::calc(rls,inp);
	if(stocksize()<=1){
		for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
			rls.at(i).null(true);		
		}
		return;
	}
	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		rls.at(i).r( 100.0 * rls.at(i).r()/ kgMvMean::get(i));
	}
}
// =============================================================================
// kgMvMin 最小値
// =============================================================================
//--------------------------------------------------------------
// 計算
//--------------------------------------------------------------
void kgmvstats_::kgMvMin::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvBase::update(inp);
	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		vector<double> v = gets(i);
		double min=DBL_MAX;
		for(size_t j = 0 ;j < stocksize() ;j++){
			if(min > v[j]){ min = v[j]; }
		}
		rls.at(i).r(min);
	}
}
// =============================================================================
// kgMvMin 最大値
// =============================================================================
//--------------------------------------------------------------
// 計算
//--------------------------------------------------------------
void kgmvstats_::kgMvMax::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvBase::update(inp);
	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		vector<double> v = gets(i);
		double max=-DBL_MAX;
		for(size_t j = 0 ;j < stocksize() ;j++){
			if(max < v[j]){ max = v[j]; }
		}
		rls.at(i).r(max);
	}
}
// =============================================================================
// kgMvRane 範囲
// =============================================================================
//--------------------------------------------------------------
// 計算
//--------------------------------------------------------------
void kgmvstats_::kgMvRange::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvBase::update(inp);
	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		vector<double> v = gets(i);
		double min=  DBL_MAX;
		double max= -DBL_MAX;
		for(size_t j = 0 ;j < stocksize() ;j++){
			if(min > v[j]){ min = v[j]; }
			if(max < v[j]){ max = v[j]; }
		}
		rls.at(i).r(max-min);
	}
}
// =============================================================================
// kgMvSkew 歪度
// =============================================================================
//--------------------------------------------------------------
// 計算
//--------------------------------------------------------------
void kgmvstats_::kgMvSkew::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvMean::calc(rls,inp);
	size_t cnt =stocksize();
	if(cnt <= 2 ){
		for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
			rls.at(i).null(true);		
		}
		return;
	}
	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		vector<double> v = gets(i);
		double t2,t3;
		t2=0; t3=0;
		for(size_t j = 0 ;j < cnt ;j++){
			double dv = v[j]-rls.at(i).r();
			t2 += ( dv*dv );
			t3 += ( dv*dv*dv );
		}
		if(t2==0){ 
			rls.at(i).null(true);		
			continue;		
		}		
		rls.at(i).r( t3 / (pow(t2/cnt,1.5) * cnt) );
	}
}
// =============================================================================
// kgMvUskew 歪度 (不偏推定値)
// =============================================================================
//--------------------------------------------------------------
// 計算
//--------------------------------------------------------------
void kgmvstats_::kgMvUskew::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvMean::calc(rls,inp);
	size_t cnt =stocksize();
	if(cnt <= 2 ){
		for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
			rls.at(i).null(true);		
		}
		return;
	}
	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		vector<double> v = gets(i);
		double t2,t3;
		t2=0; t3=0;
		for(size_t j = 0 ;j < cnt ;j++){
			double dv = v[j]-rls.at(i).r();
			t2 += ( dv*dv );
			t3 += ( dv*dv*dv );
		}
		if(t2==0){ 
			rls.at(i).null(true);		
			continue;		
		}		
		rls.at(i).r( t3 / pow(t2/(cnt-1),1.5) * cnt / ((cnt-1)*(cnt-2))   );
	}
}
// =============================================================================
// kgMvKurt 尖度
// =============================================================================
//--------------------------------------------------------------
// 計算
//--------------------------------------------------------------
void kgmvstats_::kgMvKurt::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvMean::calc(rls,inp);
	size_t cnt =stocksize();
	if(cnt == 0 ){
		for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
			rls.at(i).null(true);		
		}
		return;
	}
	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		vector<double> v = gets(i);
		double t2,t4;
		t2=0; t4=0;
		for(size_t j = 0 ;j < cnt ;j++){
			double dv = v[j]-rls.at(i).r();
			t2 += ( dv*dv );
			t4 += ( dv*dv*dv*dv );
		}
		if(t2==0){ 
			rls.at(i).null(true);		
			continue;		
		}
		rls.at(i).r( t4 / (pow(t2/cnt,2) * cnt) -3.0);
	}
}
// =============================================================================
// kgMvUkurt 尖度 (不偏推定値)
// =============================================================================
//--------------------------------------------------------------
// 計算
//--------------------------------------------------------------
void kgmvstats_::kgMvUkurt::calc(vector<kgVal>& rls ,vector<double>& inp)
{
	kgMvMean::calc(rls,inp);
	size_t cnt =stocksize();
	if(cnt <= 3 ){
		for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
			rls.at(i).null(true);		
		}
		return;
	}
	for(vector<kgVal>::size_type i = 0 ;i < rls.size() ;i++){
		vector<double> v = gets(i);
		double t2,t4;
		t2=0; t4=0;
		for(size_t j = 0 ;j < cnt ;j++){
			double dv = v[j]-rls.at(i).r();
			t2 += ( dv*dv );
			t4 += ( dv*dv*dv*dv );
		}
		if(t2==0){ 
			rls.at(i).null(true);		
			continue;		
		}
		rls.at(i).r( t4/(pow(t2/(cnt-1),2))*cnt*(cnt+1)/(cnt-1)/(cnt-2)/(cnt-3)
									- 3.0*(cnt-1)*(cnt-1)/(cnt-2)/(cnt-3));
	}
}

//--------------------------------------------------------------
// コンストラクタ(モジュール名，バージョン登録)
// -----------------------------------------------------------------------------
kgMvstats::kgMvstats(void){
	#ifdef ENG_FORMAT
		#include <help/en/kgmvstatsHelp.h>
	#else
		#include <help/jp/kgmvstatsHelp.h>
	#endif
	_name    = "kgmvstats";
	_version = "1.0";
}
// -----------------------------------------------------------------------------
// 入出力ファイルオープン
// -----------------------------------------------------------------------------
void kgMvstats::setArgs(void){
	// パラメータチェック
	_args.paramcheck("f=,i=,o=,k=,c=,skip=,t=");

	// 入出力ファイルオープン
	_iFile.open(_args.toString("i=",false), _env,_nfn_i);
  _oFile.open(_args.toString("o=",false), _env,_nfn_o);
  _oFile.setPrecision(_precision);
	_iFile.read_header();

	// f= 項目引数のセット
	vector< vector<kgstr_t> > vs_f = _args.toStringVecVec("f=",':',2,true,true);
	_fField.set(vs_f, &_iFile,_fldByNum);

	// k= 項目引数のセット
	vector<kgstr_t> vs = _args.toStringVector("k=",false);
	_kField.set(vs,  &_iFile,_fldByNum);

	// t= の値をセット
	kgstr_t strT=_args.toString("t=",true);
	_term = aToSizeT(strT.c_str());

	kgstr_t strS=_args.toString("skip=",false);
	if(strS.empty()){
		_skip = _term-1;
	}else{
		_skip = atoi(strS.c_str());
		if(_skip>=_term){ _skip = _term -1; }
	}
	// c= 計算方法のセット
	_c_type = _args.toString("c=", true);
    	 	 if(_c_type=="sum"   ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvSum  >(),lambda::_1,lambda::_2,lambda::_3); }
	  else if(_c_type=="mean"  ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvMean >(),lambda::_1,lambda::_2,lambda::_3); }
		else if(_c_type=="devsq" ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvDevsq>(),lambda::_1,lambda::_2,lambda::_3); }
		else if(_c_type=="var"   ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvVar  >(),lambda::_1,lambda::_2,lambda::_3); }
		else if(_c_type=="uvar"  ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvUvar >(),lambda::_1,lambda::_2,lambda::_3); }
		else if(_c_type=="sd"    ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvSd   >(),lambda::_1,lambda::_2,lambda::_3); }
		else if(_c_type=="usd"   ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvUsd  >(),lambda::_1,lambda::_2,lambda::_3); }
		else if(_c_type=="cv"    ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvCv   >(),lambda::_1,lambda::_2,lambda::_3); }
		else if(_c_type=="min"   ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvMin  >(),lambda::_1,lambda::_2,lambda::_3); }
		else if(_c_type=="max"   ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvMax >(),lambda::_1,lambda::_2,lambda::_3); }
		else if(_c_type=="range" ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvRange>(),lambda::_1,lambda::_2,lambda::_3); }
		else if(_c_type=="skew"  ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvSkew >(),lambda::_1,lambda::_2,lambda::_3); }
		else if(_c_type=="uskew" ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvUskew>(),lambda::_1,lambda::_2,lambda::_3); }
		else if(_c_type=="kurt"  ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvKurt >(),lambda::_1,lambda::_2,lambda::_3); }
		else if(_c_type=="ukurt" ){_function = lambda::bind(lambda::new_ptr<kgmvstats_::kgMvUkurt>(),lambda::_1,lambda::_2,lambda::_3); }
	else {
		ostringstream ss;
		ss << "unknown keyword: " << _c_type << ": c=sum|mean|devsq|var|uvar|sd|usd|cv|min|max|range|skew|kurt|uskew|ukurt" << _c_type;
		throw kgError(ss.str());	
	}
}
// -----------------------------------------------------------------------------
// 実行
// -----------------------------------------------------------------------------
void kgMvstats::run(void) try {

	setArgs();
	// 入力ファイルにkey項目番号をセットする．
	_iFile.setKey(_kField.getNum());
	
	// 項目名の出力
  _oFile.writeFldName(_fField, true);

	size_t fldSize=_fField.size();

	// 結果格納変数の領域確保
	vector<kgVal>  result(fldSize ,kgVal('N'));
	vector<double> inp   (fldSize ,0);
	
	// 計算クラスセット
	kgmvstats_::kgMvBase* kmvb = _function(_skip,_term,fldSize);

	while(_iFile.blkset()!=EOF){

		while(  EOF != _iFile.blkread() ){
			for(size_t i=0; i<fldSize; i++){
				char* str=_iFile.getBlkVal(_fField.num(i));
				if(*str=='\0') { inp[i]=0; }
				else					 { inp[i]=atof(str);}
			}
			kmvb->calc(result,inp);
			if(kmvb->stocksize()>_skip){
				_oFile.writeFld(_iFile.getBlkFld(),_fField.getFlg_p(),result);
			}
		}
		kmvb->clear();
	}
	_iFile.close();
	_oFile.close();

	// 終了処理(メッセージ出力,thread pipe終了通知)
	successEnd();

// 例外catcher
}catch(kgError& err){
	errorEnd(err);
}
