<?php

require_once "Date.php";
require_once "Date/Calc.php";
require_once "Date/Span.php";
require_once "Date/TimeZone.php";
Zend_Loader::loadClass("CFW_Util_String");
class CFW_Util_Date{
	/*
	 * 日付+時刻
	 */
	const DATETIME_LONG = "^\d{4}[/\-]([0]?\d|1[0-2])[/\-]([0-2]?[0-9]|3[0-1]) +([01]?[0-9]|2[0-3]):[0-5]?[0-9]:[0-5]?[0-9]$";
	/*
	 * 日付+時刻
	 */
	const DATETIME_SMALL = "^\d{4}[/\-]([0]?\d|1[0-2])[/\-]([0-2]?[0-9]|3[0-1]) +([01]?[0-9]|2[0-3]):[0-5]?[0-9]$";
	/*
	 * 日付  2005/11/1 2005/1/1 2005/02/03 05/01/01  NG：2005/01 2005/1/32
	 */
	const DATE_LONG = "^\d{4}[/\-]([0]?\d|1[0-2])[/\-]([0-2]?[0-9]|3[0-1])$";
	/*
	 * 短い日付
	 */
	const DATE_SHORT = "^\d{2}[/\-]([0]?\d|1[0-2])[/\-]([0-2]?[0-9]|3[0-1])$";
	/*
	 * 日付+時刻 区切りなし
	 */
	const DATETIME_LONG_NO_DELIMITER = "^\d{4}([0]\d|1[0-2])([0-2]\d|3[0-1])([01][0-9]|2[0-3])[0-5][0-9][0-5][0-9]$";
	/*
	 * 日付+時刻 区切りなし
	 */
	const DATETIME_SMALL_NO_DELIMITER = "^\d{4}([0]\d|1[0-2])([0-2]\d|3[0-1])([01]\d|2[0-3])[0-5]\d$";
	/*
	 * 長い日付 区切りなし
	 */
	const DATE_LONG_NO_DELIMITER = "^\d{4}(0[0-9]|1[012])([012][0-9]|3[01])$";
	/*
	 * 短い日付 区切りなし
	 */
	const DATE_SHORT_NO_DELIMITER = "^\d{2}(0[0-9]|1[0-2])([012][0-9]|3[01])$";

	/**
	 * 年齢を月表示にするための年数
	 * @var integer
	 */
	public static $YearsForAgeByMonth = 2;

	/**
	 * 曜日インデックスの開始
	 * @var integer
	 */
	public static $WeekDayBase = 0;

	/**
	 * US月表示を大文字にする
	 * @var bool
	 */
	public static $UseUpperCase = true;

	/**
	 * 年を示唆するための日数
	 * @var integer
	 */
	public static $DaysToSuggestYear = -3;

	/**
	 * Dateが期待される値をDateTimeに
	 * @param $value 変換する対象の文字列型の式
	 * @return Date 変換した日付
	 */
	public static function toDate($value){
		if ($value == null) return null;
		if (CFW_Util_String::NormalizeString(CFW_Util_String::toString($value), 3) == "") {
			return null;
		}
		if ($value instanceof Date) return $value;
		return self::parse(CFW_Util_String::toString($value));
	}
	/**
	 * 今日(時刻を含まない= 0時0分)
	 * @return  今日(時刻を含まない= 0時0分)
	 */
	public static function today(){
		$date = new Date();
		$result = new Date($date->getYear().$date->getMonth().$date->getDay());
		return $result;
	}

	/**
	 * 現在(時刻を含む)
	 * @return 現在(時刻を含む)
	 */
	public static function now(){
		$result = new Date();
		return $result;
	}

	/**
	 * 年推論の基準にする日付
	 * @return object 年推論の基準にする日付
	 */
	public static function dateToSuggestYear(){
		return self::addDays(self::today(), self::$DaysToSuggestYear);
	}
	/**
	 * 年推論
	 * @param object $d
	 * @return object 年推論の基準にする日付以降でもっとも現在に近い日付
	 */
	public static function inferYear($d){
		$d = new Date($d);
		if ($d->compare($d, self::dateToSuggestYear()) < 0) {
			return self::addYears($d, 1);
		}
		return $d;
	}

	/**
	 * Date型をString型にして出力
	 * @param date $date 変換対象Date型
	 * @param string $formatString フォーマット方式
	 * @return string 変換したString型
	 */
	public static function format($date, $formatString = "%Y/%m/%d %H:%M:%S"){
		try{
			if ($date != null){
			    if($date instanceof DateTime){
			        $dformat = str_replace("%","", $formatString);
                    $result = $date->format($dformat);
                    return $result;
			    }
			    else{
    				$datetime = new Date($date);
    				$result = $datetime->format($formatString);

    				return $result;
			    }
			}
			else {
				return "";
			}
		}catch (Exception $e){
			return "";
		}
	}

	/**
	 * 年、月、日の要素から日付生成
	 * @param integer $y 年をあらわす数値
	 * @param integer $m 月をあらわす数値
	 * @param integer $d 日をあらわす数値
	 * @return 指定年月日から成る日付
	 */
	public static function create($y, $m , $d){
		$y = CFW_Util_Number::formatZero($y, 4);
		$m = CFW_Util_Number::formatZero($m, 2);
		$d = CFW_Util_Number::formatZero($d, 2);
		return new Date($y.$m.$d);
	}
	/**
	 * 年、月の要素から日付生成
	 * 日は1日を仮定
	 * @param integer $y 年をあらわす数値
	 * @param integer $m 月をあらわす数値
	 * @return 指定年月＋1日の日付
	 */
	public static function createFromYearMonth($y, $m){
		$y = CFW_Util_Number::formatZero($y, 4);
		$m = CFW_Util_Number::formatZero($m, 2);
		return new Date($y.$m."01");
	}
	/**
	 * 月日の要素から日付生成
	 * 年は基準日以降の最初の年を仮定
	 * @param integer $m 月をあらわす数値
	 * @param integer $d 日をあらわす数値
	 * @return object 指定月日でかつ基準日以降の日付
	 */
	public static function createFromMonthDay($m, $d){
		$m = CFW_Util_Number::formatZero($m, 2);
		$d = CFW_Util_Number::formatZero($d, 2);
		$today = self::today();
		$newDate = new Date($today->getYear().$m.$d);

		return self::inferYear($newDate);
	}

	/**
	 * 時刻を捨てる
	 * @param object $object
	 * @return 指定日付の0時0分を表す日付
	 */
	public static function truncateTime($object){
		if ($object == null) return null;

		$date = new Date($object);
		return new Date($date->getYear().$date->getMonth().$date->getDay());
	}
	 /**
	  * 日数計算
	  * @param object $d1 基準日
	  * @param object $d2 比較対象日
	  * @return $intDiff 基準日から比較対象日までの日数
	  */
	public static function diffDays($d1, $d2){
		$standardDay = new Date($d1);
		$comparisonDay = new Date($d2);
		$span = new Date_Calc();
		$intDiff = $span->dateToDays($comparisonDay->getDay(),$comparisonDay->getMonth(),$comparisonDay->getYear()) -
							$span->dateToDays($standardDay->getDay(),$standardDay->getMonth(),$standardDay->getYear());

		return $intDiff;
	}

	/**
	 * 年齢
	 * @param object $birthday 誕生日
	 * @param object $baseDate 基準日
	 * @return integer 年齢
	 */
	public static function ageYear($birthday, $baseDate){
		if ($birthday == null) return 0;
		if ($baseDate == null) return 0;
		$d = new Date($birthday);
		$x = new Date($baseDate);
		return $x->getYear() - $d->getYear() -
				((($d->getMonth() * 100 + $d->getDay()) > ($x->getMonth() * 100 + $x->getDay())) ? 1 : 0);
	}

	/**
	 * 年齢(月)
	 * @param object $birthday 誕生日
	 * @param object $baseDate 基準日
	 * @return integer 年齢(月単位)
	 */
	public static function ageMonth($birthday, $baseDate){
		if ($birthday == null) return 0;
		if ($baseDate == null) return 0;
		$d = new Date($birthday);
		$x = new Date($baseDate);
		return (($x->getYear() - $d->getYear()) * 12) + $x->getMonth() - $d->getMonth() -
				(($d->getDay() > $x->getDay()) ? 1 : 0);
	}

	/**
	 * 基準日と生年月日から99Y（24ヶ月以上）または99M（２４ヶ月未満）の文字列を得る
	 * @param object $birthday 誕生日
	 * @param object $baseDate 基準日
	 * @return string
	 */
	public static function ageString($birthday, $baseDate){
		$result = "";
		$age = self::ageYear($birthday, $baseDate);
		if ($age < self::$YearsForAgeByMonth) {
			$result = strval(self::ageMonth($birthday, $baseDate))."M";
		} else {
			$result = strval($age)."Y";
		}
		return $result;
	}

	/**
	 * 基準日と生年月日から99歳または99ヶ月（２４ヶ月未満）の文字列を得る
	 * @param object $birthday 誕生日
	 * @param object $baseDate 基準日
	 * @return string
	 */
	public static function ageStringJP($birthday, $baseDate){
		$result = "";
		$age = self::ageYear($birthday, $baseDate);
		if ($age < self::$YearsForAgeByMonth) {
			$result = strval(self::ageMonth($birthday, $baseDate))."ヶ月";
		} else {
			$result = strval($age)."歳";
		}
		return $result;
	}

	/**
	 * 曜日のインデックス(WeekDayBaseから始まる)
	 * @param object $object 指定日付
	 * @return integer 曜日のインデックス
	 */
	public static function dayOfWeekIndex($object){
		if ($object == null) return self::$WeekDayBase - 1;
		$date = new Date($object);
		$index = $date->getDayofWeek();
		return $index + self::$WeekDayBase;
	}

	/**
	 * 曜日文字列
	 * @param object $object 指定日付
	 * @return string 指定日付の曜日
	 */
	public static function dayOfWeekString($object){
		if ($object == null) return "";
		$date = new Date($object);
		$result = $date->getDayName(3);
		return (self::$UseUpperCase) ? mb_strtoupper($result) : $result;
	}

	/**
	 * 曜日文字列
	 * @param object $object 指定日付
	 * @return string 指定日付の曜日
	 */
	public static function dayOfWeekStringJP($object){
		if ($object == null) return "";
		$weekDay = array("日","月","火","水","木","金","土");
		$date = new Date($object);
		$result = $weekDay[$date->getDayOfWeek()];
		return $result;

	}

	/**
	 * 範囲の正当性を検証
	 * @param object $object 検証する対象
	 * @param object $min 範囲(最小値)
	 * @param object $max 範囲(最大値)
	 * @return boolean 範囲が正当ならばtrue,でなければfalse
	 */
	public static function isInRange($object, $min, $max){
		if ($object == null) return false;
		if (($min == null) && ($max == null)) return false;

		$target = new Date($object);
		$maxDate = new Date($max);
		$minDate = new Date($min);
		if ($min == null) {
			if ($target->compare($target,$maxDate) > 0) return false;
		}
		elseif ($max == null) {
			if ($target->compare($target,$minDate) < 0) return false;
		}
		else {
			if ($target->compare($target,$minDate) < 0) return false;
			if ($target->compare($target,$maxDate) > 0) return false;
		}
		return true;
	}

	/**
	 * 指定日より前か
	 * @param object $d1 対象日
	 * @param object $d2 基準日
	 * @return boolean 対象日が基準日より前ならtrue
	 */
	public static function isBefore($d1, $d2){
		if ($d1 == null) { return false; }
		if ($d2 == null) { return false; }
		$objectDate = new Date($d1);
		$baseDate = new Date($d2);
		if ($objectDate->compare($objectDate, $baseDate) < 0){
			return true;
		}
		else {
			return false;
		}
	}
	/**
	 * 指定日より前か同じか
	 * @param object $d1 対象日
	 * @param object $d2 基準日
	 * @return boolean 対象日が基準日より前か同じならtrue
	 */
	public static function isBeforeOrSame($d1, $d2){
		if ($d1 == null) { return false; }
		if ($d2 == null) { return false; }
		$objectDate = new Date($d1);
		$baseDate = new Date($d2);
		if ($objectDate->compare($objectDate, $baseDate) <= 0){
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * 指定日より後か
	 * @param object $d1 対象日
	 * @param object $d2 基準日
	 * @return boolean 対象日が基準日より後ならtrue
	 */
	public static function isAfter($d1, $d2){
		if ($d1 == null) { return false; }
		if ($d2 == null) { return false; }
		$objectDate = new Date($d1);
		$baseDate = new Date($d2);
		if ($objectDate->compare($objectDate, $baseDate) > 0){
			return true;
		}
		else {
			return false;
		}
	}
	/**
	 * 指定日より後か同じか
	 * @param object $d1 対象日
	 * @param object $d2 基準日
	 * @return boolean 対象日が基準日より後か同じならtrue
	 */
	public static function isAfterOrSame($d1, $d2){
		if ($d1 == null) { return false; }
		if ($d2 == null) { return false; }
		$objectDate = new Date($d1);
		$baseDate = new Date($d2);
		if ($objectDate->compare($objectDate, $baseDate) >= 0){
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * 指定日と同じか
	 * @param object $d1 対象日
	 * @param object $d2 基準日
	 * @return boolean 対象日が基準日と同じならtrue
	 */
	public static function isSame($d1, $d2){
		if ($d1 == null) { return false; }
		if ($d2 == null) { return false; }
		$objectDate = new Date($d1);
		$baseDate = new Date($d2);
		if ($objectDate->compare($objectDate, $baseDate) == 0){
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * 文字列を解析して日付型に変換
	 * @param string $s 日付形式の文字列
	 * @return object 解析結果の日付。解析エラーの場合null
	 */
	public static function parse($s){
		if ($s == "") return null;
		$date = $s;
		//区切りあり
        //yyyy/M/d H:m:s
        if (mb_ereg_match(self::DATETIME_LONG, $s)) {
        	$dateAndTimeArray = mb_split(" ", $s);
        	$dateArray = $dateAndTimeArray[0];
        	$timeArray = $dateAndTimeArray[1];
        	$dateArray = mb_split("-", mb_ereg_replace("/", "-", $dateArray));
        	$year = $dateArray[0];
        	$month = $dateArray[1];
        	$day = $dateArray[2];
        	$timeArray = mb_split(":", $timeArray);
        	$hour = $timeArray[0];
        	$minute = $timeArray[1];
        	$second = $timeArray[2];
        	$month = CFW_Util_Number::formatZero($month, 2);
        	$day = CFW_Util_Number::formatZero($day, 2);
        	$hour = CFW_Util_Number::formatZero($hour, 2);
        	$minute = CFW_Util_Number::formatZero($minute, 2);
        	$second = CFW_Util_Number::formatZero($second, 2);

        	$date = $year.$month.$day.$hour.$minute.$second;
        	return new Date($date);
        }
        //yyyy/M/d H:m
        if (mb_ereg_match(self::DATETIME_SMALL, $s)) {
        	$dateAndTimeArray = mb_split(" ", $s);
        	$dateArray = $dateAndTimeArray[0];
        	$timeArray = $dateAndTimeArray[1];
        	$dateArray = mb_split("-", mb_ereg_replace("/", "-", $dateArray));
        	$year = $dateArray[0];
        	$month = $dateArray[1];
        	$day = $dateArray[2];
        	$timeArray = mb_split(":", $timeArray);
        	$hour = $timeArray[0];
        	$minute = $timeArray[1];
        	$month = CFW_Util_Number::formatZero($month, 2);
        	$day = CFW_Util_Number::formatZero($day, 2);
        	$hour = CFW_Util_Number::formatZero($hour, 2);
        	$minute = CFW_Util_Number::formatZero($minute, 2);
        	$date = $year.$month.$day.$hour.$minute."00";
        	return new Date($date);
        }
        //yyyy/M/d
        if (mb_ereg_match(self::DATE_LONG, $s)) {
        	$dateArray = mb_split("-" ,mb_ereg_replace("/" ,"-", $s));
        	$year = $dateArray[0];
        	$month = $dateArray[1];
        	$day = $dateArray[2];

        	$month = CFW_Util_Number::formatZero($month, 2);
        	$day = CFW_Util_Number::formatZero($day, 2);
        	$date = $year.$month.$day;
        	return new Date($date);
        }
        //yy/M/d
        if (mb_ereg_match(self::DATE_SHORT, $s)) {
        	$dateArray = mb_split("-", mb_ereg_replace("/", "-", $s));
        	$year = $dateArray[0];
        	$month = $dateArray[1];
        	$day = $dateArray[2];

        	if ($year > 69) {
        		$year = "19".$year;
        	}
         	elseif ($year <= 69) {
        		$year = "20".$year;
        	}
        	$month = CFW_Util_Number::formatZero($month, 2);
        	$day = CFW_Util_Number::formatZero($day, 2);
        	$date = $year.$month.$day;
        	return new Date($date);
        }
        //区切りなし
        //yyyyMMddHHmmss
        if (mb_ereg_match(self::DATETIME_LONG_NO_DELIMITER, $s)) {
        	return new Date($date);
        }
        //yyyyMMddHHmm
        if (mb_ereg_match(self::DATETIME_SMALL_NO_DELIMITER, $s)) {
        	$date = $s."00";
        	return new Date($date);
        }
        //yyyyMMdd
        if (mb_ereg_match(self::DATE_LONG_NO_DELIMITER, $s)) {
        	return new Date($date);
        }
        //yyMMdd
        if (mb_ereg_match(self::DATE_SHORT_NO_DELIMITER, $s)) {
        	$shortYear = mb_substr($s, 0, 2);
        	if ($shortYear > 69) {
        		$date = "19".$s;
        	} elseif ($shortYear <= 69) {
        		$date = "20".$s;
        	}
        	return new Date($date);
        }
        return null;
	}

	/**
	 * 日付計算
	 * @param object $date 計算の基準日
	 * @param integer $diff 加算する日数、月数、年数
	 * @param string $unit 単位をあらわす文字 y,m,d
	 * @return object 加算結果
	 */
	public static function addDate($date, $diff, $unit){
		$d = new Date($date);
		if (mb_strtolower($unit) == "y") {
			$d->addYears($diff);
		}
		elseif (mb_strtolower($unit) == "m") {
			$d->addMonths($diff);
		}
		elseif (mb_strtolower($unit) == "d") {
			$d->addDays($diff);
		}
		return $d;
	}

	/**
	 * 何ヶ月か後
	 * @param integer $year 年
	 * @param integer $month 月
	 * @param integer $after 加算する月数
	 * @return object 指定年月のafterヶ月後の月初
	 */
	public static function monthAfter($year, $month, $after){
		if ($year < 0) return null;
		if ($month < 1) return null;
		if ($month > 12) return null;

		$baseDate = self::create($year, $month, 1);
		return self::addDate($baseDate, $after, "m");
	}
	/**
	 * 何ヶ月か後の月末
	 * @param integer $year 年
	 * @param integer $month 月
	 * @param integer $after 加算する月数
	 * @return object 指定年月のafterヶ月後の月末
	 */
	public static function endOfMonthAfter($year, $month, $after){
		return self::endOfMonth(self::monthAfter($year, $month, $after));
	}
	/**
	 * 指定日付の月の月末
	 * @param object $d 指定日付
	 * @return object 指定日付の月の月末
	 */
	public static function endOfMonth($d){
		if ($d == null) return null;
		$next = self::addDate($d, 1, "m");
		$days = $next->getDay();
		return self::addDate($next, -$days, "d");
	}
	/**
	 * 指定月の月末
	 * @param integer $year 指定年
	 * @param integer $month 指定月
	 * @return object 指定年月の月末
	 */
	public static function endOfYearMonth($year, $month){
		$next = self::addDate(self::createFromYearMonth($year, $month), 1, "m");
		$days = $next->getDay();
		return self::addDate($next, -$days, "d");
	}
	/**
	 * 月末チェック
	 * @param object $d 指定日付
	 * @return boolean 月末ならtrue
	 */
	public static function isEndOfMonth($d){
		if ($d == null) return false;
		$date = new Date($d);
		$end = self::endOfMonth($date);
		return self::isSame($date, $end);
	}

	/**
	 * 日数加算
	 * @param object $d 日付
	 * @param integer $days 加算日数
	 * @return object 加算後の日付
	 */
	public static function addDays($d, $days){
		if ($d == null) return null;
		$next = self::addDate($d, $days, "d");

		return $next;
	}
	/**
	 * 月数加算
	 * @param object $d 日付
	 * @param integer $months 加算月数
	 * @return object 加算後の日付
	 */
	public static function addMonths($d, $months){
		if ($d == null) return null;
		$next = self::addDate($d, $months, "m");

		return $next;
	}
	/**
	 * 年加算
	 * @param object $d 日付
	 * @param integer $years 加算年数
	 * @return object 加算後の日付
	 */
	public static function addYears($d, $years){
		if ($d == null) return null;
		$next = self::addDate($d, $years, "y");

		return $next;
	}

	/**
	 * 次の月
	 * @param integer $year 年
	 * @param integer $month 月
	 * @return object 指定年月の翌月の１日
	 */
	public static function nextMonth($year, $month){
		return self::addMonths(self::createFromYearMonth($year, $month), 1);
	}
	/**
	 * 前の月
	 * @param integer $year 年
	 * @param integer $month 月
	 * @return object 指定年月の昨月の１日
	 */
	public static function previousMonth($year, $month){
		return self::addMonths(self::createFromYearMonth($year, $month), -1);
	}
}