/**
 * @desc 郵便番号検索
 * 
 * 郵便番号検索サービスを使用するUIの実装.
 * 
 * 一般的にはXHRの仕様により、別サイトへのXHR要求は出来ないので自サイトにプロキシを用意し、プロキシに対して検索要求を送る。
 * 別途郵便番号検索サービスが必要。(postalsearch/postalSearchService.phpが用意されている)
 */
postalSearch = {
	/**
	 * @desc 最後に検索した時のパラメータ。同じパラメータで検索したら前の結果をそのまま表示する
	 */
	lastParameter : {zip : "",pref : "",city : ""},
	/**
	 * @desc 設定。 
	 * loadingIcon : ロード中に表示するアイコン。
	 * serviceUrl : 郵便番号検索サービスのURL 
	 * 
	 */
	config: {},
	/**
	 * @desc 初期化
	 */
	init : function(config){
		this.config = config;
    	//ダイアログ
    	this.createDialog();
    	//データソース
    	this.createDataSource();
    	//データテーブル
    	this.createDataTable();
    	this.createButtons();
    	//行を選択したイベント定義
    	this.selectedEvent = new YAHOO.util.CustomEvent("selected", this);
	}
	/**
	 * @desc 検索結果表示ダイアログを生成
	 */
	,createDialog : function(){
    	//郵便番号リスト表示ダイアログ作成
    	this.dialog = new YAHOO.widget.Panel("postalSearchDialog", 
    	    	{ width:"620px", visible:false, constraintoviewport:true,draggable:true,context:["zip","tl","tr"] } 
    	);
    	this.dialog.setHeader("郵便番号検索");
    	this.dataTableId = YAHOO.util.Dom.generateId();
    	this.dialog.setBody("<div id=\"" + this.dataTableId +"\"></div>");
    	
    	this.dialog.render(document.body);
	}
	/**
	 * @desc プロクシに接続するデータソースの生成
	 */
	,createDataSource : function(){
        this.myDataSource = new YAHOO.util.DataSource(this.config.serviceUrl + "?");
        this.myDataSource.responseType = YAHOO.util.DataSource.TYPE_JSON;
        this.myDataSource.connXhrMode = "queueRequests";
        this.myDataSource.responseSchema = {
            resultsList: "data",
            fields: ["zipcode7","pref","city","town"]
        };
	}
	/**
	 * @desc 一覧表示用データテーブルを生成
	 */
	,createDataTable : function(){
    	//データテーブル
        var myColumnDefs = [
                            {key:"zipcode7", label:"zipcode7",width:55},
                            {key:"pref",width:60},
                            {key:"city",width:120},
                            {key:"town",width:250}
                            
                        ];
        
    	this.myDataTable = new YAHOO.widget.ScrollingDataTable(this.dataTableId, 
    			myColumnDefs,
        		this.myDataSource, {height:"20em",width:"600px",selectionMode:"single",initialLoad :false }
    	);
       	this.myDataTable.subscribe("rowMouseoverEvent", this.myDataTable.onEventHighlightRow);
       	this.myDataTable.subscribe("rowMouseoutEvent", this.myDataTable.onEventUnhighlightRow);
       	this.myDataTable.subscribe("rowClickEvent", this.myDataTable.onEventSelectRow);
       	this.myDataTable.subscribe("rowDblclickEvent", this.datatable_rowdoubleclick,null,this);
       	
	}
	/**
	 * @desc 制御用ボタンを生成
	 */
	,createButtons : function(){
		var okButton = new YAHOO.widget.Button({
		    type: "button", 
		    label: "選択",
		    onclick :  { fn:this.selectRow_onclick,scope:this }
		}); 
		var cancelButton = new YAHOO.widget.Button({
		    type: "button", 
		    label : "閉じる",
		    onclick :  { fn:function(){this.dialog.hide();},scope:this }
		});
        var oSpan = document.createElement("span");
		okButton.appendTo(oSpan);
		cancelButton.appendTo(oSpan);
		this.dialog.setFooter(oSpan);		
		
	} 
	/**
	 * @desc 検索実行
	 * @param zip   郵便番号を指定する(7桁) 少なくとも3桁は必要。ただし都道府県、市区町村郡を指定する場合は自由
	 * @param pref  都道府県を指定
	 * @param city  市区町村郡を指定
	 */
	,doSearch : function(zip,pref,city){
		if( (this.lastParameter.zip == zip )&&(this.lastParameter.pref == pref )&&(this.lastParameter.city == city )){
			this.dialog.show();
			return;
		}
    	this.showLoadingIcon();
    	var query = "zip=" + zip + "&pref=" + pref + "&city=" + city; 
        var callback1 = {
                success : this.postalSearch_complete,
                failure : this.postalSearch_complete,
                scope : this
            };
        this.myDataSource.sendRequest(query,callback1);
        this.lastParameter.zip = zip;
        this.lastParameter.pref = pref;
        this.lastParameter.city = city;
        
	}
	/**
	 * @desc 検索プロクシからレスポンスを受け取った処理
	 */
	,postalSearch_complete : function(){
    	this.hideLoadingIcon();
       	 this.myDataTable.set("sortedBy", null); 
    	 this.myDataTable.onDataReturnReplaceRows.apply(this.myDataTable,arguments);
    	 this.dialog.show();
	}
	/**
	 * 一覧表示の行をダブルクリックした処理
	 */
	,datatable_rowdoubleclick : function(event,target){
		this.selectRow();
	}
	/**
	 * @desc 「選択」ボタンをクリックした処理 
	 */
	,selectRow_onclick : function(event){
		this.selectRow();
	}
	/**
	 * @desc 行選択を確定した処理
	 */
	,selectRow : function(){
		var recordId = this.myDataTable.getLastSelectedRecord();
		if(recordId == null) return ;
		var record =  this.myDataTable.getRecord(recordId);
    		
		this.selectedEvent.fire(record.getData());
		this.dialog.hide();
	}
	/**
	 * @desc ロード中アイコンを表示する
	 */
	,showLoadingIcon : function(){
		if(typeof(this.config.loadingIcon) == "undefined") return;
    	YAHOO.util.Dom.setStyle(this.config.loadingIcon,"visibility","visible");
	}		
	/**
	 * @desc ロード中アイコンを消す
	 */
	,hideLoadingIcon : function(){
		if(typeof(this.config.loadingIcon) == "undefined") return;
    	YAHOO.util.Dom.setStyle(this.config.loadingIcon,"visibility","hidden");
	}		
}
