using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;

namespace CFW.Util.Test
{
    [TestFixture]
    public class StringUtilTest
    {
        [NUnit.Framework.Test]
        public void JoinLine()
        {
            List<string> testList = new List<string>();
            testList.Add("1234568790");
            testList.Add("abcdefghijklmnopqrstuvwxyz");
            testList.Add("");

            string result = CFW.Util.StringUtil.JoinLine(testList);

            Assert.IsNotNull(result);
            Assert.AreEqual("1234568790\r\nabcdefghijklmnopqrstuvwxyz\r\n", result);
        }
        [NUnit.Framework.Test]
        public void JoinLine2()
        {
            string[] testList = new string[] { "1234568790", "abcdefghijklmnopqrstuvwxyz", "" };

            string result = CFW.Util.StringUtil.JoinLine(testList);

            Assert.IsNotNull(result);
            Assert.AreEqual("1234568790\r\nabcdefghijklmnopqrstuvwxyz\r\n", result);
        }
        [NUnit.Framework.Test]
        public void SplitByWidth()
        {
            //                   ----+----1----+----2----+----3----+----4----+----5----+-
            string testString = "1234568790abcdefghijklmnopqrstuvwxy";
            List<string> result = CFW.Util.StringUtil.SplitByWidth(testString, 10);

            Assert.AreEqual(6, result.Count);
            Assert.AreEqual("1234568790", result[0]);
            Assert.AreEqual("abcdefghij", result[1]);
            Assert.AreEqual("klmnopqrst", result[2]);
            Assert.AreEqual("uvwxy", result[3]);
            Assert.AreEqual("", result[4]);
            Assert.AreEqual("", result[5]);


        }
        [NUnit.Framework.Test]
        public void Repeat()
        {
            string result = CFW.Util.StringUtil.Repeat("a", 10);
            Assert.AreEqual("aaaaaaaaaa", result);

        }
        [NUnit.Framework.Test]
        public void Repeat2()
        {
            string result = CFW.Util.StringUtil.Repeat("123", 10);
            Assert.AreEqual("123123123123123123123123123123", result);

        }
        [NUnit.Framework.Test]
        public void Blank()
        {
            string result = CFW.Util.StringUtil.Blank(10);
            Assert.AreEqual("          ", result);

        }
        [NUnit.Framework.Test]
        public void WideBlank()
        {
            string result = CFW.Util.StringUtil.WideBlank(10);
            Assert.AreEqual("@@@@@@@@@@", result);

        }
        [NUnit.Framework.Test]
        public void FillLeftWith()
        {
            string result = CFW.Util.StringUtil.FillLeftWith("abcdefg", "+", 10);
            Assert.AreEqual("+++abcdefg", result);

        }
        [NUnit.Framework.Test]
        public void FillLeftWith2()
        {
            string result = CFW.Util.StringUtil.FillLeftWith("abcdefghijk", "+", 10);
            Assert.AreEqual("abcdefghijk",result);

        }
        [NUnit.Framework.Test]
        public void FillRightWith()
        {
            string result = CFW.Util.StringUtil.FillRightWith("abcdefg", "+", 10);
            Assert.AreEqual("abcdefg+++", result);

        }
        [NUnit.Framework.Test]
        public void FillRightWith2()
        {
            string result = CFW.Util.StringUtil.FillRightWith("abcdefghijk", "+", 10);
            Assert.AreEqual("abcdefghijk", result);

        }
        [NUnit.Framework.Test]
        public void FillBlankLeft()
        {
            string result = CFW.Util.StringUtil.FillBlankLeft("abcdefg", 10);
            Assert.AreEqual("   abcdefg", result);

        }
        [NUnit.Framework.Test]
        public void FillBlankLeft2()
        {
            string result = CFW.Util.StringUtil.FillBlankLeft("abcdefghijk", 10);
            Assert.AreEqual("abcdefghijk", result);

        }
        [NUnit.Framework.Test]
        public void FillBlankRight()
        {
            string result = CFW.Util.StringUtil.FillBlankRight("abcdefg", 10);
            Assert.AreEqual("abcdefg   ", result);

        }
        [NUnit.Framework.Test]
        public void FillBlankRight2()
        {
            string result = CFW.Util.StringUtil.FillBlankRight("abcdefghijk", 10);
            Assert.AreEqual("abcdefghijk", result);

        }
        [NUnit.Framework.Test]
        public void FillWideBlankLeft()
        {
            string result = CFW.Util.StringUtil.FillWideBlankLeft("abcdefg", 10);
            Assert.AreEqual("@@@abcdefg", result);

        }
        [NUnit.Framework.Test]
        public void FillWideBlankLeft2()
        {
            string result = CFW.Util.StringUtil.FillWideBlankLeft("abcdefghijk", 10);
            Assert.AreEqual("abcdefghijk", result);

        }
        [NUnit.Framework.Test]
        public void FillWideBlankRight()
        {
            string result = CFW.Util.StringUtil.FillWideBlankRight("abcdefg", 10);
            Assert.AreEqual("abcdefg@@@", result);

        }
        [NUnit.Framework.Test]
        public void FillWideBlankRight2()
        {
            string result = CFW.Util.StringUtil.FillWideBlankRight("abcdefghijk", 10);
            Assert.AreEqual("abcdefghijk", result);

        }
        [NUnit.Framework.Test]
        public void FillZeroLeft()
        {
            string result = CFW.Util.StringUtil.FillZeroLeft("abcdefg", 10);
            Assert.AreEqual("000abcdefg", result);

        }
        [NUnit.Framework.Test]
        public void FillZeroLeft2()
        {
            string result = CFW.Util.StringUtil.FillZeroLeft("abcdefghijk", 10);
            Assert.AreEqual("abcdefghijk", result);

        }
        [NUnit.Framework.Test]
        public void FillZeroRight()
        {
            string result = CFW.Util.StringUtil.FillZeroRight("abcdefg", 10);
            Assert.AreEqual("abcdefg000", result);

        }
        [NUnit.Framework.Test]
        public void FillZeroRight2()
        {
            string result = CFW.Util.StringUtil.FillZeroRight("abcdefghijk", 10);
            Assert.AreEqual("abcdefghijk", result);

        }

        [NUnit.Framework.Test]
        public void FindKeywordIndex()
        {
            int result = CFW.Util.StringUtil.FindKeywordIndex(new string[] { "aa", "ab", "ac" }, "abcdefghijk");
            Assert.AreEqual(1,result);

            result = CFW.Util.StringUtil.FindKeywordIndex(new string[] { "za", "zb", "zc" }, "abcdefghijk");
            Assert.AreEqual(-1, result);

            result = CFW.Util.StringUtil.FindKeywordIndex(new string[] { "za", "zb", "ab" }, "abcdefghijk");
            Assert.AreEqual(2, result);
        }
        [NUnit.Framework.Test]
        public void FindKeywordIndexRight()
        {
            int result = CFW.Util.StringUtil.FindKeywordIndexRight(new string[] { "aa", "ijk", "ac" }, "abcdefghijk");
            Assert.AreEqual(1, result);

            result = CFW.Util.StringUtil.FindKeywordIndexRight(new string[] { "za", "zb", "zc" }, "abcdefghijk");
            Assert.AreEqual(-1, result);

            result = CFW.Util.StringUtil.FindKeywordIndexRight(new string[] { "za", "zb", "hijk" }, "abcdefghijk");
            Assert.AreEqual(2, result);
        }
        [NUnit.Framework.Test]
        public void Join()
        {
            string[] testList = new string[] { "1234568790", "abcdefghijklmnopqrstuvwxyz", "" };

            string result = CFW.Util.StringUtil.Join(testList);

            Assert.IsNotNull(result);
            Assert.AreEqual("1234568790,abcdefghijklmnopqrstuvwxyz,", result);
        }
        [NUnit.Framework.Test]
        public void Join2()
        {
            string[] testList = new string[] { "1234568790", "abcdefghijklmnopqrstuvwxyz", "" };

            string result = CFW.Util.StringUtil.Join(testList, ":");

            Assert.IsNotNull(result);
            Assert.AreEqual("1234568790:abcdefghijklmnopqrstuvwxyz:", result);
        }
        [NUnit.Framework.Test]
        public void Join3()
        {
            string[] testList = new string[] { "1234568790", "abcdefghijklmnopqrstuvwxyz", "" };

            string result = CFW.Util.StringUtil.Join(testList, ",", "\"");

            Assert.IsNotNull(result);
            Assert.AreEqual("\"1234568790\",\"abcdefghijklmnopqrstuvwxyz\",\"\"", result);
        }
        [NUnit.Framework.Test]
        public void JoinIgnreEmpty()
        {
            string[] testList = new string[] { "1234568790", "", "" };

            string result = CFW.Util.StringUtil.JoinIgnoreEmpty(testList, ",", "\"");

            Assert.IsNotNull(result);
            Assert.AreEqual("\"1234568790\",\"\"", result);
        }
        [NUnit.Framework.Test]
        public void JoinIgnreEmpty2()
        {
            string[] testList = new string[] { "1234568790", null, "" };

            string result = CFW.Util.StringUtil.JoinIgnoreEmpty(testList, ",", "\"");

            Assert.IsNotNull(result);
            Assert.AreEqual("\"1234568790\",\"\"", result);
        }
        [NUnit.Framework.Test]
        public void JoinIgnreEmpty3()
        {
            string[] testList = new string[] { null, "1234568790", "" };

            string result = CFW.Util.StringUtil.JoinIgnoreEmpty(testList, ",", "\"");

            Assert.IsNotNull(result);
            Assert.AreEqual("\"1234568790\",\"\"", result);
        }
        [NUnit.Framework.Test]
        public void JoinIgnreEmpty4()
        {
            string[] testList = new string[] { "1234568790", "", null};

            string result = CFW.Util.StringUtil.JoinIgnoreEmpty(testList, ",", "\"");

            Assert.IsNotNull(result);
            Assert.AreEqual("\"1234568790\",\"\"", result);
        }
        [NUnit.Framework.Test]
        public void BytesToString()
        {
            //ABAɂȂ͂B

            string s = "oCgzϊꗗ\\";
            byte[] bytes = CFW.Util.StringUtil.StringToBytes(s);
            string sout = CFW.Util.StringUtil.BytesToString(bytes);

            Assert.AreEqual("oCgzϊꗗ\\", sout);

        }
        [NUnit.Framework.Test]
        public void FindIndex()
        {
            //               ----+----1----+----2----+----3----+----4----+----5----+-
            string source = "1234568790abcdefghijklmnopqrstuvwxyz";

            int result = CFW.Util.StringUtil.FindIndex(source,"abc");
            Assert.AreEqual(10,result);

            result = CFW.Util.StringUtil.FindIndex(source, "aaa");
            Assert.AreEqual(-1, result);

            result = CFW.Util.StringUtil.FindIndex(source, "123");
            Assert.AreEqual(0, result);

            result = CFW.Util.StringUtil.FindIndex(source, "");
            Assert.AreEqual(41, result);
        }
        [NUnit.Framework.Test]
        public void FindIndex2()
        {
            //               ----+----1----+----2----+----3----+----4----+----5----+-
            string source = null;

            int result = CFW.Util.StringUtil.FindIndex(source, "abc");
            Assert.AreEqual(-1, result);

        }
        [NUnit.Framework.Test]
        public void FindValidString()
        {
            string[] testList = new string[] { "1234568790", null, "" };

            string result = CFW.Util.StringUtil.FindValidString(testList);

            Assert.AreEqual("1234568790", result);
        }
        [NUnit.Framework.Test]
        public void FindValidString2()
        {
            string[] testList = new string[] { null, "1234568790", "" };

            string result = CFW.Util.StringUtil.FindValidString(testList);

            Assert.AreEqual("1234568790", result);
        }
        [NUnit.Framework.Test]
        public void FindValidIndex()
        {
            string[] testList = new string[] { "1234568790", null, "" };

            int result = CFW.Util.StringUtil.FindValidIndex(testList);

            Assert.AreEqual(0, result);
        }
        [NUnit.Framework.Test]
        public void FindValidIndex2()
        {
            string[] testList = new string[] { null, "1234568790", "" };

            int result = CFW.Util.StringUtil.FindValidIndex(testList);

            Assert.AreEqual(1, result);
        }

        [NUnit.Framework.Test]
        public void IsValidPattern()
        {
            string source = "1234567890abcdefghijklmn";
            string pattern = "1234567890";
            bool valid = CFW.Util.StringUtil.IsValidPattern(source,pattern);

            Assert.IsFalse(valid);

            source = "9876543210";
            pattern = "1234567890";
            valid = CFW.Util.StringUtil.IsValidPattern(source, pattern);

            Assert.IsTrue(valid);

            source = "9876543210ABCDEFG";
            pattern = "ABCDEFG";
            valid = CFW.Util.StringUtil.IsValidPattern(source, pattern);

            Assert.IsFalse(valid);

            source = "ABCDEFG";
            pattern = "GFEDCBA";
            valid = CFW.Util.StringUtil.IsValidPattern(source, pattern);

            Assert.IsTrue(valid);

            source = "9876543210-";
            pattern = "-1234568790";
            valid = CFW.Util.StringUtil.IsValidPattern(source, pattern);

            Assert.IsTrue(valid);
        }
        [NUnit.Framework.Test]
        public void IsDigit()
        {
            string source = "1234567890abcdefghijklmn";
            bool valid = CFW.Util.StringUtil.IsDigit(source);

            Assert.IsFalse(valid);

            source = "9876543210";
            valid = CFW.Util.StringUtil.IsDigit(source);
            Assert.IsTrue(valid);


            source = "-9876543210";
            valid = CFW.Util.StringUtil.IsDigit(source);

            Assert.IsFalse(valid);

        }
        [NUnit.Framework.Test]
        public void IsNumeric()
        {
            string source = "1234567890abcdefghijklmn";
            bool valid = CFW.Util.StringUtil.IsNumeric(source);

            Assert.IsFalse(valid);

            source = "9876543210";
            valid = CFW.Util.StringUtil.IsNumeric(source);
            Assert.IsTrue(valid);


            source = "-9876543210";
            valid = CFW.Util.StringUtil.IsNumeric(source);

            Assert.IsTrue(valid);

            source = "-9876543.210";
            valid = CFW.Util.StringUtil.IsNumeric(source);

            Assert.IsTrue(valid);

        }
        [NUnit.Framework.Test]
        public void IsCaptial()
        {
            string source = "1234567890abcdefghijklmn";
            bool valid = CFW.Util.StringUtil.IsCapitalAlpha(source);

            Assert.IsFalse(valid);

            source = "9876543210";
            valid = CFW.Util.StringUtil.IsCapitalAlpha(source);
            Assert.IsFalse(valid);


            source = "abcdefghij";
            valid = CFW.Util.StringUtil.IsCapitalAlpha(source);

            Assert.IsFalse(valid);

            source = "BGFDDSAQWERT";
            valid = CFW.Util.StringUtil.IsCapitalAlpha(source);

            Assert.IsTrue(valid);

        }
        [NUnit.Framework.Test]
        public void IsCaptialDigit()
        {
            string source = "1234567890abcdefghijklmn";
            bool valid = CFW.Util.StringUtil.IsCapitalAlphaDigit(source);

            Assert.IsFalse(valid);

            source = "9876543210";
            valid = CFW.Util.StringUtil.IsCapitalAlphaDigit(source);
            Assert.IsTrue(valid);


            source = "abcdefghij";
            valid = CFW.Util.StringUtil.IsCapitalAlphaDigit(source);

            Assert.IsFalse(valid);

            source = "BGFDDSAQWERT";
            valid = CFW.Util.StringUtil.IsCapitalAlphaDigit(source);

            Assert.IsTrue(valid);

            source = "BG444FDDSA1QW3ERT5";
            valid = CFW.Util.StringUtil.IsCapitalAlphaDigit(source);

            Assert.IsTrue(valid);
        }
        [NUnit.Framework.Test]
        public void NormalizeNumeric()
        {
            string source = "1234567890abcdefghijklmn";
            string result = CFW.Util.StringUtil.NormalizeNumeric(source);

            Assert.AreEqual("1234567890",result);

            source = "1234567890.9876543210";
            result = CFW.Util.StringUtil.NormalizeNumeric(source);
            Assert.AreEqual("1234567890.9876543210", result);

            source = "abcd1234567890.9876543210";
            result = CFW.Util.StringUtil.NormalizeNumeric(source);
            Assert.AreEqual("1234567890.9876543210", result);
        }
        [NUnit.Framework.Test]
        public void NormalizeAlphaNumeric()
        {
            string source = "1234567890abcdefghijklmn";
            string result = CFW.Util.StringUtil.NormalizeAlphaNumeric(source);

            Assert.AreEqual("1234567890abcdefghijklmn", result);

            source = "1234567890.9876543210";
            result = CFW.Util.StringUtil.NormalizeAlphaNumeric(source);
            Assert.AreEqual("1234567890.9876543210", result);

            source = "abcd1234567890.9876543210";
            result = CFW.Util.StringUtil.NormalizeAlphaNumeric(source);
            Assert.AreEqual("abcd1234567890.9876543210", result);
        }
        [NUnit.Framework.Test]
        public void Left()
        {
            //               ----+----1----+----2----+
            string source = "1234567890abcdefghijklmn";
            string result = CFW.Util.StringUtil.Left(source,10);

            Assert.AreEqual("1234567890", result);

            source = "9876543210";
            result = CFW.Util.StringUtil.Left(source, 10);
            Assert.AreEqual("98765", result);

            source = "abcd1";
            result = CFW.Util.StringUtil.Left(source,10);
            Assert.AreEqual("abcd1", result);
        }
        [NUnit.Framework.Test]
        public void stripAffix()
        {
            string p = "[@1234568790]";
            string stripped = CFW.Util.StringUtil.stripAffix(p, "[@","]");
            Assert.AreEqual("1234568790", stripped);

            p = "[[--1234568790--]]hoge";
            stripped = CFW.Util.StringUtil.stripAffix(p, "[[--", "--]]");
            Assert.AreEqual("1234568790", stripped);

            p = "[[--1234568790";
            stripped = CFW.Util.StringUtil.stripAffix(p, "[[--", "--]]");
            Assert.AreEqual("1234568790", stripped);

            p = "[[--1234568790--]]hoge";
            stripped = CFW.Util.StringUtil.stripAffix(p, "[[--", "");
            Assert.AreEqual("1234568790--]]hoge", stripped);
        }
        [NUnit.Framework.Test]
        public void Mid()
        {
            string sample = "1234567890";
            string result = StringUtil.Mid(sample, 5);
            Assert.AreEqual("67890", result);
        }
        [NUnit.Framework.Test]
        public void Mid2()
        {
            string sample = "1234567890";
            string result = StringUtil.Mid(sample, 5,10);
            Assert.AreEqual("67890", result);
        }
        [NUnit.Framework.Test]
        public void Mid3()
        {
            string sample = "1234567890";
            string result = StringUtil.Mid(sample, 10);
            Assert.AreEqual("", result);
        }
        [NUnit.Framework.Test]
        public void Mid4()
        {
            string sample = "1234567890";
            string result = StringUtil.Mid(sample, 100,5);
            Assert.AreEqual("", result);
        }
        [NUnit.Framework.Test]
        public void Mid5()
        {
            string sample = "1234567890";
            string result = StringUtil.Mid(sample, 2, 5);
            Assert.AreEqual("34567", result);
        }
    }
}
