using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using NUnit.Framework;
using CFW.Util;

namespace CFW.Common.Test
{
    [TestFixture]
    public class XmlUtilTest
    {
        string testXmlString = @"<?xml version=""1.0"" encoding=""utf-16""?> 
<root> 
    <first>test&lt;test&gt;</first> 
    <second>second&lt;test&gt;</second> 
    <intValue>102</intValue> 
    <decimalValue>102.45</decimalValue> 
    <yesOrNo>Y</yesOrNo> 
    <no>N</no> 
    <subElement><subElementChild>q̗vf</subElementChild><subElementChild>q̗vfQ</subElementChild></subElement>
    <subElement2>
        <subElementChild>q̗vf</subElementChild>
        <subElementChild>q̗vfQ</subElementChild>
    </subElement2>
    
    <cdataSection><![CDATA[CDATAłB
FX܂B
ǂȊł傤B
]]></cdataSection> 
</root>";

        string errorXmlString = @"<?xml version=""1.0"" encoding=""utf-16""?> 
<root> 
    <first>test&lt;test&gt;</first> 
    <second>second&lt;test&gt;</second> 
    <intValue>102
    <decimalValue>102.45</decimalValue> 
    <yesOrNo>Y</yesOrNo> 
    <cdataSection><![CDATA[CDATAłB
FX܂B
ǂȊł傤B
]]></cdataSection> 
</root>";

        [NUnit.Framework.Test]
        public void LoadXml()
        {
            XmlDocument doc = XmlUtil.LoadXml(testXmlString);
            Assert.IsNotNull(doc);

            doc = XmlUtil.LoadXml(errorXmlString);
            Assert.IsNull(doc);

        }

        [NUnit.Framework.Test]
        public void GetText()
        {
            XmlDocument doc = XmlUtil.LoadXml(testXmlString);
            Assert.IsNotNull(doc);

            XmlNode node = doc.SelectSingleNode("/root/first");
            Assert.IsNotNull(node);

            string text = XmlUtil.GetText(node);
            Assert.AreEqual("test<test>",text);

            node = doc.SelectSingleNode("/root/cdataSection");
            Assert.IsNotNull(node);

            text = XmlUtil.GetText(node);
            Assert.AreEqual(@"CDATAłB
FX܂B
ǂȊł傤B
", text);

            //2̎qvfGgm[h
            node = doc.SelectSingleNode("/root/subElement");
            Assert.IsNotNull(node);
            text = XmlUtil.GetText(node);
            Assert.AreEqual(@"q̗vfq̗vfQ", text);

            //2̎qvfGgm[h(`̂߂ɉs܂܂Ă)
            node = doc.SelectSingleNode("/root/subElement2");
            Assert.IsNotNull(node);
            text = XmlUtil.GetText(node);
            Assert.AreEqual(@"q̗vfq̗vfQ", text);

        }
        [NUnit.Framework.Test]
        public void GetText2()
        {
            XmlDocument doc = XmlUtil.LoadXml(testXmlString);
            Assert.IsNotNull(doc);

            XmlNode node = doc.SelectSingleNode("/root");
            Assert.IsNotNull(node);

            string text = XmlUtil.GetText(node,"first");
            Assert.AreEqual("test<test>", text);

        }
        [NUnit.Framework.Test]
        public void GetTextOptional()
        {
            XmlDocument doc = XmlUtil.LoadXml(testXmlString);
            Assert.IsNotNull(doc);

            XmlNode node = doc.SelectSingleNode("/root");
            Assert.IsNotNull(node);

            string text = XmlUtil.GetTextOptional(node, "first");
            Assert.AreEqual("test<test>", text);

            text = XmlUtil.GetTextOptionalDefault(node, "noTag","DEFAULT VALUE");
            Assert.AreEqual("DEFAULT VALUE", text);

        }
        [NUnit.Framework.Test]
        public void GetInteger()
        {
            XmlDocument doc = XmlUtil.LoadXml(testXmlString);
            Assert.IsNotNull(doc);

            XmlNode node = doc.SelectSingleNode("/root/intValue");
            Assert.IsNotNull(node);

            int val= XmlUtil.GetInteger(node);
            Assert.AreEqual(102, val);

            node = doc.SelectSingleNode("/root");
            Assert.IsNotNull(node);

            val = XmlUtil.GetInteger(node,"intValue");
            Assert.AreEqual(102, val);


        }
        [NUnit.Framework.Test]
        public void GetIntegerOptional()
        {
            XmlDocument doc = XmlUtil.LoadXml(testXmlString);
            Assert.IsNotNull(doc);

            XmlNode node = doc.SelectSingleNode("/root/intValue");

            int val = XmlUtil.GetIntegerOptional(node);
            Assert.AreEqual(102, val);

            node = doc.SelectSingleNode("/root/noTag");

            val = XmlUtil.GetIntegerOptional(node);
            Assert.AreEqual(0, val);

            node = doc.SelectSingleNode("/root");
            Assert.IsNotNull(node);

            val = XmlUtil.GetIntegerOptionalDefault(node, "noTag",123);
            Assert.AreEqual(123, val);


        }
        [NUnit.Framework.Test]
        public void GetDecimalOptional()
        {
            XmlDocument doc = XmlUtil.LoadXml(testXmlString);
            Assert.IsNotNull(doc);

            XmlNode node = doc.SelectSingleNode("/root/decimalValue");

            decimal val = XmlUtil.GetDecimalOptional(node);
            Assert.AreEqual(102.45M, val);

            node = doc.SelectSingleNode("/root/noTag");

            val = XmlUtil.GetDecimalOptional(node);
            Assert.AreEqual(0, val);

            node = doc.SelectSingleNode("/root");
            Assert.IsNotNull(node);

            val = XmlUtil.GetDecimalOptionalDefault(node, "noTag", 123);
            Assert.AreEqual(123, val);

        }
        [NUnit.Framework.Test]
        public void GetBoolOptional()
        {
            XmlDocument doc = XmlUtil.LoadXml(testXmlString);
            Assert.IsNotNull(doc);

            XmlNode node = doc.SelectSingleNode("/root/yesOrNo");

            bool val = XmlUtil.GetBoolOptional(node);
            Assert.AreEqual(true, val);

            node = doc.SelectSingleNode("/root/no");
            val = XmlUtil.GetBoolOptional(node);
            Assert.AreEqual(false, val);

            node = doc.SelectSingleNode("/root/noTag");

            val = XmlUtil.GetBoolOptional(node);
            Assert.AreEqual(false, val);


        }

    }
}
