/*
 * Copyright (C) 2009 - 2010 Funambol, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by
 * the Free Software Foundation with the addition of the following permission
 * added to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED
 * WORK IN WHICH THE COPYRIGHT IS OWNED BY FUNAMBOL, FUNAMBOL DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see http://www.gnu.org/licenses or write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA.
 *
 * You can contact Funambol, Inc. headquarters at 643 Bair Island Road, Suite
 * 305, Redwood City, CA 94063, USA, or at email address info@funambol.com.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Powered by Funambol" logo. If the display of the logo is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Powered by Funambol".
 */
/* $Id$ */

#ifndef IDATASTORAGE_H_INCLUDED
#define IDATASTORAGE_H_INCLUDED

#include <cstdio>
#include <vector>
#include "commontypes.h"
#include "platform.h"

namespace NS_DM_Client
{

class Buffer;

namespace NS_DataStorage
{
/// platform independent interface for handle stream data
class IStreamHandler
{
public:
    virtual ~IStreamHandler() { }
    IStreamHandler() { }
    /// write portion of data to stream
    virtual bool Write(const void* buffer, size_t size) = 0;
    virtual bool Write(const Buffer& buffer) = 0;

    /// indicate that writing is finished and we can read from stream
    virtual bool FinishWriting() = 0;
    /// release stream resources
    virtual void Release() = 0;
private:
    IStreamHandler(const IStreamHandler&);
    IStreamHandler& operator=(const IStreamHandler&);
};

/// platform independent interface for store program data
class IDataStorage
{
public:
    virtual ~IDataStorage() { }
    IDataStorage() { }

    virtual bool Init(const String& base_path = "") = 0;
    virtual void Release() = 0;

    /// functions for work with stream data
    /// return stream handler which uniquely identify stream
    /// NOTE: need call Release() for release object resources
    virtual IStreamHandler* CreateStream() = 0;

    /// functions for daemon's private data
    /// key - identify unique private data
    virtual bool SavePrivateData(const String& key, const void* buffer, size_t size, bool profileSpecific = true) = 0;
    virtual bool SavePrivateData(const String& key, const Buffer& buffer, bool profileSpecific = true) = 0;
    virtual bool LoadPrivateData(const String& key, Buffer& buffer, bool profileSpecific = true) = 0;
    virtual bool RemovePrivateData(const String& key) = 0;

    // get child list
    virtual bool GetChildList(const String& key, StringArray& list, bool profileSpecific = true) = 0;

    // check peresistence
    virtual bool Exist(const String& key, bool profileSpecific = true) = 0;
    virtual const char* GetBasePath() = 0;

    virtual bool GetPrivateDataSize(const String& key, size_t& size, bool profileSpecific = true) = 0;

private:
    IDataStorage(const IDataStorage& data);
    IDataStorage& operator=(const IDataStorage& data);
};

class IConfigurationStorage
{
public:
    virtual ~IConfigurationStorage() { }
    IConfigurationStorage() { }

    virtual bool Init() = 0;
    virtual void Release() = 0;

    /// functions for daemon's configuration
    virtual bool SaveConfiguration(const void* buffer, size_t size) = 0;
    virtual bool SaveConfiguration(const Buffer& buffer) = 0;
    virtual bool LoadConfiguration(Buffer& buffer) = 0;

private:
    IConfigurationStorage(const IConfigurationStorage& data);
    IConfigurationStorage& operator=(const IConfigurationStorage& data);
};

/// factory method for IDataStorage instance creation
IDataStorage* CreateDataStorage(const String& profile, const String& base_path = "");

/// factory method for IConfigurationStorage instance creation
IConfigurationStorage* CreateConfigurationStorage();

const char* GetBasePath();
}

}
#endif // IDATASTORAGE_H_INCLUDED
