/*
 * Copyright (C) 2009 - 2010 Funambol, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by
 * the Free Software Foundation with the addition of the following permission
 * added to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED
 * WORK IN WHICH THE COPYRIGHT IS OWNED BY FUNAMBOL, FUNAMBOL DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see http://www.gnu.org/licenses or write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA.
 *
 * You can contact Funambol, Inc. headquarters at 643 Bair Island Road, Suite
 * 305, Redwood City, CA 94063, USA, or at email address info@funambol.com.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Powered by Funambol" logo. If the display of the logo is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Powered by Funambol".
 */
/* $Id$ */

#pragma once

#include <cwchar>
#include <boost/thread.hpp>
#include <cstdarg>
#include <string>

#include "platform.h"
#include "commontypes.h"

namespace boost
{
	template <typename T>
	class thread_specific_ptr;
}
namespace NS_Logging
{
	class IOutStream;

	// NS_Logging category, allows to make some Log-messages more important then others
	enum Category
	{
		e_criticalError = 0,
		e_error = 1,
		e_warning,
		e_info,
		e_debug,
		e_dump
	};

	class Logger
	{
		friend class LoggerFactory;
		friend struct FunctionNameLogHelper;

	public:
		Logger(const char* name);
		~Logger();

		bool IsEnabled() { return m_enabled; };

		void SetMinimalCategory(Category cat);
		void SetMaxMessageSize(size_t size);

		// logs error messages
		void Error(const char* format, ...);
		// logs regular info messages
		void Info(const char* format, ...);
		void Warning(const char* format, ...);
		// logs debug messages, this category is recommended for messages that is required only for deep diagnosis and usually is turned off by NS_Logging::SetMinimalCategory() */
		void Debug(const char* format, ...);
		// log with info category
		void Log(const char* format, ...);

	private:
		void enable();
		void log(Category category, char const* functionName, char const* format, va_list va);

	private:
		long volatile m_minimalCategory; // long instead of Category for interlocked exchange
		long volatile m_bufSize;
		boost::thread_specific_ptr<std::string>* m_buffer; // m_buffer's pointer is required for mac only
		IOutStream* m_ostream;
		bool m_enabled;
		std::string m_name;
	};

	// get logger by name. If logger doesn't exist - create it
	Logger& GetLogger(const char* name);
	void ReleaseLoggers();

 	// trick to be able to implement LOG macro that logs function name and accepts "ellipsis" parameters
	struct FunctionNameLogHelper
	{
		FunctionNameLogHelper(const char* funcName, Category cat);
		void Log(Logger& logger, char const* format, ...);

	private:
		char const* m_funcName;
		Category m_category;
	};
}

// following macros allows automatic logging of function's name where logging is occurred */
#define LOG_ NS_Logging::FunctionNameLogHelper(__FUNCTION__, NS_Logging::e_info).Log
#define LOG_ERROR_ NS_Logging::FunctionNameLogHelper(__FUNCTION__, NS_Logging::e_error).Log
#define LOG_WARNING_ NS_Logging::FunctionNameLogHelper(__FUNCTION__, NS_Logging::e_warning).Log
#define LOG_DEBUG_ NS_Logging::FunctionNameLogHelper(__FUNCTION__, NS_Logging::e_debug).Log
#define LOG_INFO_ NS_Logging::FunctionNameLogHelper(__FUNCTION__, NS_Logging::e_info).Log
#define LOG_DUMP_ NS_Logging::FunctionNameLogHelper(__FUNCTION__, NS_Logging::e_dump).Log
#define LOG_CRITICAL_ NS_Logging::FunctionNameLogHelper(__FUNCTION__, NS_Logging::e_criticalError).Log

