/*
 * Copyright (C) 2009 - 2010 Funambol, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by
 * the Free Software Foundation with the addition of the following permission
 * added to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED
 * WORK IN WHICH THE COPYRIGHT IS OWNED BY FUNAMBOL, FUNAMBOL DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see http://www.gnu.org/licenses or write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA.
 *
 * You can contact Funambol, Inc. headquarters at 643 Bair Island Road, Suite
 * 305, Redwood City, CA 94063, USA, or at email address info@funambol.com.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Powered by Funambol" logo. If the display of the logo is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Powered by Funambol".
 */

#include "DaemonDM/ProfileManager.h"
#include "Logger/Logger.h"
#include "commontypes.h"
#include "IFIFOWrapper.h"
#include "MessengerDefs.h"

const char* const c_ProfileManagerLog = "ProfileManagerAPI";

namespace NS_DM_Client
{
//------------------------------------------------------------------------------------------------------
ProfileManager::ProfileManager()
    : m_fifo_request(0), m_fifo_response(0)
{
}
//------------------------------------------------------------------------------------------------------
void ProfileManager::release()
{
    if (m_fifo_request)
    {
        m_fifo_request->Release();
    }
    if (m_fifo_response)
    {
        m_fifo_response->Release();
    }
}
//------------------------------------------------------------------------------------------------------
ProfileManager::~ProfileManager()
{
    release();
}
//------------------------------------------------------------------------------------------------------
bool ProfileManager::Init()
{
    if (m_fifo_request && m_fifo_response)
    {
        return true;
    }

    m_fifo_request = 0;
    m_fifo_response = 0;

    m_fifo_request = CreateFIFOWrapper(false);
    if (!m_fifo_request)
    {
        LOG_ERROR_(NS_Logging::GetLogger(c_ProfileManagerLog), "can't create fifo request wrapper");
        return false;
    }

    if (!m_fifo_request->Open(c_IPCRequestFifoName, false, false, true))
    {
        LOG_ERROR_(NS_Logging::GetLogger(c_ProfileManagerLog), "can't open request fifo");
        m_fifo_request->Release();
        m_fifo_request = 0;
        return false;
    }

    m_fifo_response = CreateFIFOWrapper(false);
    if (!m_fifo_response)
    {
        LOG_ERROR_(NS_Logging::GetLogger(c_ProfileManagerLog), "can't create fifo responce wrapper");
        m_fifo_request->Release();
        m_fifo_request = 0;
        m_fifo_response->Release();
        m_fifo_response = 0;
        return false;
    }

    if (!m_fifo_response->Open(c_IPCResponseFifoName, true, false, true))
    {
        LOG_ERROR_(NS_Logging::GetLogger(c_ProfileManagerLog), "can't open response fifo");
        m_fifo_request->Release();
        m_fifo_request = 0;
        m_fifo_response->Release();
        m_fifo_response = 0;
        return false;
    }

    return true;
}
//------------------------------------------------------------------------------------------------------
bool ProfileManager::Release()
{
    if (!(m_fifo_request && m_fifo_response))
    {
        return true;
    }

    release();
    return true;
}
//------------------------------------------------------------------------------------------------------
StatusCode ProfileManager::FetchFirstProfileInfo(ProfileInfo& profile)
{
    StatusCode res = e_Failed;

    m_profiles.clear();
    m_profiles_index = 0;
    UIExchangeHead request_head;
    request_head.m_type = e_GetProfiles;
    request_head.m_payload_size = 0;
    Buffer buffer(sizeof(request_head));
    memcpy(buffer.GetPointer(), &request_head, sizeof(request_head));

    if (m_fifo_request->Write(buffer) == e_Ok)
    {
        UIResponseExchangeHead response_head;
        buffer.Allocate(sizeof(response_head));
        if (m_fifo_response->Read(buffer) == e_Ok)
        {
            memcpy(&response_head, buffer.GetPointer(), buffer.Size());
            if (response_head.m_perform_code == e_Ok)
            {
                buffer.Allocate(response_head.m_payload_size);
                if (m_fifo_response->Read(buffer, true) == e_Ok)
                {
                    char* ptr = (char*)buffer.GetPointer();
                    size_t data_size;
                    ProfileInfo current_profile;

                    size_t index = 0;
                    while (index < response_head.m_payload_size)
                    {
                        // fetching profile name
                        data_size = *((size_t*)ptr);
                        current_profile.m_Name = (char*)malloc(data_size + 1);
                        index += sizeof(data_size);
                        ptr += sizeof(data_size);
                        memcpy(current_profile.m_Name, ptr, data_size);
                        current_profile.m_Name[data_size] = '\0';
                        index += data_size;
                        ptr += data_size;
                        // fetching description
                        data_size = *((size_t*)ptr);
                        current_profile.m_Description = (char*)malloc(data_size + 1);
                        index += sizeof(data_size);
                        ptr += sizeof(data_size);
                        memcpy(current_profile.m_Description, ptr, data_size);
                        current_profile.m_Description[data_size] = '\0';
                        index += data_size;
                        ptr += data_size;
                        // fetching is enabled
                        current_profile.m_IsEnabled = *((bool*)ptr);
                        index += sizeof(bool);
                        ptr += sizeof(bool);
                        // add to vector
                        m_profiles.push_back(current_profile);
                    }
                    if (!m_profiles.empty())
                    {
                        m_profiles_index = 1;
                        profile = m_profiles[0];
                        res = e_Ok;
                    }
                    else
                    {
                        res = e_NotFound;
                    }
                }
                else
                {
                    LOG_ERROR_(NS_Logging::GetLogger(c_ProfileManagerLog), "read response payload from response fifo failed");
                }
            }
            else
            {
                LOG_ERROR_(NS_Logging::GetLogger(c_ProfileManagerLog), "IPC daemon engine failed with request");
            }
        }
        else
        {
            LOG_ERROR_(NS_Logging::GetLogger(c_ProfileManagerLog), "read response head from response fifo failed");
        }
    }
    else
    {
        LOG_ERROR_(NS_Logging::GetLogger(c_ProfileManagerLog), "write to request fifo failed");
    }

    return res;
}
//------------------------------------------------------------------------------------------------------
StatusCode ProfileManager::FetchNextProfileInfo(ProfileInfo& profile)
{
    StatusCode res = e_Failed;
    if (m_profiles_index >= m_profiles.size())
    {
        res = e_NotFound;
    }
    else
    {
        profile = m_profiles[m_profiles_index++];
        res = e_Ok;
    }
    return res;
}
//------------------------------------------------------------------------------------------------------
void ProfileManager::ReleaseProfileInfo(ProfileInfo& profile)
{
    free(profile.m_Name);
    free(profile.m_Description);
}
//------------------------------------------------------------------------------------------------------
StatusCode ProfileManager::EnableProfile(const String& Profile_name, bool enable)
{
    StatusCode res = e_Failed;

    if (Profile_name.empty())
    {
        return res;
    }

    UIExchangeHead request_head;
    request_head.m_type = e_SetProfileState;
    request_head.m_payload_size = sizeof(size_t) + Profile_name.length() + sizeof(enable);
    Buffer buffer(sizeof(request_head) + sizeof(size_t) + Profile_name.length() + sizeof(enable));
    memcpy(buffer.GetPointer(), &request_head, sizeof(request_head));
    size_t name_length = Profile_name.length();
    memcpy((char*)buffer.GetPointer() + sizeof(request_head), &name_length, sizeof(name_length));
    memcpy((char*)buffer.GetPointer() + sizeof(request_head) + sizeof(name_length), Profile_name.c_str(), name_length);
    memcpy((char*)buffer.GetPointer() + sizeof(request_head) + sizeof(name_length) + name_length, &enable, sizeof(enable));

    if (m_fifo_request->Write(buffer) == e_Ok)
    {
        UIResponseExchangeHead response_head;
        buffer.Allocate(sizeof(response_head));
        if (m_fifo_response->Read(buffer, true) == e_Ok)
        {
            memcpy(&response_head, buffer.GetPointer(), buffer.Size());
            if (response_head.m_perform_code == e_Ok)
            {
                res = e_Ok;
            }
            else
            {
                LOG_ERROR_(NS_Logging::GetLogger(c_ProfileManagerLog), "IPC daemon engine failed with request");
            }
        }
        else
        {
            LOG_ERROR_(NS_Logging::GetLogger(c_ProfileManagerLog), "read response head from response fifo failed");
        }
    }
    else
    {
        LOG_ERROR_(NS_Logging::GetLogger(c_ProfileManagerLog), "write to request fifo failed");
    }

    return res;
}
//------------------------------------------------------------------------------------------------------

ProfileManager* GetProfileManager()
{
	return new ProfileManager;
}

}

