/*
 * Copyright (C) 2009 - 2010 Funambol, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by
 * the Free Software Foundation with the addition of the following permission
 * added to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED
 * WORK IN WHICH THE COPYRIGHT IS OWNED BY FUNAMBOL, FUNAMBOL DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see http://www.gnu.org/licenses or write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA.
 *
 * You can contact Funambol, Inc. headquarters at 643 Bair Island Road, Suite
 * 305, Redwood City, CA 94063, USA, or at email address info@funambol.com.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Powered by Funambol" logo. If the display of the logo is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Powered by Funambol".
 */

/* $Id$ */

#include "posix/Timer.h"
#include "Logger/LoggerMacroses.h"
//#include <sys/types.h>
#include <time.h>


namespace NS_DM_Client
{
	namespace NS_Common
	{
		void TimerAction(union sigval sigval);
	}
}

using namespace NS_DM_Client;
using namespace NS_DM_Client::NS_Common;

const char* const c_LogName = "Timer";

Timer::Timer() :
	m_isRunning(false), m_ticksCount(0), m_ticksToCount(0), m_timerInterval(0), m_callback(NULL), m_timerid(0)
{
}


Timer::~Timer()
{
	Stop();
}


bool Timer::IsRunning()
{
	return m_isRunning;
}


void Timer::SetCallback(Callback *c)
{
	m_callback = c;
}


void Timer::SetInterval(uint t)
{
	m_timerInterval = t;
}


void Timer::SetTicksCount(int count)
{
	m_ticksCount = count;
}


void Timer::Start()
{
	if (m_isRunning) return;

	if (m_timerInterval)
	{
		if (m_ticksCount == -1 || m_ticksCount > 0)
			startTimer();
	}
}


void Timer::Stop()
{
	if (!m_isRunning) return;

	timer_delete(m_timerid);
	m_timerid = 0;
	m_isRunning = false;
}


void Timer::notifyCallback()
{
	if (m_callback) m_callback->Call();
}


void Timer::startTimer()
{
	m_ticksToCount = m_ticksCount;
	
	// tick once before starting the timer
	if (m_ticksToCount)
		tick();
	
	if (!m_ticksToCount) return;
	
	struct sigevent sigev;
	struct itimerspec itval;
	struct itimerspec oitval;

	memset(&sigev, 0, sizeof(sigevent));

	// Create the POSIX timer and pass pointer to self
	sigev.sigev_notify = SIGEV_THREAD;
	sigev.sigev_value.sival_ptr = this;
	sigev.sigev_notify_function = TimerAction;

	// CLOCK_REALTIME - counts systemwide time
	if (timer_create(CLOCK_REALTIME, &sigev, &m_timerid) == 0)
	{
		GDLDEBUG("created timer: %x", m_timerid);
		itval.it_value.tv_sec = m_timerInterval;
		itval.it_value.tv_nsec = 0;

		itval.it_interval.tv_sec = itval.it_value.tv_sec;
		itval.it_interval.tv_nsec = itval.it_value.tv_nsec;

		if (timer_settime(m_timerid, 0, &itval, &oitval) == 0)
		{
			m_isRunning = true;
		}
		else
		{
			GDLDEBUG("time_settime error!");
		}
	}
	else
	{
		GDLDEBUG("timer_create failed !");
	}
}


void Timer::tick()
{
	if (m_ticksToCount == 0)
	{
		Stop();
	}
	else
	{
		if (m_ticksToCount > 0)
			m_ticksToCount--;
		notifyCallback();				
	}
}


void NS_DM_Client::NS_Common::TimerAction(union sigval val)
{
	Timer *pTimer = (Timer*) val.sival_ptr;
	pTimer->tick();
	pthread_exit(NULL);
}
