/*
 * Copyright 2010 Funambol, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* $Id$ */

#pragma once

#include <http/TransportAgentFactory.h>
#include <syncml/core/Status.h>

#include "serverexchange/ConnectionInfo.h"
#include "serverexchange/commands/AlertCodes.h"
#include "serverexchange/CommandStorage.h"
#include "serverexchange/ICommandsSink.h"
#include "serverexchange/wrappers/SAlertCommand.h"
#include "serverexchange/wrappers/SStatusCommand.h"


namespace NS_DM_Client
{
	class ProfileComponentsHolder;
	
	namespace NS_Communication
	{
		/// Class ISession defines interface of the connection session abstraction.
		class ISession
		{
		public:
			/// ConnectionState stores settings and states of the connection
			typedef struct tagState {
				bool AddInitialAlert;
				bool ClientAuthenticated;
				bool ServerAuthenticated;
				NS_SyncMLCommand::AlertCode InitialAlertCode;
				NS_SyncMLCommand::SAlertCommandPtr  InitialAlert;
				
				tagState() {
					AddInitialAlert     = true;
					ClientAuthenticated = false;
					ServerAuthenticated = false;
					InitialAlertCode    = NS_SyncMLCommand::AC_CLIENT_INITIATED_MGMT;
				}
			} ConnectionState;

		public:
			virtual      ~ISession() {};

			/// Check whether session support authentication with nonce exchange
			virtual bool AcceptsNonces() = 0;
			
			/// Check whether credentials were already sent to the recipient
			virtual bool CredentialsSent() = 0;

			/// Start connection and send commands
			/// Connection will be started even if commands list is empty
			/// Specific implementation should insert Alert,Replace[DevInfo] to initiate connection
			virtual void Connect(NS_SyncMLCommand::CommandStorage & commands) = 0;
			
			/// Create a session-specific Status on SyncHdr
			virtual NS_SyncMLCommand::SStatusCommand * 
			             CreateSyncHdrStatus(int code, const char * responseURI) = 0;
			
			/// Return name of the session authentication type
			/// Returned value should not be deleted
			virtual const char * GetTypeName() = 0;
			
			/// Check whether caller passes authentication
			virtual bool IsAuthenticationPassed(Funambol::Status &) = 0;
			
			/// Check whether response is valid according to session requirements 
			virtual	bool IsResponseValid(const char *, uint resplength) = 0;
			
			/// Check whether passed Cred contains valid auth data of the server
			virtual bool IsServerCredValid(Funambol::Cred *) = 0;
			
			/// Check whether session is in state of sending large object
			virtual bool IsSendingLO() = 0;

			/// Read authentication info from the Chal
			virtual bool ReadChal(Funambol::Chal &) = 0;

			/// Send message to server
			virtual void SendMessage(FStringBuffer &) = 0;
			
			virtual void SetCommandsSink(NS_SyncMLCommand::ICommandsSink &) = 0;
			
			virtual void SetPCH(ProfileComponentsHolder &) = 0;
			
			// configure session
			const ConnectionState& State()       { return m_state; };
			void SetAddInitialAlert(bool b)     { m_state.AddInitialAlert     = b; };
			void SetClientAuthenticated(bool b) { m_state.ClientAuthenticated = b; };
			void SetServerAuthenticated(bool b) { m_state.ServerAuthenticated = b; };
			void SetStartAlertCode(const NS_SyncMLCommand::AlertCode alertCode) { m_state.InitialAlertCode = alertCode; }
			void SetStartAlert(const NS_SyncMLCommand::SAlertCommandPtr &alert) { m_state.InitialAlert = alert; }
			
		protected:
			ConnectionState m_state;
		};
	}
}

