// -*- coding: sjis; tab-width: 4; -*-

#include <iostream>
#include <fstream>

#include <sys/stat.h>

typedef char  int8;
typedef short int16;
typedef int   int32;
typedef long  int64;

typedef unsigned char  uint8;
typedef unsigned short uint16;
typedef unsigned int   uint32;
typedef unsigned long  uint64;

void show_size_of_type()
{
	std::cout << "sizeof( char  ): " << sizeof( char  ) << std::endl;
	std::cout << "sizeof( short ): " << sizeof( short ) << std::endl;
	std::cout << "sizeof( int   ): " << sizeof( int   ) << std::endl;
	std::cout << "sizeof( long  ): " << sizeof( long  ) << std::endl;
}

struct pcf_table
{
	uint32 type;
    uint32 format;
    uint32 size;
    uint32 offset;
};

struct font_metrics /* xCharInfo */
{
	int16 left_side_bearing;
	int16 right_side_bearing;
	int16 character_width;
	int16 ascent;
	int16 descent;
	uint16 attributes;
};

struct character_info /* CharInfoRec */
{
	font_metrics metrics; /* info preformatted for Queries */
	char *bits;           /* pointer to glyph image */
};

uint8* file_memory;
int position;

pcf_table* tables;
int count;

character_info* metrics;
int n_metrics;

uint32* offsets;
int n_bitmaps;
uint8* bitmaps;

uint32* glyph_offsets;
int glyph_count;
char* string;

void load_to_memory()
{
	const char* filename = "mplus_j12r.pcf";
	//const char* filename = "mplus_j10r.pcf";

	struct stat s;
	stat( filename, &s );

	::file_memory = new uint8[ s.st_size ];

	std::ifstream in( filename, std::ios_base::binary );
	if ( in )
		in.read( (char*)file_memory, s.st_size );
	in.close();
}

void free_memory()
{
	delete[] ::string;
	delete[] ::glyph_offsets;
	delete[] ::bitmaps;
	delete[] ::offsets;
	delete[] ::metrics;
	delete[] ::tables;
	delete[] ::file_memory;
}

enum byte_order { LE, BE };

uint32 get_uint32_le( uint8* p )
{
	return ( uint32( p[0] ) | uint32( p[1] ) << 8 | uint32( p[2] ) << 16 | uint32( p[3] ) << 24 );
}

uint32 get_uint32_be( uint8* p )
{
	return ( uint32( p[0] ) << 24 | uint32( p[1] ) << 16 | uint32( p[2] ) << 8 | uint32( p[3] ) );
}

uint32 get_uint32( uint8* p, byte_order b )
{
	if ( b == LE )
		return ::get_uint32_le( p );
	else if ( b == BE )
		return ::get_uint32_be( p );
}

uint32 next_uint32( byte_order b )
{
	uint32 c = ::get_uint32( ::file_memory + ::position, b );
	::position += 4;
	return c;
}

uint16 get_uint16_le( uint8* p )
{
	return ( uint16( p[0] ) | uint16( p[1] ) << 8 );
}

uint16 get_uint16_be( uint8* p )
{
	return ( uint16( p[0] ) << 8 | uint16( p[1] ) );
}

uint16 get_uint16( uint8* p, byte_order b )
{
	if ( b == LE )
		return ::get_uint16_le( p );
	else if ( b == BE )
		return ::get_uint16_be( p );
}

uint16 next_uint16( byte_order b )
{
	uint16 c = ::get_uint16( ::file_memory + ::position, b );
	::position += 2;
	return c;
}

uint8 get_uint8( uint8* p )
{
	return *p;
}

uint8 next_uint8()
{
	uint8 c = ::get_uint8( ::file_memory + ::position );
	::position += 1;
	return c;
}

enum type_t
{
	PCF_PROPERTIES       = 1 << 0,
	PCF_ACCELERATORS     = 1 << 1,
	PCF_METRICS          = 1 << 2,
	PCF_BITMAPS          = 1 << 3,
	PCF_INK_METRICS      = 1 << 4,
	PCF_BDF_ENCODINGS    = 1 << 5,
	PCF_SWIDTHS          = 1 << 6,
	PCF_GLYPH_NAMES      = 1 << 7,
	PCF_BDF_ACCELERATORS = 1 << 8
};

enum format_t
{
	PCF_FORMAT_MASK        = 0xffffff00,

	PCF_DEFAULT_FORMAT     = 0x00000000,
	PCF_INKBOUNDS          = 0x00000200,
	PCF_ACCEL_W_INKBOUNDS  = 0x00000100,
	PCF_COMPRESSED_METRICS = 0x00000100,

	PCF_GLYPH_PAD_MASK     = 3 << 0,
	PCF_BYTE_MASK          = 1 << 2,
	PCF_BIT_MASK           = 1 << 3,
	PCF_SCAN_UNIT_MASK     = 3 << 4
};

void show_type( uint32 c )
{
	const char* s;

	switch ( c )
	{
		case PCF_PROPERTIES:
			s = "PCF_PROPERTIES";
			break;
		case PCF_ACCELERATORS:
			s = "PCF_ACCELERATORS";
			break;
		case PCF_METRICS:
			s = "PCF_METRICS";
			break;
		case PCF_BITMAPS:
			s = "PCF_BITMAPS";
			break;
		case PCF_INK_METRICS:
			s = "PCF_INK_METRICS";
			break;
		case PCF_BDF_ENCODINGS:
			s = "PCF_BDF_ENCODINGS";
			break;
		case PCF_SWIDTHS:
			s = "PCF_SWIDTHS";
			break;
		case PCF_GLYPH_NAMES:
			s = "PCF_GLYPH_NAMES";
			break;
		case PCF_BDF_ACCELERATORS:
			s = "PCF_BDF_ACCELERATORS";
			break;
	}

	std::cout << "type: " << s << std::endl;
}

void show_format( uint32 c )
{
	const char* s;

	switch ( c & PCF_FORMAT_MASK )
	{
		case PCF_DEFAULT_FORMAT:
			s = "PCF_DEFAULT_FORMAT";
			break;
		case PCF_INKBOUNDS:
			s = "PCF_INKBOUNDS";
			break;
		case 0x00000100:
			s = "PCF_ACCEL_W_INKBOUNDS / PCF_COMPRESSED_METRICS";
			break;
	}

	std::cout << "format: " << s << std::endl;
}

void read_toc()
{
	::position = 0;

	uint32 c = ::next_uint32( LE );
	uint32 d = 'p' << 24 | 'c' << 16 | 'f' << 8 | 1;

	if ( c == d )
		std::cout << "c == d" << std::endl;

	::count = ::next_uint32( LE );
	std::cout << ::count << std::endl;

	::tables = new pcf_table[ count ];

	for ( int i = 0 ; i < count ; i++ )
	{
		::tables[ i ].type   = ::next_uint32( LE );
		::tables[ i ].format = ::next_uint32( LE );
		::tables[ i ].size   = ::next_uint32( LE );
		::tables[ i ].offset = ::next_uint32( LE );

		::show_type( ::tables[ i ].type     );
		::show_format( ::tables[ i ].format );
		std::cout << "size: "   << ::tables[ i ].size   << std::endl;
		std::cout << "offset: " << ::tables[ i ].offset << std::endl;
		std::cout << "------------------------------" << std::endl;
	}
}

void print_binary( uint32 c )
{
	char s[33];
	uint32 mask;

	for ( int i = 0 ; i < 32 ; i++ )
	{
		mask = 1 << ( 31 - i );
		if ( c & mask )
			s[ i ] = '1';
		else
			s[ i ] = '0';
	}

	s[32] = '\0';

	std::cout << s << std::endl;
}

void read_metric()
{
	std::cout << "------------------------------ read_metric()" << std::endl;

	for ( int i = 0 ; i < ::count ; i++ )
		if ( ::tables[ i ].type == PCF_METRICS )
			::position = ::tables[ i ].offset;

	uint32 format = ::next_uint32( LE );
	::print_binary( format );

	std::ios_base::fmtflags saveflags = std::cout.setf( std::ios_base::hex | std::ios_base::showbase );
	std::cout.unsetf( std::ios_base::dec );
	std::cout << "format: " << format << std::endl;
	std::cout.flags( saveflags );

	byte_order b = ( format & PCF_BYTE_MASK ) ? BE : LE;

	if ( b == LE )
		std::cout << "LE" << std::endl;
	else if ( b == BE )
		std::cout << "BE" << std::endl;

	switch ( format & PCF_FORMAT_MASK )
	{
		case PCF_DEFAULT_FORMAT:
			::n_metrics = ::next_uint32( b );
			std::cout << "!compressed" << std::endl;
			break;
		case PCF_COMPRESSED_METRICS:
			::n_metrics = ::next_uint16( b );
			std::cout << "compressed" << std::endl;
			break;
	}
	std::cout << "::n_metrics: " << ::n_metrics << std::endl;

	::metrics = new character_info[ ::n_metrics ];
	for ( int i = 0 ; i < ::n_metrics ; i++ )
	{
		font_metrics* m = &( ::metrics[ i ].metrics );

		switch ( format & PCF_FORMAT_MASK )
		{
			case PCF_DEFAULT_FORMAT:
				m->left_side_bearing  = ::next_uint16( b );
				m->right_side_bearing = ::next_uint16( b );
				m->character_width    = ::next_uint16( b );
				m->ascent             = ::next_uint16( b );
				m->descent            = ::next_uint16( b );
				m->attributes         = ::next_uint16( b );
				break;

			case PCF_COMPRESSED_METRICS:
				m->left_side_bearing  = ::next_uint8() - 0x80;
				m->right_side_bearing = ::next_uint8() - 0x80;
				m->character_width    = ::next_uint8() - 0x80;
				m->ascent             = ::next_uint8() - 0x80;
				m->descent            = ::next_uint8() - 0x80;
				m->attributes         = ::next_uint8() - 0x80;
				break;
		}

		std::cout << "left_side_bearing: "  << m->left_side_bearing  << std::endl;;
		std::cout << "right_side_bearing: " << m->right_side_bearing << std::endl;;
		std::cout << "character_width: "    << m->character_width    << std::endl;;
		std::cout << "ascent: "             << m->ascent             << std::endl;;
		std::cout << "descent: "            << m->descent            << std::endl;;
		std::cout << "attributes: "         << m->attributes         << std::endl;;
		std::cout << "------------------------------" << std::endl;
	}
}

enum bit_order { LSB_FIRST, MSB_FIRST };

enum data_type { BYTE = 0, SHORT = 1, INT = 2 };

void read_bitmap()
{
	std::cout << "------------------------------ read_bitmap()" << std::endl;

	for ( int i = 0 ; i < ::count ; i++ )
		if ( ::tables[ i ].type == PCF_BITMAPS )
			::position = ::tables[ i ].offset;

	uint32 format = ::next_uint32( LE );
	::print_binary( format );

	std::ios_base::fmtflags saveflags = std::cout.setf( std::ios_base::hex | std::ios_base::showbase );
	std::cout.unsetf( std::ios_base::dec );
	std::cout << "format: " << format << std::endl;
	std::cout.flags( saveflags );

	byte_order b = ( format & PCF_BYTE_MASK ) ? BE : LE;

	if ( b == LE )
		std::cout << "LE" << std::endl;
	else if ( b == BE )
		std::cout << "BE" << std::endl;

	::n_bitmaps = ::next_uint32( b );
	std::cout << "::n_bitmaps: " << ::n_bitmaps << std::endl;

	::offsets = new uint32[ ::n_bitmaps ];
	for ( int i = 0 ; i < ::n_bitmaps ; i++ )
	{
		::offsets[ i ] = ::next_uint32( b );
		std::cout << "::offsets[ " << i << " ]: " << ::offsets[ i ] << std::endl;
	}

	uint32 bitmap_sizes[ 4 ];
	for ( int i = 0 ; i < 4 ; i++ )
	{
		bitmap_sizes[ i ] = ::next_uint32( b );
		std::cout << "bitmap_sizes[ " << i << " ]: " << bitmap_sizes[ i ] << std::endl;
	}

	int bitmap_size = bitmap_sizes[ format & PCF_GLYPH_PAD_MASK ];
	std::cout << "bitmap_size: " << bitmap_size << std::endl;

	::bitmaps = new uint8[ bitmap_size ];

	for ( int i = 0 ; i < bitmap_size ; i++ )
		::bitmaps[ i ] = ::file_memory[ ::position + i ];
	::position += bitmap_size;

	bit_order d = ( format & PCF_BIT_MASK ) ? MSB_FIRST : LSB_FIRST;

	data_type padding;
	switch ( format & PCF_GLYPH_PAD_MASK )
	{
		case 0:
			padding = BYTE;
			break;
		case 1:
			padding = SHORT;
			break;
		case 2:
			padding = INT;
			break;
	}
	std::cout << "padding: " << padding << std::endl;

	data_type scan_unit;
	switch ( ( format & PCF_SCAN_UNIT_MASK ) >> 4 )
	{
		case 0:
			scan_unit = BYTE;
			break;
		case 1:
			scan_unit = SHORT;
			break;
		case 2:
			scan_unit = INT;
			break;
	}
	std::cout << "scan_unit: " << scan_unit << std::endl;

	uint8 c;
	uint32 pos;
	for ( int i = 0 ; i < ::n_bitmaps ; i++ )
	{
		pos = ::offsets[ i ];

		for ( int row = 0 ; row < 13 ; row++ )
		//for ( int row = 0 ; row < 11 ; row++ )
		{
			for ( int col = 0 ; col < 4 ; col++ )
			{
				c = ::bitmaps[ pos++ ];
				for ( int bit = 7 ; bit >= 0 ; bit-- )
					if ( ( ( c >> bit ) & 1 ) == 0 )
						std::cout << "__";
					else
						std::cout << "##";
			}
			std::cout << std::endl;
		}

		std::cout << "******************************" << std::endl;
	}
}

void read_glyph_name()
{
	std::cout << "------------------------------ read_glyph_name()" << std::endl;

	for ( int i = 0 ; i < ::count ; i++ )
		if ( ::tables[ i ].type == PCF_BITMAPS )
			::position = ::tables[ i ].offset;

	uint32 format = ::next_uint32( LE );
	::print_binary( format );

	std::ios_base::fmtflags saveflags = std::cout.setf( std::ios_base::hex | std::ios_base::showbase );
	std::cout.unsetf( std::ios_base::dec );
	std::cout << "format: " << format << std::endl;
	std::cout.flags( saveflags );

	byte_order b = ( format & PCF_BYTE_MASK ) ? BE : LE;

	if ( b == LE )
		std::cout << "LE" << std::endl;
	else if ( b == BE )
		std::cout << "BE" << std::endl;

	::glyph_count = ::next_uint32( b );
	std::cout << "::glyph_count: " << ::glyph_count << std::endl;

	::glyph_offsets = new uint32[ ::glyph_count ];
	for ( int i = 0 ; i < ::glyph_count ; i++ )
	{
		::glyph_offsets[ i ] = ::next_uint32( b );
		std::cout << "::glyph_offsets[ " << i << " ]: " << ::glyph_offsets[ i ] << std::endl;
	}

	uint32 string_size = ::next_uint32( b );
	std::cout << "string_size: " << string_size << std::endl;

	::string = new char[ string_size + 1 ];

	for ( int i = 0 ; i < string_size ; i++ )
	{
		::string[ i ] = ::next_uint8();
		//std::cout << ::string[ i ];
	}
	//std::cout << std::endl;
}

int main()
{
	::load_to_memory();

	::read_toc();
	::read_metric();
	::read_bitmap();
	::read_glyph_name();

	::free_memory();
	return 0;
}
