/*
 * OpenI2CRADIO
 * Internal eeprom Handler
 * Copyright (C) 2013-06-13 K.Ohta <whatisthis.sowhat ai gmail.com>
 * License: GPL2+LE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2,
 *  or (at your option) any later version.
 *  This library / program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; see the file COPYING. If not, write to the
 *  Free Software Foundation, 51 Franklin Street, Fifth Floor, Boston,
 *  MA 02110-1301, USA.
 *
 *  As a special exception, if you link this(includeed from sdcc) library
 *  with other files, some of which are compiled with SDCC,
 *  to produce an executable, this library does not by itself cause
 *  the resulting executable to be covered by the GNU General Public License.
 *  This exception does not however invalidate any other reasons why
 *  the executable file might be covered by the GNU General Public License.
 */

#include "iodef.h"
#include <sdcc-lib.h>
#include <pic18fregs.h> /* ONLY FOR PIC18x */
#include <signal.h>

unsigned char eeprom_readbyte(unsigned int offset)
{
    unsigned char eecon1;

    // Set address.
    EEADR = (offset & 0x0ff);
#if defined(pic46k20) || defined(pic26k20)
    EEADRH = (offset & 0x300)>>8;
#endif
    // set eecon to read.
    eecon1 = EECON1;
    eecon1 &= ~(_EEPGD | _CFGS);
    EECON1 = eecon1 | _RD;

    return EEDATA;
}

unsigned char eeprom_writebyte(unsigned int offset, unsigned char data)
{
    unsigned char eecon1;

    // Set address.
    EEADR = (offset & 0x0ff);
#if defined(pic46k20) || defined(pic26k20)
    EEADRH = (offset & 0x300)>>8;
#endif
    EEDATA = data;
    // set eecon to read.
    eecon1 = EECON1;
    eecon1 &= ~(_EEPGD | _CFGS );
    EECON1 = eecon1 | _WREN;
    INTCONbits.GIE = 0; // Disable Interrupt
    // Dummy write , needs from datasheet.
    EECON2 = 0x55;
    Nop();
    EECON2 = 0xaa;
    Nop();
    eecon1 = EECON1;
    Nop();
    INTCONbits.GIE = 1; // Enable Interrupt
    EECON1 = eecon1 & ~_WREN;

    // Verify
    if(eeprom_readbyte(offset) != data) {
        return 0x00; // Bad
    } else {
        return 0xff;
    }
}

unsigned int calcsum_byte(unsigned int seed, unsigned char byte)
{
    return seed + byte;
}

unsigned char checksum_eeprom(unsigned int seed, unsigned int offset, unsigned int bytes)
{
    unsigned int i;
    unsigned int p = offset;
    unsigned int sum = seed;
    unsigned char d;
    unsigned int check;

    for(i = 0; i < bytes; i++) {
        d = eeprom_readbyte(p);
        sum = calcsum_byte(sum, d);
        p++;
    }
    check = (eeprom_readbyte(p) << 8) + eeprom_readbyte(p+1);
    if(check != sum) return 0x00; // Bad
    return 0xff;
}

unsigned int format_eeprom(unsigned int start, unsigned int bytes)
{
    unsigned int i;
    unsigned int p = start;
    
    for(i = 0; i < bytes; i++) {
        if(eeprom_writebyte(p, 0xff) == 0) return p;
        p++;
    }
    return 0xffff; // Normal end
}