/*
 * OpenI2CRADIO
 * Config & Main routine.
 * Copyright (C) 2013-06-10 K.Ohta <whatisthis.sowhat ai gmail.com>
 * License: GPL2+LE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2,
 *  or (at your option) any later version.
 *  This library / program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; see the file COPYING. If not, write to the
 *  Free Software Foundation, 51 Franklin Street, Fifth Floor, Boston,
 *  MA 02110-1301, USA.
 *
 *  As a special exception, if you link this(includeed from sdcc) library
 *  with other files, some of which are compiled with SDCC,
 *  to produce an executable, this library does not by itself cause
 *  the resulting executable to be covered by the GNU General Public License.
 *  This exception does not however invalidate any other reasons why
 *  the executable file might be covered by the GNU General Public License.
 */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sdcc-lib.h>
#include <pic18fregs.h> /* ONLY FOR PIC18x */
#include <signal.h>
#include <delay.h>

#include "iodef.h"
#include "idle.h"
#include "i2c_io.h"
#include "akc6955.h"
#include "lcd_acm1602.h"
#include "ui.h"
#include "eeprom.h"

/*
 * Config words.
 */
#if defined(pic18f23k22) || defined(pic18f24k22) || defined(pic18f25k22) || defined(pic18f26k22)
//#pragma stack 0x200 256 // Set stack size to 256bytes.
#pragma config FOSC=INTIO67,BORV=190,BOREN=ON,PWRTEN=ON
#pragma config WDTEN=ON,WDTPS=32768
#pragma config PBADEN=OFF,MCLRE=EXTMCLR,STVREN=ON,LVP=OFF,DEBUG=ON//,XINST=ON
//#pragma config PBADEN=OFF,MCLRE=EXTMCLR,STVREN=ON,LVP=OFF//,XINST=ON
#pragma config CP0=OFF,CP1=OFF,CPB=OFF,CPD=OFF
#pragma config WRT0=OFF,WRT1=OFF,WRTB=OFF,WRTC=OFF,WRTD=OFF
#pragma config EBTR0=OFF,EBTR1=OFF,EBTRB=OFF
#endif
// For 4xK20 or 2xK20 Series
#if defined(pic18f43k20) || defined(pic18f44k20) || defined(pic18f45k20) || defined(pic18f46k20) || \
    defined(pic18f23k20) || defined(pic18f24k20) || defined(pic18f25k20) || defined(pic18f26k20)
//#pragma stack 0x200
#pragma config FOSC=HS,FCMEN=ON,PWRT=ON,BOREN=ON,BORV=22
#pragma config WDTEN=ON,WDTPS=32768,PBADEN=OFF,HFOFST=ON,LPT1OSC=OFF,MCLRE=ON
#pragma config STVREN=ON,DEBUG=ON
#pragma config CP0=OFF,CP1=OFF,CP2=OFF,CP3=OFF
#pragma config CPB=OFF,CPD=OFF
#pragma config WRT0=OFF,WRT1=OFF,WRT2=OFF,WRT3=OFF
#pragma config WRTC=OFF,WRTB=OFF,WRTD=OFF
#pragma config EBTR0=OFF,EBTR1=OFF,EBTR2=OFF,EBTR3=OFF,EBTRB=OFF
#endif

//#define _LCD_DEBUG 1

SIGHANDLER(TMR0_handler)
{
   unsigned char tmr0f;
   unsigned char t0con;

   // Stop timer0
   t0con = T0CON;
   t0con &= ~_IDLEN;
   T0CON = t0con;

   // Read IOKEYS
//   readkey_io();

   // Clear interrupt flag
   tmr0f = INTCON;
   tmr0f &= ~(_TMR0IF | _TMR0IE);
   INTCON = tmr0f;

   return;
}


DEF_INTLOW(intlow_handler)
  DEF_HANDLER(SIG_TMR0, TMR0_handler)
END_DEF


unsigned int amfreq;
unsigned int fmfreq;
unsigned char amband;
unsigned char fmband;
unsigned char fm;
unsigned char am_mode3k;
unsigned char am_userbandnum;
unsigned char fm_userbandnum;
typedef struct {
    unsigned char mode3k; // mode3k if am
    unsigned char start;
    unsigned char stop;
    unsigned int freq;
} _userband_t;
#define USER_BAND_NUM 4
_userband_t am_usrbands[USER_BAND_NUM];
_userband_t fm_usrbands[USER_BAND_NUM];

unsigned char enter_mode;
unsigned char numeric_mode;
unsigned char menu_node;
int backlight_long;
unsigned char help_flag;
int help_line;
int help_section;
int ui_language;
unsigned int ui_idlecount;
unsigned char scanflag;

int recv_signal;
int backlight_counter;
unsigned char backlight_level;

unsigned int writeword_eeprom(unsigned int p, unsigned int *sum, unsigned int word)
{
    if(eeprom_writebyte(p, word >> 8) == 0) return p; // Error
    *sum = calcsum_byte(*sum, (word >> 8));

    if(eeprom_writebyte(p, word & 0xff) == 0) return p+1; // Error
    *sum = calcsum_byte(*sum, word & 0xff);
    return 0xffff;
}

unsigned int writebyte_eeprom(unsigned int p, unsigned int *sum, unsigned char b)
{
    if(eeprom_writebyte(p, b) == 0) return p; // Error
    *sum = calcsum_byte(*sum, b);
    return 0xffff;
}

void save_eeprom(void)
{
    unsigned int p = 0;
    unsigned int sum = 0x0000;
    unsigned char i;

    // Magic word
    writeword_eeprom(p, &sum, 0x1298);
    p+= 2;
    // amfreq
    writeword_eeprom(p, &sum, amfreq);
    p+= 2;
    // amfreq
    writeword_eeprom(p, &sum, fmfreq);
    p+= 2;

    writebyte_eeprom(p, &sum, amband);
    p++;
    writebyte_eeprom(p, &sum, fmband);
    p++;
    writebyte_eeprom(p, &sum, fm);
    p++;
    writebyte_eeprom(p, &sum, am_mode3k);
    p++;
    writebyte_eeprom(p, &sum, am_userbandnum);
    p++;
    writebyte_eeprom(p, &sum, fm_userbandnum);
    p++;

    for(i = 0 ; i < USER_BAND_NUM; i++){
        writebyte_eeprom(p, &sum, am_usrbands[i].mode3k);
        writebyte_eeprom(p + 1, &sum, am_usrbands[i].start);
        writebyte_eeprom(p + 2, &sum, am_usrbands[i].stop);
        writeword_eeprom(p + 3, &sum, am_usrbands[i].freq);
        p += 5;
    }
    for(i = 0 ; i < USER_BAND_NUM; i++){
        writebyte_eeprom(p, &sum, fm_usrbands[i].mode3k);
        writebyte_eeprom(p + 1, &sum, fm_usrbands[i].start);
        writebyte_eeprom(p + 2, &sum, fm_usrbands[i].stop);
        writeword_eeprom(p + 3, &sum, fm_usrbands[i].freq);
        p += 5;
    }
    // Write checksum
    eeprom_writebyte(p, sum >> 8);
    eeprom_writebyte(p + 1, sum & 0xff);
    p+= 2;
}

unsigned int readword_eeprom(unsigned int p, unsigned int *sum)
{
    unsigned char h,l;
    unsigned int s;

    h = eeprom_readbyte(p);
    *sum = calcsum_byte(*sum, h);

    l = eeprom_readbyte(p + 1);
    *sum = calcsum_byte(*sum, l);

    s = (h << 8) + l;
    return s;
}

unsigned char readbyte_eeprom(unsigned int p, unsigned int *sum)
{
    unsigned char b;

    b = eeprom_readbyte(p);
    *sum = calcsum_byte(*sum, b);

    return b;
}


unsigned char load_eeprom(void)
{
    unsigned int p = 0;
    unsigned int sum = 0x0000;
    unsigned char i;
    unsigned int magic;

    // Magic word
    magic = readword_eeprom(p, &sum);
    if(magic != 0x1298) return 0x01; // NO MAGICWORD
    p+= 2;
    // amfreq
    amfreq = readword_eeprom(p, &sum);
    p+= 2;
    // fmfreq
    fmfreq = readword_eeprom(p, &sum);
    p+= 2;

    amband = readbyte_eeprom(p, &sum);
    p++;
    fmband = readbyte_eeprom(p, &sum);
    p++;
    fm = readbyte_eeprom(p, &sum);
    p++;
    am_mode3k = readbyte_eeprom(p, &sum);
    p++;
    am_userbandnum = readbyte_eeprom(p, &sum);
    p++;
    fm_userbandnum = readbyte_eeprom(p, &sum);
    p++;

    for(i = 0 ; i < USER_BAND_NUM; i++){
        am_usrbands[i].mode3k = readbyte_eeprom(p, &sum);
        am_usrbands[i].start  = readbyte_eeprom(p + 1, &sum);
        am_usrbands[i].stop   = readbyte_eeprom(p + 2, &sum);
        am_usrbands[i].freq   = readword_eeprom(p + 3, &sum);
        p += 5;
    }
    for(i = 0 ; i < USER_BAND_NUM; i++){
        fm_usrbands[i].mode3k = readbyte_eeprom(p, &sum);
        fm_usrbands[i].start  = readbyte_eeprom(p + 1, &sum);
        fm_usrbands[i].stop   = readbyte_eeprom(p + 2, &sum);
        fm_usrbands[i].freq   = readword_eeprom(p + 3, &sum);
        p += 5;
    }
    // Write checksum
    magic = (eeprom_readbyte(p) << 8) + eeprom_readbyte(p+1);

    p+= 2;
    if(sum != magic) return 0x00;
    return 0xff;
}


void toggle_amfm(void)
{
    if(fm != 0){
        fm = 0;
#ifndef _LCD_DEBUG
        akc6955_chg_fm(fm);
        akc6955_set_amband(amband);
        akc6955_set_freq(amfreq);
#endif
    } else {
        fm = 0xff;
#ifndef _LCD_DEBUG
        akc6955_chg_fm(fm);
        akc6955_set_fmband(fmband);
        akc6955_set_freq(fmfreq);
#endif
    }
}

static void update_status(void)
{
#ifndef _LCD_DEBUG
        recv_signal = akc6955_read_level();
        if(fm != 0){
            fmfreq = akc6955_get_freq();
        } else {
            amfreq = akc6955_get_freq();
        }
#endif

}


void set_volume(void)
{
}

void update_display(void)
{
    if(fm != 0){ // FM
        _LOCATE(0,0);
        if(fmband < AKC6955_BAND_TV1) {
            printstr("FM");
            _PUTCHAR('1' + (fmband & 7));
        } else if(fmband < AKC6955_BAND_FMUSER){
            printstr("TV");
            _PUTCHAR('1' + fmband - AKC6955_BAND_TV1);
        } else { // USER
            printstr("FMUSR");
        }
    } else { // AM
        _LOCATE(0,1);
        if(amband == AKC6955_BAND_LW) {
            printstr("LW");
        } else if(amband <AKC6955_BAND_SW1) { //MW
            printstr("MW");
            _PUTCHAR('1' + amband - AKC6955_BAND_MW1);
        } else if(amband < AKC6955_BAND_AMUSER) { //MW
            printstr("SW");
            _PUTCHAR('1' + amband - AKC6955_BAND_SW1);
        } else if(amband == AKC6955_BAND_MW4){
            printstr("MW4");
        } else {
            printstr("AMUSR");
        }
     }
     _LOCATE(16-4 ,1);
     if(fm != 0){
         printstr("MHz");
     } else {
         printstr("KHz");
     }
     _LOCATE(16-5, 1);
     if(fm != 0){
         int freq_lo = fmfreq % 100;
         int freq_hi = fmfreq / 100;
         print_numeric(freq_hi, 0xff);
         _PUTCHAR('.');
         print_numeric(freq_lo, 0xff);
     } else {
         print_numeric(amfreq, 0xff);
     }
     // Signal
     _LOCATE(0, 0);
     printstr("S=");
     print_numeric(recv_signal, 0xff);
}

void scan_start()
{
    unsigned char input_flag;
    unsigned char c;

    do {
        if(scanflag == 0){
        // New Scan
            _CLS();
            printstr("Scan A=ABORT");
            _LOCATE(0,1);
            printstr("U=6, D=4");
            do {
                input_flag = readkey_compare();
                idle(0xff80);
            } while(input_flag == 0);
            c = pop_keyinfifo();
            if(c == charcode_6){
                    akc6955_do_scan(0xff);
            } else if(c == charcode_4){
                    akc6955_do_scan(0);
            } else {
                break;
            }
            scanflag = 0xff;
        } else {
            do {
                input_flag = readkey_compare();
                idle(0xff80);
            } while(input_flag == 0);
            c = pop_keyinfifo();

            if(c == charcode_a){
                akc6955_abort_scan();
                break;
            } else if(c == charcode_4){
                akc6955_abort_scan();
                akc6955_do_scan(0);
                continue;
            } else if(c == charcode_6){
                akc6955_abort_scan();
                akc6955_do_scan(0xff);
                continue;
            }
            if(akc6955_chk_donescan() != 0) break;
        }
        idle(0xff00);
    } while(1);
    scanflag=0;
    _CLS();
    update_status();
    update_display();
}

void setfreq_direct(void)
{
    unsigned int val;
    _CLS();
    if(fm != 0){
        // FM
        _LOCATE(0,0);
        printstr("Set Freq:FM");
        val = fmfreq;
        val = read_numeric(val, 5, 7, 1);
        fmfreq = val;
        akc6955_set_freq(val);
    } else {
        // FM
        _LOCATE(0,0);
        printstr("Set Freq:AM");
        val = amfreq;
        val = read_numeric(val, 5, 7, 1);
        amfreq = val;
        akc6955_set_freq(val);
    }
    idle(0xff00);
    update_status();
    update_display();
}

void setband_direct(void)
{
    unsigned int band;
    _CLS();
    _LOCATE(0,0);
    if(fm != 0){
        printstr("Set Band:FM");
        band = fmband & 7;
        band = read_numeric(band, 2, 7, 1);
        akc6955_set_fmband(band);
        akc6955_do_tune();
        idle(0xff00);
    } else {
        printstr("Set Band:AM");
        band = amband & 0x1f;
        band = read_numeric(band, 2, 7, 1);
        akc6955_set_amband(band);
        akc6955_do_tune();
    }
    idle(0xff00);
    update_status();
    update_display();
}

void call_userband(unsigned char num)
{
    unsigned int freq;
    unsigned int ch;
    if(num >= USER_BAND_NUM) return;
    if(fm != 0){
        freq = fm_usrbands[num].freq;
        ch = ((freq - 3000) / 25) * 10;
        akc6955_set_userband(fm_usrbands[num].start, fm_usrbands[num].stop, ch,
                            fm_usrbands[num].mode3k);
    } else {
        unsigned int p = 5;
        if(am_usrbands[num].mode3k != 0) p = 3;
        freq = am_usrbands[num].freq;
        ch = freq / p;
        akc6955_set_userband(am_usrbands[num].start, am_usrbands[num].stop, ch,
                            am_usrbands[num].mode3k);
    }
    if(fm != 0) {
        fmband = AKC6955_BAND_AMUSER;
    } else {
        amband = AKC6955_BAND_AMUSER;
    }
    idle(0xff00);
    update_status();
    update_display();
}

void set_userband(void)
{
    unsigned int from,to;
    unsigned char c;
    unsigned char p;
    unsigned char mode3k;
    unsigned int input_flag;
    char cc;

    _CLS();
    _LOCATE(0,0);
    printstr("User ch:");
    do {
        input_flag = readkey_compare();
        idle(0xff80);
    } while(input_flag == 0);
    c = pop_keyinfifo();

    if(c > charcode_0) return;
    if(c < charcode_1) return;
    if(c == charcode_0) {
        c = 0;
    } else {
        c = c - charcode_1 + 1;
    }
    if(c >= USER_BAND_NUM) return;
    if(fm != 0){
        from = fm_usrbands[c].start * 80 + 3000; // 32*25/10
        to = fm_usrbands[c].stop * 80 + 3000;
        _CLS();
        _LOCATE(0,0);
        printstr("FM #");
        print_numeric_nosupress(c, 1);
        printstr(" From:");
        from = read_numeric(from, 5, 7, 1);
        _CLS();
        _LOCATE(0,0);
        printstr("FM #");
        print_numeric_nosupress(c, 1);
        printstr(" To:");
        to = read_numeric(to, 5, 7, 1);
        fm_usrbands[c].start = (from - 3000) / 80;
        fm_usrbands[c].stop = (to - 3000) / 80;
        fm_usrbands[c].freq = from * 80 + 3000;
        fm_userbandnum = c;
    } else {
        mode3k = am_usrbands[c].mode3k;
        p = 96; // 3*32
        if(mode3k == 0) p = 160; // 5*32
        from = am_usrbands[c].start * p; 
        to = am_usrbands[c].stop * p;
        _CLS();
        _LOCATE(0,0);
        printstr("AM #");
        print_numeric_nosupress(c, 1);
        printstr(" Step:");
        _LOCATE(0,1);
        printstr("0=3k 1=5k");
        do {
            input_flag = readkey_compare();
            idle(0xff80);
        } while(input_flag == 0);
        cc = pop_keyinfifo();

        if(cc == charcode_0){
            p = 96;
            mode3k = 0xff;
        } else if(cc = charcode_1) {
            p = 160;
            mode3k = 0;
        }
        _CLS();
        _LOCATE(0,0);
        printstr("AM #");
        print_numeric_nosupress(c, 1);
        printstr(" From:");
        from = read_numeric(from, 5, 7, 1);
        _CLS();
        _LOCATE(0,0);
        printstr("AM #");
        print_numeric_nosupress(c, 1);
        printstr(" To:");
        to = read_numeric(to, 5, 7, 1);
        am_usrbands[c].start = from / p;
        am_usrbands[c].stop = to  / p;
        am_usrbands[c].mode3k = mode3k;
        am_usrbands[c].freq = from * p;
        am_userbandnum = c;
    }
    call_userband(c);
}

void input_userband(void)
{
    unsigned char c;
    unsigned char input_flag;
    do{
    _CLS();
    _LOCATE(0,0);
    printstr("User Band");
    _LOCATE(0,1);
    printstr("   #");
    do {
        input_flag = readkey_compare();
        idle(0xff80);
    } while(input_flag == 0);
    c = pop_keyinfifo();

    if((c >= charcode_a) && (c <= charcode_f)){
        break;
    }
    if(c == charcode_0) {
        _PUTCHAR('0');
        if(fm != 0){
           fm_userbandnum = 0;
        } else {
           am_userbandnum = 0;
        }
        call_userband(0);
    } else {
        c = c - charcode_1 + 1;
        if(c < USER_BAND_NUM) {
            _PUTCHAR(c + '0');
            if(fm != 0){
               fm_userbandnum = c;
            } else {
                am_userbandnum = c;
            }
            call_userband(c);
        }
    }
    idle(0xff00);
    } while(1);
    _CLS();
}


void main_menu(void)
{
    unsigned char c;
    unsigned int input_flag;
    _CLS();
    _LOCATE(0,0);
    printstr("Menu:F=HELP");
    _LOCATE(1,0);
    printstr("A=CANCEL");
    do{
        do {
            input_flag = readkey_compare();
            idle(0xff80);
        } while(input_flag == 0);

        c = pop_keyinfifo();
        if((c < charcode_1) || ( c >charcode_s3)) {
            idle(0xff00);
            continue; // Error
        }
        if(c == charcode_f){
            // HELP
        } else if(c == charcode_a){
            // Cancel
            break;
        } else if(c == charcode_1){
            // AM
            fm = 0;
            akc6955_chg_fm(fm);
            akc6955_set_amband(amband);
            akc6955_set_freq(amfreq);
            break;
        } else if(c == charcode_2){
            // Band
            setband_direct();
            break;
        } else if(c == charcode_3){
            // Band
            setfreq_direct();
            break;
        } else if(c == charcode_4){
            // fm
            fm = 0xff;
            akc6955_chg_fm(fm);
            akc6955_set_fmband(fmband);
            akc6955_set_freq(fmfreq);
            break;
        } else if(c == charcode_5){
            // Scan
            break;
        } else if(c == charcode_6){
            // Set gain
            break;
        } else if(c == charcode_7){
            // Set volume
            break;
        } else if(c == charcode_8){
            // Set sensitivity
            break;
        } else if(c == charcode_9){
            // Set NF
            break;
        } else if(c == charcode_0){
            // Setup Menu
            break;
        } else if(c == charcode_b){
            // Call userband
            input_userband();
            break;
        } else if(c == charcode_c){
            // Set userband
            set_userband();
            break;
        } else if(c == charcode_d){
            // Reserve
            break;
        } else if(c == charcode_e){
            // Reserve
            break;
        }
        idle(0xff00);
    } while(1);
}

void setfreq_updown(unsigned char ctlword)
{
#ifndef _LCD_DEBUG
    switch(ctlword){
        case charcode_8: // Change band
            if(fm != 0){
                amband++;
                if(amband > 18) amband = 0;
//                amfreq = akc6955_setfreq(amfreq)
                akc6955_set_amband(amband);
                _AKC6955_WAIT_62_5MS(); // 62.5ms
                amband = akc6955_get_amband();
                amfreq = akc6955_get_freq();
            } else {
                fmband++;
                if(fmband > 7) fmband = 0;
//                amfreq = akc6955_setfreq(amfreq)
                akc6955_set_fmband(fmband);
                _AKC6955_WAIT_62_5MS(); // 62.5ms
                fmband = akc6955_get_fmband();
                fmfreq = akc6955_get_freq();
            }
            break;
        case charcode_2: // Change band
            if(fm != 0){
                amband--;
                if(amband == 0) amband = 18;
                if(amband >= 18) amband = 18;
//                amfreq = akc6955_setfreq(amfreq)
                akc6955_set_amband(amband);
                _AKC6955_WAIT_62_5MS(); // 62.5ms
                amband = akc6955_get_amband();
                amfreq = akc6955_get_freq();
            } else {
                fmband--;
                if(fmband == 0) fmband = 7;
                if(fmband >= 7) fmband = 7;
//                amfreq = akc6955_setfreq(amfreq)
                akc6955_set_fmband(fmband);
                _AKC6955_WAIT_62_5MS(); // 62.5ms
                fmband = akc6955_get_fmband();
                fmfreq = akc6955_get_freq();
            }
            break;
        case charcode_4: // Down Freq;
            if(fm != 0){
                fmfreq = akc6955_down_freq(10); // DOWN 100KHz
            } else {
                amfreq = akc6955_down_freq(10); // DOWN 10KHz
            }
            break;
        case charcode_6: // Down Freq;
            if(fm != 0){
                fmfreq = akc6955_up_freq(10); // UP 100KHz
            } else {
                amfreq = akc6955_up_freq(10); // UP 10KHz
            }
            break;
        case charcode_7: // Down Fast;
            if(fm != 0){
                fmfreq = akc6955_down_freq(50); // DOWN 500KHz
            } else {
                amfreq = akc6955_down_freq(50); // DOWN 50KHz
            }
            break;
        case charcode_9: // Down Fast;
            if(fm != 0){
                fmfreq = akc6955_up_freq(50); // UP 100KHz
            } else {
                amfreq = akc6955_up_freq(50); // UP 10KHz
            }
            break;
        case charcode_1: // Down Slow;
            if(fm != 0){
                fmfreq = akc6955_down_freq(5); // DOWN 50KHz
            } else {
                amfreq = akc6955_down_freq(5); // DOWN 50KHz
            }
            break;
        case charcode_3: // Down Slow;
            if(fm != 0){
                fmfreq = akc6955_up_freq(5); // UP 50KHz
            } else {
                amfreq = akc6955_up_freq(5); // UP 5KHz
            }
            break;
        case charcode_0: // Step
            if(fm == 0){
                if(am_mode3k == 0) {
                    am_mode3k = 0xff;
                } else {
                    am_mode3k = 0;
                }
                amfreq = akc6955_mode3k(am_mode3k);
            }
            break;
        case charcode_a: // Toggle FM
            toggle_amfm();
            break;
        case charcode_b:
            input_userband();
            break;
        case charcode_c:
            set_userband();
            break;
        case charcode_d:
            set_volume();
            break;
        case charcode_e: // Backlight ON/OFF
            if(backlight_counter > 0) {
	       backlight_counter = 0;
	    } else {
	       backlight_counter = backlight_long;
	    }
            break;
        case charcode_f:
            main_menu();
            break;
        default:
            break;
    }
#endif
}
/*
 * 
 */
static void setdefault(void)
{
    char i;
    amfreq = 954;
    fmfreq = 8000; // 10KHz order.
    amband = AKC6955_BAND_MW2;
    fmband = AKC6955_BAND_FM2;
    am_mode3k = 0xff;
    fm = 0;
    recv_signal = 0;
    am_userbandnum = 0;
    fm_userbandnum = 0;
    for(i = 0; i < 4; i++){
        am_usrbands[i].start = 0x19;
        am_usrbands[i].stop  = 0x32;
    }
    for(i = 0; i < 4; i++){
        fm_usrbands[i].start = 0x19;
        fm_usrbands[i].stop  = 0x32;
    }


}

int main(void)
{
    char readchar;
    unsigned char input_flag;
    unsigned char c;
    unsigned int i;
#ifdef _LCD_DEBUG
    unsigned char power_flag;
#endif
    OSCCON =  _IDLEN & 0b11111100;

    keyin_init();
    keyin_ioinit();
    idle_init();

    //i2c1_init();

    _AKC6955_WAIT_125_0MS(); // Wait 125ms
#ifdef _LCD_DEBUG
    power_flag = 0xff;
#endif
    backlight_long = 256;
    backlight_counter = backlight_long;
    backlight_level = 255;
    ui_idlecount = 0xf800;

    acm1602_init(0xa0, 1); //Init LCD
    _AKC6955_WAIT_125_0MS(); // Wait 125ms
    _LOCATE(0,0);
    printstr("Hello;-)");
    PORTDbits.RD3 = 1;
    LATDbits.LATD3 = 1;
    
#if 1
    i = 10001;
    do {
        idle(0xf800);
        ClrWdt();
        _LOCATE(0,1);
        print_numeric_nosupress(i, 5);
        i = i + 1;
    }while(1);
#else
    switch(load_eeprom()) {
        case 0x01: // No magic-word
            idle(0xff80);
            _CLS();
            setdefault();
            _LOCATE(0,0);
            printstr("EEPROM FORMATTING");
            _LOCATE(0,1);
            printstr("Press any key");
            do {
                input_flag = readkey_compare();
                idle(0xff80);
            } while(input_flag == 0);
            c = pop_keyinfifo();
            format_eeprom(0,254);
            save_eeprom();
            break;
        case 0x00: // Checksum error
            idle(0xff80);
            _CLS();
            _LOCATE(0,0);
            printstr("X-) Sum Error");
            _LOCATE(0,1);
            printstr("Press any key to format");
            do {
                input_flag = readkey_compare();
                idle(0xff80);
            } while(input_flag == 0);
            
            c = pop_keyinfifo();
            format_eeprom(0,254);
            setdefault();
            save_eeprom();
            break;
        case 0xff: // Success
            break;
        default: // Unknown error
            setdefault();
            break;
    }
    // Init AKC6955
    /* Check EEPROM */
    /* Push default parameters to AKC6955*/
    scanflag = 0;
#ifndef _LCD_DEBUG
    akc6955_chg_fm(fm); // Set to AM
    akc6955_set_amband(amband);
    akc6955_set_freq(amfreq); // Dummy, TBS (954KHz)
    akc6955_set_power(0xff); // Power ON
#endif
#ifdef _LCD_DEBUG
    idle(0xf000);
#else
    idle(0xff00);
#endif
    _CLS();
    update_status();
    update_display();
    do {
        /* Main routine*/
        input_flag = readkey_compare();
        if(input_flag != 0){
            readchar = pop_keyinfifo();
            if((readchar >= charcode_1) && (readchar <= charcode_f)) {
                    setfreq_updown(readchar);
            } else {
                // Other is skip
            }
        }
        // Check battery (include idle?)
        // Read AKJC6955's status
	update_status();
        // Putstring to LCD.

        update_display();
        if(backlight_counter > 0) {
            backlight_counter--;
            set_backlight(0xff, backlight_level); // Turn ON
        } else {
            set_backlight(0x00, 0); // Turn OFF
        }
#ifdef _LCD_DEBUG
        if(power_flag != 0x00) {
            power_flag = 0x00;
        } else {
            power_flag = 0xff;
        }
        setsignal_tune(power_flag);
#endif
        idle(ui_idlecount);
    } while(1);
#endif
}

