/***********************************************************************//**
	@file
	$Revision: 83 $
	$Author: yatsuhashi $
	$Date:: 2009-03-29 01:13:14 +0900#$
***************************************************************************/
#include <sstream>
#include <assert.h>
#include "Command.h"

namespace openmj {

const char* Command::TABLE[] = {
    "hello", 
    "error", 
    "gamestart", 
    "gameend", 
    "kyokustart", 
    "ready?", 
    "point", 
    "dice", 
    "haipai", 
    "open", 
    "dora", 
    "tsumo", 
    "sutehai?", 
    "sutehai", 
    "naku?", 
    "say", 
    "agari", 
    "ryukyoku", 
    "tenpai?", 

    "richi", 
    "ron", 
    "yes", 
    "no", 
    "ok", 

    0
};
/***********************************************************************//**
	デフォルトコンストラクタ
***************************************************************************/
Command::Command()
    : serial_(0), 
      command_(CMD_NULL)
{}
/***********************************************************************//**
	コンストラクタ
	@param	serial	シリアル番号
        @param	command	コマンドID
***************************************************************************/
Command::Command(unsigned int serial, int command)
    : serial_(serial), 
      command_(command)
{}
/***********************************************************************//**
	@return	シリアル
***************************************************************************/
unsigned int Command::getSerial() const {
    return serial_;
}
/***********************************************************************//**
	@return	コマンドID
***************************************************************************/
int Command::getCommand() const {
    return command_;
}
/***********************************************************************//**
	@return	引数の数
***************************************************************************/
int Command::getArgNum() const {
    return args_.size();
}
/***********************************************************************//**
	引数を参照する
	@param	index	インデックス
	@return		引数
***************************************************************************/
const std::string& Command::getArg(int index) const {
    return args_[index];
}
/***********************************************************************//**
	@return	引数のベクタ
***************************************************************************/
const Command::Args& Command::getArgs() const {
    return args_;
}
/***********************************************************************//**
	コマンドIDをセットする
	@param	command	コマンドID
***************************************************************************/
void Command::setCommand(int command) {
    command_ = command;
}
/***********************************************************************//**
	引数を追加する
	@param	arg	追加する引数
***************************************************************************/
void Command::append(const char* arg) {
    args_.push_back(std::string(arg));
}
/***********************************************************************//**
	コマンド文字列を解析する
	@param	command	コマンド文字列
***************************************************************************/
bool Command::parse(const char* command) {
    args_.clear();
    const char* pt = command;
    while(pt) {
        const char* sep = strchr(pt, ' ');
        if(sep) {
            args_.push_back(Decode(pt, sep - pt));
            pt = sep + 1;
       }
        else {
            args_.push_back(Decode(pt));
            pt = 0;
        }
    }
    if(args_.size() < 2) {
        return false;
    }
    serial_ = atoi(args_.front().c_str());
    args_.erase(args_.begin());
    int i = 0;
    while(TABLE[i]) {
        if(args_.front() == TABLE[i]) {
            command_ = i + 1;
            args_.erase(args_.begin());
            return true;
        }
        i++;
    }
    return false;
}
/***********************************************************************//**
	文字列に変換する
***************************************************************************/
std::string Command::toString() const {
    assert(serial_ != 0);
    std::ostringstream buff;
    if(serial_ != 0) {
        buff << serial_;
    }
    if(command_ != CMD_NULL) {
        buff << " " << TABLE[command_ - 1];
    }
    for(Args::const_iterator arg = args_.begin();
        arg != args_.end();
        arg++) {
        buff << " " << Encode(arg->c_str());
    }
    return buff.str();
}
/***********************************************************************//**
	エンコード.
***************************************************************************/
std::string Command::Encode(const char* src) {
    std::string des;
    char buff[8];
    char c;
    while((c = *src++) != '\0') {
        if(c != '%' && c > 0x20 && c < 0x7f) {
            des.append(1, c);
        }
        else {
            sprintf(buff, "%%%02X", c);
            des.append(buff);
        }
    }
    return des;
}
/***********************************************************************//**
	デコード.
***************************************************************************/
std::string Command::Decode(const char* src, int len) {
    std::string des;
    char c;
    while((c = *src++) != '\0') {
        if(c == '%') {
            int code;
            sscanf(src, "%02X", &code);
            c = static_cast<char>(code);
            src += 2;
        }
        des.append(1, c);
        if(len > 0 && (--len) == 0) {
            break;
        }
    }
    return des;
}
/***********************************************************************//**
	$Id: Command.cpp 83 2009-03-28 16:13:14Z yatsuhashi $
***************************************************************************/
}	/* namespace openmj */
