/***********************************************************************//**
	@file
	$Revision$
	$Author$
	$Date::                           $
***************************************************************************/
#include <assert.h>
#include "libmahjong.h"
#include "common/Player.h"
#include "Taku.h"
#include "Resource.h"
#include "MainFrame.h"
#include "DrawPacketHai.h"
#include "Choice.h"

namespace openmj {
namespace gui {

const wxPoint Taku::KAWA_TOP(Resource::HAI_WIDTH * (-4), 
                             Resource::HAI_WIDTH * 3 + 
                             Resource::HAI_HEIGHT);
const wxPoint Taku::HAND_TOP(Resource::HAI_WIDTH * (-7), 
                             KAWA_TOP.y + 
                             Resource::HAI_HEIGHT * 3.5f);
/***********************************************************************//**
	コンストラクタ.
***************************************************************************/
Taku::Taku(wxWindow* parent, wxWindowID id, const wxPoint& pos, 
           const wxSize& _size, long style, const wxString& name)
    : super(parent, id, pos, _size, style, name), 
      choice_(0)
{
    width = Resource::HAI_WIDTH * 20;
    wxSize size(width, width);
    SetSize(size);
    SetMinSize(size);
    wxColour colour(0, 128, 0);
    pen.SetColour(colour);
    brush.SetColour(colour);
    wxBitmap bitmap(width, width);
    buff.SelectObject(bitmap);
    buff.SetTextForeground(*wxWHITE);
    clear();
}
/***********************************************************************//**
	クライアントの参照を返す.
***************************************************************************/
const Client& Taku::getClient() const {
    return static_cast<MainFrame*>(GetParent())->getClient();
}
/***********************************************************************//**
	プレイヤーをセットする.
***************************************************************************/
void Taku::setPlayers(const std::vector<const Player*>& players) {
    this->players = players;
}
/***********************************************************************//**
	プレイヤーをクリアする.
***************************************************************************/
void Taku::clearPlayers() {
    players.clear();
}
/***********************************************************************//**
	局開始
***************************************************************************/
void Taku::startKyoku() {
    dora.clear();
}
/***********************************************************************//**
	ドラを追加する.
	@param	hai	ドラ表示牌
***************************************************************************/
void Taku::appendDora(const mahjong::Hai* hai) {
    dora.append(hai);
}
/***********************************************************************//**
	
***************************************************************************/
void Taku::draw() {
    clear();
    drawDora();
    for(int i = 0; i < players.size(); i++) {
        const Player* player = players.at(i);
        drawKawa(player->getKawa(), i);
        drawHand(player, i);
    }
    drawQueue_.flush(buff);
    /*
    drawInfo();
    for(int i = 0; i < players.size(); i++) {
        drawPlayerInfo(players.at(i), i);
    }
    */
    update();
    Refresh();
}
/***********************************************************************//**
	
***************************************************************************/
void Taku::drawHai(const wxPoint& pos, const mahjong::Hai* hai, 
                   int rotate, int flag) {
    static const int MATRIX_TABLE[4][4] = {
        { 1, 0, 0, 1 }, 
        { 0, 1, -1, 0 }, 
        { -1, 0, 0, -1 }, 
        { 0, -1, 1, 0 }
    };
    const int* matrix = MATRIX_TABLE[rotate];
    wxPoint drawPos(pos.x * matrix[0] + pos.y * matrix[1] + width / 2, 
                    pos.x * matrix[2] + pos.y * matrix[3] + width / 2);
    DrawPacketHai* packet = new DrawPacketHai(drawPos, hai, rotate, flag);
    drawQueue_.appendPacket(packet);
}
/***********************************************************************//**
	
***************************************************************************/
void Taku::drawHand(const mahjong::Hand* hand, int rotate) {
    wxPoint pos = HAND_TOP;
    const mahjong::HaiArray& menzen = hand->getMenzen();
    int flag = 0;
    if(hand->isClose()) {
        flag |= DrawPacketHai::FLAG_CLOSE;
    }
    else if(hand->isOpen() || rotate != 0) {
        flag |= DrawPacketHai::FLAG_OPEN;
    }
    for(int i = 0; i < menzen.getSize(); i++) {
        wxPoint drawPos = pos;
        if(rotate == 0) {
            if(!canChoice(i)) {
                flag |= DrawPacketHai::FLAG_DISABLE;
            }
            else if(isChoice(i)) {
                drawPos.y--;
            }
        }
        if(i % 3 == 1 && i == menzen.getSize() - 1) {
            drawPos.x++;
        }
        drawHai(drawPos, menzen[i], rotate, flag);
        flag &= ~DrawPacketHai::FLAG_DISABLE;
        pos.x += Resource::HAI_WIDTH;
    }

    /* 見えない牌(他家) */
    flag &= DrawPacketHai::FLAG_CLOSE;
    for(int i = 0; i < hand->getFusehaiNum(); i++) {
        drawHai(pos, 0, rotate, flag);
        pos.x += Resource::HAI_WIDTH;
    }

    pos.x = Resource::HAI_WIDTH * 10;
    pos.y += 12;
    for(int i = 0; i < hand->getOpenMentsuNum(); i++) {
        drawOpenMentsu(pos, hand->getOpenMentsu(i), rotate);
    }
}
/***********************************************************************//**
	晒した面子を描画する
	@param	mentsu	面子
	@param	x	右端のX座標
	@param	y	Y座標
	@param	rotate	向き
	@return		次の面子の右端のX座標
***************************************************************************/
void Taku::drawOpenMentsu(wxPoint& pos, const mahjong::Mentsu& mentsu, 
                          int rotate) {
    for(int i = mentsu.getSize() - 1; i >= 0; i--) {
        pos.x -= Resource::HAI_WIDTH;
        drawHai(pos, mentsu.getHai(i), rotate, DrawPacketHai::FLAG_OPEN);
    }
}
/***********************************************************************//**
	
***************************************************************************/
void Taku::drawKawa(const mahjong::Kawa& kawa, int rotate) {
    wxPoint pos = KAWA_TOP;
    int six = 0;
    int line = 1;
    for(int i = 0; i < kawa.getSize(); i++) {
        const mahjong::Sutehai& sutehai = kawa[i];
        int flag = DrawPacketHai::FLAG_OPEN;
        if(sutehai.isNaki()) {
            flag |= DrawPacketHai::FLAG_NAKI;
        }
        else if(&sutehai == getClient().getLastSutehai()) {
            flag |= DrawPacketHai::FLAG_LAST;
        }
        if(sutehai.isTsumogiri()) {
            flag |= DrawPacketHai::FLAG_TSUMOGIRI;
        }
        if(sutehai.isRichi()) {
            flag |= DrawPacketHai::FLAG_BEND;
        }
        drawHai(pos, kawa[i].getHai(), rotate, flag);
        if(line < 3 && (++six) % 6 == 0) {
            pos.x = KAWA_TOP.x;
            pos.y += Resource::HAI_HEIGHT;
            line++;
        }
        else {
            if(flag & DrawPacketHai::FLAG_BEND) {
                pos.x += Resource::HAI_HEIGHT;
            }
            else {
                pos.x += Resource::HAI_WIDTH;
            }
        }
    }
}
/***********************************************************************//**
	ドラ表示牌を表示する.
***************************************************************************/
void Taku::drawDora() {
    static const int TOP_X = Resource::HAI_WIDTH * (-2.5f);
    wxPoint pos(TOP_X, 0);
    for(int i = 0; i < 5; i++) {
        drawHai(pos, 0, 0, DrawPacketHai::FLAG_CLOSE);
        pos.x += Resource::HAI_WIDTH;
    }
    pos.x = TOP_X;
    pos.y -= 12;
    for(int i = 0; i < dora.getSize(); i++) {
        drawHai(pos, dora[i], 0, (DrawPacketHai::FLAG_OPEN | 
                                  DrawPacketHai::FLAG_UPPER));
        pos.x += Resource::HAI_WIDTH;
    }
    for(int i = 0; i < 5 - dora.getSize(); i++) {
        drawHai(pos, 0, 0, (DrawPacketHai::FLAG_CLOSE | 
                            DrawPacketHai::FLAG_UPPER));
        pos.x += Resource::HAI_WIDTH;
    }
}
/***********************************************************************//**
	
***************************************************************************/
void Taku::drawInfo() {
    static const wxString kaze[] = {
        wxT("東"), wxT("南"), wxT("西"), wxT("北")
    };
    const Client& client = getClient();
    const wxString& bakaze = kaze[client.getBakaze()->number - 1];
    wxString info = wxString::Format(wxT("%s%d局"), bakaze.GetData(), 
                                     client.getOya() + 1);
    if(client.getTsumibou() > 0) {
        info << wxString::Format(wxT(" %d本場"), 
                                 client.getTsumibou() + 1);
    }
    buff.DrawText(info, width / 2 - Resource::HAI_WIDTH * 2, 
                  width / 2);
}
/***********************************************************************//**
	
***************************************************************************/
void Taku::drawPlayerInfo(const Player* player, int seat) {
    int x = (seat == 0 || seat == 1) ? width - PLAYER_INFO_W : 0;
    int y = (seat == 0 || seat == 3) ? width - PLAYER_INFO_H : 0;
    wxString name = wxString::FromUTF8(player->getName().c_str());
    wxString info = wxString::Format(wxT("%s\n%d\n"), 
                                     name.GetData(), player->getPoint());
    buff.DrawText(info, x, y);
}
/***********************************************************************//**
	
***************************************************************************/
void Taku::update() {
    wxPaintDC dc(this);
    dc.Blit(0, 0, width, width, &buff, 0, 0);
}
/***********************************************************************//**
	
***************************************************************************/
void Taku::clear() {
    buff.SetPen(pen);
    buff.SetBrush(brush);
    const wxSize& size = GetSize();
    buff.DrawRectangle(0, 0, size.GetWidth(), size.GetHeight());
}
/***********************************************************************//**
	牌選択を開始する
***************************************************************************/
void Taku::beginChoice(const Choice* choice) {
    choice_ = choice;
}
/***********************************************************************//**
	牌選択を終了する
***************************************************************************/
void Taku::endChoice() {
    choice_ = 0;
}
/***********************************************************************//**
	カーソル位置の手牌のインデックスを返す
***************************************************************************/
int Taku::getChoiceIndex(int x, int y) {
    const int menzenSize = players.at(0)->getMenzen().getSize();
    y -= width / 2 + HAND_TOP.y - 
        (Resource::HAI_HEIGHT + Resource::HAI_THICK);
    if(y >= 0 && y < Resource::HAI_HEIGHT + Resource::HAI_THICK) {
        x -= width / 2 + HAND_TOP.x;
        int index = x / Resource::HAI_WIDTH;
        if(index >= 0 && index < menzenSize) {
            return index;
        }
    }
    return -1;
}
/***********************************************************************//**
	
***************************************************************************/
bool Taku::isChoice(int index) {
    return (choice_ && choice_->getCurrent() == index);
}
/***********************************************************************//**
	
***************************************************************************/
bool Taku::canChoice(int index) {
    return (!choice_ || choice_->canChoice(index));
}
/***********************************************************************//**
	$Id$
***************************************************************************/
}	/* namespace gui */
}	/* namespace openmj */
