//############################################################################//
//Made in 2005-2013 by Artyom Litvinovich
//Modified 2013-02-10 by Friedrich Kastner-Masilko
//Dynamic vessel DLL
//############################################################################// 
#pragma once
#define ORBITER_MODULE
#define _CRT_SECURE_NO_WARNINGS
#define _CRT_NONSTDC_NO_DEPRECATE    
//############################################################################//
#include <math.h>
#include <stdio.h>
#include "orbitersdk.h"
#include "resource.h"
//############################################################################//
#include "genericvessel.h"

//############################################################################//
//Global definitions

//Does it check out?
int inited;

//xves.dll handle
HMODULE hlib;

//The call to the xves for data block
void (__stdcall *make_vessel)(DWORD,DWORD,char*,char*);

//############################################################################//
//From DG code
// ==============================================================
// Airfoil coefficient functions
// Return lift, moment and zero-lift drag coefficients as a
// function of angle of attack (alpha or beta)
// ==============================================================
//############################################################################//
// 1. vertical lift component (wings and body)
void VLiftCoeff (VESSEL *v, double aoa, double M, double Re, void *context, double *cl, double *cm, double *cd)
{
 int i;
 const int nabsc = 9;
 static const double AOA[nabsc] = {-180*RAD,-60*RAD,-30*RAD, -2*RAD, 15*RAD,20*RAD,25*RAD,60*RAD,180*RAD};
 static const double CL[nabsc]  = {       0,      0,   -0.4,      0,    0.7,     1,   0.8,     0,      0};
 static const double CM[nabsc]  = {       0,      0,  0.014, 0.0039, -0.006,-0.008,-0.010,     0,      0};
 for (i = 0; i < nabsc-1 && AOA[i+1] < aoa; i++);
 double f = (aoa-AOA[i]) / (AOA[i+1]-AOA[i]);
 *cl = CL[i] + (CL[i+1]-CL[i]) * f;  // aoa-dependent lift coefficient
 *cm = CM[i] + (CM[i+1]-CM[i]) * f;  // aoa-dependent moment coefficient
 double saoa = sin(aoa);
 double pd = 0.015 + 0.4*saoa*saoa;  // profile drag
 *cd = pd + oapiGetInducedDrag (*cl, 1.5, 0.7) + oapiGetWaveDrag (M, 0.75, 1.0, 1.1, 0.04);
 // profile drag + (lift-)induced drag + transonic/supersonic wave (compressibility) drag
}
//############################################################################//
// 2. horizontal lift component (vertical stabilisers and body)
void HLiftCoeff (VESSEL *v, double betta, double M, double Re, void *context, double *cl, double *cm, double *cd)
{
 int i;
 const int nabsc = 8;
 static const double BETA[nabsc] = {-180*RAD,-135*RAD,-90*RAD,-45*RAD,45*RAD,90*RAD,135*RAD,180*RAD};
 static const double CL[nabsc]   = {       0,    +0.3,      0,   -0.3,  +0.3,     0,   -0.3,      0};
 for (i = 0; i < nabsc-1 && BETA[i+1] < betta; i++);
 *cl = CL[i] + (CL[i+1]-CL[i]) * (betta-BETA[i]) / (BETA[i+1]-BETA[i]);
 *cm = 0.0;
 *cd = 0.015 + oapiGetInducedDrag (*cl, 1.5, 0.6) + oapiGetWaveDrag (M, 0.75, 1.0, 1.1, 0.04);
}        
//############################################################################// 
void GenericVessel::do_init()
{
 ph_h=NULL;
 th_h=NULL;
 thg_h=NULL; 
 exh_h=NULL;
 msh_h=NULL;
 msh_idh=NULL;
 ani=NULL;
 att_h=NULL;
}      
//############################################################################// 
void GenericVessel::do_clean()
{    
     
 for(int i=0;i<sc3.Config.attcnt;i++)DelAttachment(att_h[i]);
 for(int i=0;i<sc3.Config.propcnt;i++)DelPropellantResource(ph_h[i]);
 for(int i=0;i<sc3.Config.thcnt;i++)DelThruster(th_h[i]);
 for(int i=0;i<sc3.Config.thgcnt;i++)DelThrusterGroup(thg_h[i]);
 for(int i=0;i<sc3.Config.exhcnt;i++)DelExhaust(exh_h[i]);

 for(int i=0;i<sc3.Config.mshcnt;i++)if(sc3.Data.msh[i].typ==0){
  DelMesh(msh_idh[i]); 
  //msh_h[i]=oapiLoadMeshGlobal(gv_data.msh[i].nam);
 }

 for(int i=0;i<sc3.Config.anicnt;i++)DelAnimation(ani[i].id);


 //for(i=0;i<gv_config.beaccnt;i++)DelBeacon(&gv_data.beac[i].bl);
 //for(i=0;i<gv_config.dockcnt;i++)DelDock(gv_data.dock[i].pos,gv_data.dock[i].dir,gv_data.dock[i].rot);
  
 delete [] ph_h;
 delete [] th_h;
 delete [] thg_h;
 delete [] exh_h;
 delete [] msh_h;
 delete [] msh_idh;
 if (ani) for(int i=0;i<sc3.Config.anicnt;i++)
 {
  for(int j=0;j<ani[i].grpcnt;j++) delete [] ani[i].grp[j];
  delete [] ani[i].grp;
  delete [] ani[i].comp;
 }
 delete [] ani;
 delete [] att_h;    
}
//############################################################################//
GenericVessel::GenericVessel(OBJHANDLE hObj,int fmodel):VESSEL3(hObj,fmodel){do_init();}   
GenericVessel::~GenericVessel(){do_clean();}
//############################################################################//
//############################################################################//
void GenericVessel::clbkSetClassCaps(FILEHANDLE cfg)
{
 int i,j,g,k,n,a,b,arm_grap;
 THRUSTER_HANDLE **grps;
 int *grpc;     
 VECTOR3 xp,xr;

 //Get the data block
 char *iniFile=NULL;
 char *className=GetClassName(), *cn=NULL;
 char *name=GetName();
 strcpy(cn=new char[(k=strlen(className))+1], className);

 strlwr(cn);
 for(i=0;i<k;i++)if(cn[i]=='/')cn[i]='\\';
 if (strncmp(cn, SPACECRAFTCLASS, strlen(SPACECRAFTCLASS))==0){
  //INI file at legacy location - fixed prefix here, because it is not clear if SC3 respected custom config paths
  iniFile=new char[strlen(name)+strlen(SPACECRAFTINIPATH)];
  sprintf(iniFile, SPACECRAFTINIPATH, name);
 }else{
  //configuration file is also INI file, read orbiter.cfg first to get custom config paths
  FILEHANDLE f=oapiOpenFile(ORBITERCONFIG, FILE_IN);
  if (!oapiReadItem_string(f, CONFIGDIRTAG, configDir)) strcpy(configDir, CONFIGDIRDEFAULT);
  if (!oapiReadItem_string(f, MESHDIRTAG, meshDir)) strcpy(meshDir, MESHDIRDEFAULT);
  if (!oapiReadItem_string(f, TEXTUREDIRTAG, textureDir)) strcpy(textureDir, TEXTUREDIRDEFAULT);
  if (!oapiReadItem_string(f, HIGHTEXDIRTAG, hightexDir)) strcpy(hightexDir, HIGHTEXDIRDEFAULT);
  if (!oapiReadItem_string(f, SCENARIODIRTAG, scenarioDir)) strcpy(scenarioDir, SCENARIODIRDEFAULT);
  oapiCloseFile(f, FILE_IN);

  //try standard path
  iniFile=new char[strlen(cn)+strlen(configDir)+strlen(CONFIGPATH)];
  sprintf(iniFile, CONFIGPATH, configDir, cn);
  DWORD attr=GetFileAttributes(iniFile);
  FILE *test=fopen(iniFile, "r");
  if (test==NULL){
   //try alternate vessel config path
   delete [] iniFile;
   iniFile=new char[strlen(cn)+strlen(configDir)+strlen(CONFIG2PATH)];
   sprintf(iniFile, CONFIG2PATH, configDir, cn);
   test=fopen(iniFile, "r");
   if (test==NULL){
    //vessel configuration not found - this is a severe error, as it should be there
    delete [] iniFile;
    iniFile=new char[strlen(cn)*2+strlen(configDir)*2+strlen(CONFIGEXCEPTION)];
    sprintf(iniFile, CONFIGEXCEPTION, configDir, cn, configDir, cn);
    oapiWriteLog(iniFile);
    exit(-1); //Bail out
   }   
  }
  fclose(test); //We don't really need the handle
 }
 //Load the configuration
 sc3.Init(this, iniFile, make_vessel);
 delete [] cn;

 //Attachments
 att_h=new ATTACHMENTHANDLE[sc3.Config.attcnt];
 for(i=0;i<sc3.Config.attcnt;i++)att_h[i]=CreateAttachment(sc3.Data.att[i].ischild!=0,sc3.Data.att[i].pos,sc3.Data.att[i].dir,sc3.Data.att[i].rot,sc3.Data.att[i].id,sc3.Data.att[i].loose!=0);
          
 att_on=0; 
 att_sel=0; 
 att_blank=0;

 //Robotic arm
 arm_grap=-1;
 arm_on=0;
 arm_sel=0;
 arm_blank=0;
 if(sc3.Config.armcnt){  
  arm_grap=sc3.Data.arm[0].grap_seq;
  arm_tip[0]=sc3.Data.arm[0].tip[0];
  arm_tip[1]=sc3.Data.arm[0].tip[1];
  arm_tip[2]=sc3.Data.arm[0].tip[2];    
  xp=arm_tip[1]-arm_tip[0];normalise(xp);
  xr=arm_tip[2]-arm_tip[0];normalise(xr);
  if(sc3.Data.arm[0].grap_attach!=-1)SetAttachmentParams(att_h[sc3.Data.arm[0].grap_attach],arm_tip[0],xp,xr);
 }

 //Base parametres
 SetSize(sc3.Config.size);
 SetEmptyMass(sc3.Config.mass);
 SetPMI(sc3.Config.pmi);
 SetCrossSections(sc3.Config.cross);
 SetRotDrag(sc3.Config.rdrag);
 if(sc3.Config.tdcnt>0)SetTouchdownPoints(sc3.Data.td[0].pta,sc3.Data.td[0].ptb,sc3.Data.td[0].ptc);
                 else SetTouchdownPoints(_V(0,0,1),_V(-1,0,-1),_V(1,0,-1));
 SetSurfaceFrictionCoeff(sc3.Config.frc1,sc3.Config.frc2);
 SetCW(sc3.Config.cw1,sc3.Config.cw2,sc3.Config.cw3,sc3.Config.cw4);
 EnableTransponder(sc3.Config.transponder!=0);
 InitNavRadios(sc3.Config.navrcnt);
 SetCameraOffset(sc3.Config.defcamera);
 
 SetVisibilityLimit(sc3.Config.vislim1,sc3.Config.vislim2);
 SetAlbedoRGB(_V((float)sc3.Config.albedo.r/255,(float)sc3.Config.albedo.g/255,(float)sc3.Config.albedo.b/255));
 SetGravityGradientDamping(sc3.Config.gravgrad);
 SetWingAspect(sc3.Config.wingasp);
 SetWingEffectiveness(sc3.Config.wingeff);
 SetMaxWheelbrakeForce(sc3.Config.maxbrakef);
 SetBankMomentScale(sc3.Config.bankms);
 SetPitchMomentScale(sc3.Config.pitchms);
     
 //Fuel
 ph_h=new PROPELLANT_HANDLE[sc3.Config.propcnt];
 for(i=0;i<sc3.Config.propcnt;i++)ph_h[i]=CreatePropellantResource(sc3.Data.prop[i].mass);

 //Thrusters and groups
 grps=new THRUSTER_HANDLE*[sc3.Config.thgcnt];
 grpc=new int[sc3.Config.thgcnt];
 for(i=0;i<sc3.Config.thgcnt;i++){grps[i]=new THRUSTER_HANDLE[sc3.Config.thcnt];grpc[i]=0;}
 th_h=new THRUSTER_HANDLE[sc3.Config.thcnt];
 thg_h=new THGROUP_HANDLE[sc3.Config.thgcnt]; 

 for(i=0;i<sc3.Config.thcnt;i++){
  th_h[i]=CreateThruster(sc3.Data.th[i].pos,sc3.Data.th[i].dir,sc3.Data.th[i].maxthr,ph_h[sc3.Data.th[i].phid],sc3.Data.th[i].isp,sc3.Data.th[i].ispref,sc3.Data.th[i].pref);
  for(j=0;j<8;j++){
   g=sc3.Data.th[i].grp[j];
   if(g!=9999){
    grps[g][grpc[g]]=th_h[i];
    grpc[g]++;
   }
  }
 }
 for(i=0;i<sc3.Config.thgcnt;i++){
  if(i<15)thg_h[i]=CreateThrusterGroup(grps[i],grpc[i],(THGROUP_TYPE)(THGROUP_MAIN+i));
     else thg_h[i]=CreateThrusterGroup(grps[i],grpc[i],THGROUP_USER);
 }

 //grps and grpc is not needed anymore, as the values were call-by-value in the case of grpc, and stored in th_h in the case of grps, anyway
 delete [] grps;
 delete [] grpc;

 //Exhausts  
 exh_h=new UINT[sc3.Config.exhcnt];
 for(i=0;i<sc3.Config.exhcnt;i++){
  if(sc3.Data.exh[i].tp==0)exh_h[i]=AddExhaust(th_h[sc3.Data.exh[i].th],sc3.Data.exh[i].lscl,sc3.Data.exh[i].wscl);
  if(sc3.Data.exh[i].tp==1)exh_h[i]=AddExhaust(th_h[sc3.Data.exh[i].th],sc3.Data.exh[i].lscl,sc3.Data.exh[i].wscl,sc3.Data.exh[i].pos,sc3.Data.exh[i].rot);
 }

 //Meshes
 msh_idh=new int[sc3.Config.mshcnt];
 msh_h=new MESHHANDLE[sc3.Config.mshcnt];
 for(i=0;i<sc3.Config.mshcnt;i++){
  if(sc3.Data.msh[i].typ==0)SetMeshVisibilityMode(msh_idh[i]=AddMesh(msh_h[i]=oapiLoadMeshGlobal(sc3.Data.msh[i].nam),&sc3.Data.msh[i].off),sc3.Data.msh[i].vis); 
 }

 //Animations
 ani=new genericvessel_animinfo[sc3.Config.anicnt];
 for(i=0;i<sc3.Config.anicnt;i++){
  ani[i]=sc3.Data.anim[i];

  ani[i].grp=new UINT*[sc3.Data.anim[i].grpcnt];
  for(j=0;j<sc3.Data.anim[i].grpcnt;j++){
   ani[i].grp[j]=new UINT[sc3.Data.anim[i].grp[j][0]+1];
   ani[i].grp[j][0]=sc3.Data.anim[i].grp[j][0];
   for(k=1;k<(int)sc3.Data.anim[i].grp[j][0]+1;k++)ani[i].grp[j][k]=sc3.Data.anim[i].grp[j][k];
  }

  ani[i].comp=new genericvessel_animcompinfo[sc3.Data.anim[i].compcnt];
  for(j=0;j<sc3.Data.anim[i].compcnt;j++)ani[i].comp[j]=sc3.Data.anim[i].comp[j];
 }

 n=0;
 for(i=0;i<sc3.Config.anicnt;i++){
  for(j=0;j<ani[i].compcnt;j++){
   if(ani[i].comp[j].id==arm_grap){
    ani[i].comp[j].cmpdef=new MGROUP_ROTATE(LOCALVERTEXLIST,MAKEGROUPARRAY(arm_tip),3,ani[i].comp[j].ref,ani[i].comp[j].axshsc,ani[i].comp[j].angle);
   }else{
    if(ani[i].comp[j].tp==0)ani[i].comp[j].cmpdef=new MGROUP_ROTATE   (ani[i].comp[j].msh,&(ani[i].grp[ani[i].comp[j].grp][1]),ani[i].grp[ani[i].comp[j].grp][0],ani[i].comp[j].ref,ani[i].comp[j].axshsc,ani[i].comp[j].angle);
    if(ani[i].comp[j].tp==1)ani[i].comp[j].cmpdef=new MGROUP_TRANSLATE(ani[i].comp[j].msh,&(ani[i].grp[ani[i].comp[j].grp][1]),ani[i].grp[ani[i].comp[j].grp][0],ani[i].comp[j].axshsc);
    if(ani[i].comp[j].tp==2)ani[i].comp[j].cmpdef=new MGROUP_SCALE    (ani[i].comp[j].msh,&(ani[i].grp[ani[i].comp[j].grp][1]),ani[i].grp[ani[i].comp[j].grp][0],ani[i].comp[j].ref,ani[i].comp[j].axshsc);
   }
  }
  ani[i].id=CreateAnimation(ani[i].init);
  n+=ani[i].compcnt;
 }

 //Zero out components
 for(i=0;i<sc3.Config.anicnt;i++)for(j=0;j<ani[i].compcnt;j++)ani[i].comp[j].cmpt=NULL;

 if(sc3.Config.sorted_anims==0){
  for(i=0;i<sc3.Config.anicnt;i++){
   for(j=0;j<ani[i].compcnt;j++){
    ani[i].comp[j].cmpt=AddAnimationComponent(ani[i].id,ani[i].comp[j].st0,ani[i].comp[j].st1,ani[i].comp[j].cmpdef);
   }
  }
 }else{   
  k=0;  
  for(k=0;k<n;k++){
   for(i=0;i<sc3.Config.anicnt;i++){
    for(j=0;j<ani[i].compcnt;j++){
     if(ani[i].comp[j].id!=k)continue;
     int p=ani[i].comp[j].parent;
     if((p>=0)&(p<n)){
      ANIMATIONCOMPONENT_HANDLE prnt=NULL;
      for(a=0;a<sc3.Config.anicnt;a++){for(b=0;b<ani[a].compcnt;b++)if(ani[a].comp[b].id==p)prnt=ani[a].comp[b].cmpt; }
      ani[i].comp[j].cmpt=AddAnimationComponent(ani[i].id,ani[i].comp[j].st0,ani[i].comp[j].st1,ani[i].comp[j].cmpdef,prnt);
     }else ani[i].comp[j].cmpt=AddAnimationComponent(ani[i].id,ani[i].comp[j].st0,ani[i].comp[j].st1,ani[i].comp[j].cmpdef);
     break;
    }
   }
  }
 }

 //Initialize animations
 for(i=0;i<sc3.Config.anicnt;i++){
  ani[i].status=ANI_BACK_PAUSE;
  if(ani[i].repeat)ani[i].status=ANI_RUN;
 }

 //Beacons
 for(i=0;i<sc3.Config.beaccnt;i++)AddBeacon(&sc3.Data.beac[i].bl);
     
 //Docks
 for(i=0;i<sc3.Config.dockcnt;i++)CreateDock(sc3.Data.dock[i].pos,sc3.Data.dock[i].dir,sc3.Data.dock[i].rot);

 //Airfoils
 for(i=0;i<sc3.Config.aircnt;i++){
  if(sc3.Data.air[i].tp==0)CreateAirfoil3((AIRFOIL_ORIENTATION)sc3.Data.air[i].align,sc3.Data.air[i].v,sc3.Data.air[i].align==LIFT_VERTICAL?VLiftCoeff:HLiftCoeff,0,sc3.Data.air[i].c,sc3.Data.air[i].s,sc3.Data.air[i].a);
  if(sc3.Data.air[i].tp==1)CreateControlSurface2((AIRCTRL_TYPE)sc3.Data.air[i].align,sc3.Data.air[i].s,sc3.Data.air[i].c,sc3.Data.air[i].v,sc3.Data.air[i].axis,sc3.Data.air[i].anim!=-1?ani[sc3.Data.air[i].anim].id:(UINT)-1);
  if(sc3.Data.air[i].tp==2)CreateVariableDragElement(sc3.Data.air[i].anim!=-1?&ani[sc3.Data.air[i].anim].proc:NULL,sc3.Data.air[i].c,sc3.Data.air[i].v);
 }

 //Do the new native API callback now
 sc3.clbkSetClassCaps(cfg);
}
//############################################################################//
bool GenericVessel::clbkLoadVC(int id)
{
 VCHUDSPEC hud;
 VCMFDSPEC mfd;
 int i;

 if(sc3.Config.vcon)if((id>=0)&&(id<sc3.Config.vccnt)){
  SetCameraDefaultDirection(sc3.Data.vc[id].camdir);
  SetCameraOffset(sc3.Data.vc[id].camoff);
  SetCameraShiftRange(sc3.Data.vc[id].camrng[0],sc3.Data.vc[id].camrng[1],sc3.Data.vc[id].camrng[2]);
  oapiVCSetNeighbours(sc3.Data.vc[id].nbhs[0],sc3.Data.vc[id].nbhs[1],sc3.Data.vc[id].nbhs[2],sc3.Data.vc[id].nbhs[3]);

  hud.nmesh=sc3.Data.vc[id].hudmsh;
  hud.ngroup=sc3.Data.vc[id].hudgrp;
  hud.hudcnt=sc3.Data.vc[id].hudoff;
  hud.size=sc3.Data.vc[id].hudsiz;
  oapiVCRegisterHUD(&hud);
  for(i=0;i<sc3.Data.vc[id].mfdcnt;i++){
   mfd.ngroup=sc3.Data.vc[id].mfds[i].ngroup;
   mfd.nmesh=sc3.Data.vc[id].mfds[i].nmesh;
   oapiVCRegisterMFD(sc3.Data.vc[id].mfds[i].tp,&mfd);
  }
 }
 return sc3.Config.vcon==1;
}       
//############################################################################//
//############################################################################//
void GenericVessel::clbkLoadStateEx(FILEHANDLE scn,void *status)
{
 char *line;
 int i,n;

 while(oapiReadScenario_nextline(scn,line)){ 
  #ifndef NOPM  
  if(PM_LoadState(line))continue;  
  #endif

  //Animation states
  if(!strnicmp(line,"SEQ",3)){
   sscanf(line+4,"%d",&i);
   if((i>=0)&&(i<sc3.Config.anicnt)){
    n=0;
    if(i>=10)n=1;
    if(i>=100)n=2;
    if(i>=1000)n=3;
    sscanf(line+6+n,"%d %lf",&ani[i].status,&ani[i].proc);
    SetAnimation(ani[i].id,ani[i].proc);
    if(sc3.Config.tdp_anim==i)if(abs(ani[i].status)==ANI_PAUSE){  
     if(sc3.Config.tdcnt>=1)if(ani[i].proc==0.0)SetTouchdownPoints(sc3.Data.td[0].pta,sc3.Data.td[0].ptb,sc3.Data.td[0].ptc);
     if(sc3.Config.tdcnt>=2)if(ani[i].proc==1.0)SetTouchdownPoints(sc3.Data.td[1].pta,sc3.Data.td[1].ptb,sc3.Data.td[1].ptc);
    }
   }
  }else if(!sc3.clbkLoadStateEx(line)){ //native API
   ParseScenarioLineEx(line,status);
  }
 }
}
//############################################################################//
void GenericVessel::clbkSaveState(FILEHANDLE scn)
{
 char cbuf[256],nbuf[256];
 int i;

 SaveDefaultState(scn);

 //Animation states (not SC3 compatible)
 for(i=0;i<sc3.Config.anicnt;i++){
  sprintf(cbuf,"%d %0.4f",ani[i].status,ani[i].proc);
  sprintf(nbuf,"SEQ %d",i);
  oapiWriteScenario_string(scn,nbuf,cbuf);
 }

 sc3.clbkSaveState(scn);

 #ifndef NOPM  
 baseClass::clbkSaveState(scn);
 #endif
}
//############################################################################//
//############################################################################//
bool GenericVessel::clbkDrawHUD(int mode, const HUDPAINTSPEC *hps,oapi::Sketchpad *skp)
{
 VESSEL3::clbkDrawHUD(mode,hps,skp);
 sc3.RenderHUD(skp);
 return true;
}
//############################################################################//
void GenericVessel::toggle_attach(int point)
{
 VESSEL *v;
 DWORD nAttach,i,j;   
 double dst,min_dst;
 VECTOR3 gpos,grms,pos,dir,rot;
 OBJHANDLE hv,min_hv;
 ATTACHMENTHANDLE tgt_att,min_att;
 //const char *id;

 min_dst=1e50;
 min_att=NULL;
 min_hv=NULL;
 
 hv=GetAttachmentStatus(att_h[point]);

 if(hv){  
  //release
  DetachChild(att_h[point]);
 }else{             
  //grapple
  Local2Global(sc3.Data.att[point].pos,grms);  //global position of RMS tip
  
  //Search the complete vessel list for a grappling candidate.
  for(i=0;i<oapiGetVesselCount();i++){
   hv=oapiGetVesselByIndex(i);
   if(hv==GetHandle())continue; //we don't want to grapple ourselves ...
   oapiGetGlobalPos(hv,&gpos);
   dst=dist(gpos,grms);
   if(dst<oapiGetSize(hv)+oapiGetSize(GetHandle())){ //in range
    v=oapiGetVesselInterface(hv);
    nAttach=v->AttachmentCount(true);
    for(j=0;j<nAttach;j++){ //now scan all attachment points of the candidate
     tgt_att=v->GetAttachmentHandle(true,j); 
     if(v->GetAttachmentStatus(tgt_att))continue;
     //id=v->GetAttachmentId(tgt_att);
     //if(strncmp(id,"GS",2)) continue; //attachment point not compatible
     v->GetAttachmentParams(tgt_att,pos,dir,rot);
     v->Local2Global(pos,gpos);
     dst=dist(gpos,grms);
     if((dst<sc3.Data.att[point].range)&&(dst<min_dst)){
      min_dst=dst;
      min_att=tgt_att;
      min_hv=hv;
     }
    }
   }
  } 
  //Attach closest
  if(min_att)AttachChild(min_hv,att_h[point],min_att);
 }
}
//############################################################################//
int  GenericVessel::clbkConsumeBufferedKey(DWORD key,bool down,char *kstate)
{
 int i;

 if(!down)return 0;

 if(KEYMOD_ALT(kstate)){
  //UMMU EVA handling
  if((key>=OAPI_KEY_1)&&(key<=OAPI_KEY_0)){
   i=(key-OAPI_KEY_1+1)%10;
   sc3.EVA(i); //Takes advantage of OAPI_KEY_1 to OAPI_KEY_0 being linear enumerated
  }
 }else{     
  //Robotic arm handling
  if(sc3.Config.armcnt){
   if(KEYMOD_SHIFT(kstate)){
    if(arm_on)switch(key){
     case OAPI_KEY_NUMPAD4:arm_sel--;if(arm_sel<0)arm_sel=sc3.Config.armcnt-1;break;
     case OAPI_KEY_NUMPAD6:arm_sel++;if(arm_sel>=sc3.Config.armcnt)arm_sel=0;break;
     case OAPI_KEY_NUMPAD0:if(sc3.Data.arm[0].grap_attach!=-1)toggle_attach(sc3.Data.arm[0].grap_attach);break;
    }
   }else switch(key){
    case OAPI_KEY_SPACE:arm_on=!arm_on;arm_blank=1;break;
   }
  } 
  //Attachment handling
  if(sc3.Config.attcnt){
   if(KEYMOD_SHIFT(kstate)){
    if(att_on)switch(key){
     case OAPI_KEY_NUMPAD4:att_sel--;if(att_sel<0)att_sel=sc3.Config.attcnt-1;break;
     case OAPI_KEY_NUMPAD6:att_sel++;if(att_sel>=sc3.Config.attcnt)att_sel=0;break;
     case OAPI_KEY_NUMPAD0:toggle_attach(att_sel);break;
    }
   }else switch(key){
    case OAPI_KEY_A:att_on=!att_on;att_blank=1;break;
   }
  }
  //Payload jettisoning
  if(key==OAPI_KEY_J)sc3.Jettison();


  //Animation handling
  for(i=0;i<sc3.Config.anicnt;i++)if(ani[i].tp==0){
   DWORD mod=ani[i].trigkey >> 8;
   if(!mod | KEYDOWN(kstate, mod))if(key==(ani[i].trigkey & 0xFF)){
    if(ani[i].repeat!=0){ 
     switch(ani[i].status){
      case ANI_RUN:ani[i].status=ANI_PAUSE;break;
      case ANI_PAUSE:ani[i].status=ANI_RUN;break;
      default:ani[i].status=ANI_RUN;
     }
    }else if(ani[i].pause){ 
     if(KEYMOD_LCONTROL(kstate)){
      switch(ani[i].status){
       case ANI_RUN       :ani[i].status=ANI_PAUSE;break;
       case ANI_BACK_RUN  :ani[i].status=ANI_PAUSE;break;
       case ANI_BACK_PAUSE:ani[i].status=ANI_BACK_RUN;break;
       case ANI_PAUSE     :ani[i].status=ANI_BACK_RUN;break;
       default:ani[i].status=ANI_PAUSE;
      } 
     }else{
      switch(ani[i].status){
       case ANI_RUN       :ani[i].status=ANI_PAUSE;break;
       case ANI_BACK_RUN  :ani[i].status=ANI_PAUSE;break;
       case ANI_BACK_PAUSE:ani[i].status=ANI_RUN;break;
       case ANI_PAUSE     :ani[i].status=ANI_RUN;break;
       default:ani[i].status=ANI_PAUSE;
      } 
     }
    }else{   
     switch(ani[i].status){
      case ANI_RUN       :ani[i].status=ANI_BACK_RUN;break;
      case ANI_BACK_RUN  :ani[i].status=ANI_RUN;break;
      case ANI_BACK_PAUSE:ani[i].status=ANI_RUN;break;
      case ANI_PAUSE     :ani[i].status=ANI_BACK_RUN;break;
      default:ani[i].status=ANI_PAUSE;
     } 
    }
   }
  }
 }
 return 0;
}   
//#######################################################################################//
int GenericVessel::clbkConsumeDirectKey(char *kstate)
{     
 int i;
 for(i=0;i<sc3.Config.armcnt;i++)ani[sc3.Data.arm[i].seq].status=0;   

 //Robotic arm
 if(arm_on){
  if(KEYMOD_SHIFT(kstate)){
   if(KEYDOWN(kstate,OAPI_KEY_NUMPAD2)){
    ani[sc3.Data.arm[arm_sel].seq].status=1;
    RESETKEY(kstate,OAPI_KEY_NUMPAD8);
   }
   if(KEYDOWN(kstate,OAPI_KEY_NUMPAD8)){
    ani[sc3.Data.arm[arm_sel].seq].status=-1;
    RESETKEY(kstate,OAPI_KEY_NUMPAD8);
   }
  }  
 }
 return 0;
}    
//#######################################################################################//
//#######################################################################################//  
void GenericVessel::clbkPreStep(double simt,double simdt,double mjd)
{
}
//#######################################################################################//  
void GenericVessel::clbkPostStep(double simt,double simdt,double mjd)
{     
 int i;
 double pos;

 //Animations
 for(i=0;i<sc3.Config.anicnt;i++)if((ani[i].status==ANI_RUN)||(ani[i].status==ANI_BACK_RUN)){
  double da=simdt*ani[i].spd;

  if(ani[i].repeat==1){
   ani[i].proc=fmod(ani[i].proc+da,1);
  }else if(ani[i].repeat==2){
   if(ani[i].status==ANI_RUN){
    if(ani[i].proc+da>1.0){ani[i].status=ANI_BACK_RUN;ani[i].proc=1.0;}else ani[i].proc+=da;
   }else if(ani[i].status==ANI_BACK_RUN){
    if(ani[i].proc-da<0.0){ani[i].status=ANI_RUN;ani[i].proc=0;}else ani[i].proc-=da;
   }
  }else{
   if(ani[i].status==ANI_BACK_RUN){ 
    if(ani[i].proc>0.0)ani[i].proc=max(0.0,ani[i].proc-da);
    else               ani[i].status=ANI_BACK_PAUSE;
   }else{
    if(ani[i].proc<1.0)ani[i].proc=min(1.0,ani[i].proc+da);
    else               ani[i].status=ANI_PAUSE;
   }
  }
  SetAnimation(ani[i].id,ani[i].proc);
  if(sc3.Config.tdp_anim==i)if(abs(ani[i].status)==ANI_PAUSE){
   if(sc3.Config.tdcnt>=1)if(ani[i].proc==0.0)SetTouchdownPoints(sc3.Data.td[0].pta,sc3.Data.td[0].ptb,sc3.Data.td[0].ptc);
   if(sc3.Config.tdcnt>=2)if(ani[i].proc==1.0)SetTouchdownPoints(sc3.Data.td[1].pta,sc3.Data.td[1].ptb,sc3.Data.td[1].ptc);
  }
 }

 //Robotic arm
 if(arm_on)if((arm_sel>=0)&&(arm_sel<sc3.Config.armcnt)){
  pos=ani[sc3.Data.arm[arm_sel].seq].proc;
  pos=sc3.Data.arm[arm_sel].range_low+pos*(sc3.Data.arm[arm_sel].range_high-sc3.Data.arm[arm_sel].range_low);
  VECTOR3 xr,xp;
  xp=arm_tip[1]-arm_tip[0];normalise(xp);
  xr=arm_tip[2]-arm_tip[0];normalise(xr);
  if(sc3.Data.arm[0].grap_attach!=-1){
   SetAttachmentParams(att_h[sc3.Data.arm[0].grap_attach],arm_tip[0],xp,xr);
   sc3.Data.att[sc3.Data.arm[0].grap_attach].pos=arm_tip[0];
   sc3.Data.att[sc3.Data.arm[0].grap_attach].dir=xp;
   sc3.Data.att[sc3.Data.arm[0].grap_attach].rot=xr;
  }
  sprintf(oapiDebugString(),"Robotic Arm: %s (%.3f)",sc3.Data.arm[arm_sel].name,pos);
 }

 //Attachments
 if(att_on)if((att_sel>=0)&&(att_sel<sc3.Config.attcnt)){
  sprintf(oapiDebugString(),"Attach:%s:%s",sc3.Data.att[att_sel].name,sc3.Data.att[att_sel].name);
 }

 if(arm_blank||att_blank){sprintf(oapiDebugString(),"");att_blank=0;arm_blank=0;}

 //Do the new native API callback now
 sc3.clbkPostStep(simt, simdt, mjd);
}
//#######################################################################################//        
//#######################################################################################//   
bool GenericVessel::do_reload()
{
 do_clean();
 do_init();

 return TRUE;
}
//#######################################################################################// 
//#######################################################################################//  
void reload_vessel()
{
 VESSEL *ves=oapiGetVesselInterface(oapiGetFocusObject());
 //if (ves is GenericVessel) {
  ((GenericVessel*)ves)->do_reload();
 //}
}
//#######################################################################################//
int myprm = 0;   
//#######################################################################################//
void CloseDlg(HWND hDlg)
{
 oapiCloseDialog(hDlg);
}
//#######################################################################################//
BOOL CALLBACK MsgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
 char name[256];

 switch(uMsg){
  case WM_INITDIALOG:
   sprintf(name,"%d",myprm);
   //SetWindowText(GetDlgItem(hDlg,IDC_REMEMBER),name);
   return TRUE;

  case WM_DESTROY:
   return TRUE;

  case WM_COMMAND:
   switch (LOWORD (wParam)) {

   case ID_RELOAD:
    reload_vessel();
    return TRUE;

   case IDCANCEL: //dialog closed by user
    CloseDlg (hDlg);
    return TRUE;
   }
   break;
 }
 return oapiDefDialogProc (hDlg, uMsg, wParam, lParam);
}         
//#######################################################################################//
void OpenDlgClbk(void *context)
{
 HWND hDlg=oapiOpenDialog(g_hInst,IDD_MYDIALOG,MsgProc);
}                          
//#######################################################################################//
//Link with xves.dll
void config_fromsc3(HINSTANCE hModule)
{
 inited=0;

 hlib=LoadLibrary("modules\\xves.dll");
 if(hlib){
  make_vessel    =(void(__stdcall *)(DWORD,DWORD,char*,char*))GetProcAddress(hlib,"make_vessel");
  if(make_vessel==NULL)MessageBox(NULL,"xves interface error. Check installation for version mix-up.","Error!",MB_OK);
 }

 inited=(make_vessel!=NULL);
 return;
}     
//#######################################################################################//
DLLCLBK void InitModule(HINSTANCE hModule)
{
 config_fromsc3(hModule);
 g_hInst=hModule;
 g_dwCmd=oapiRegisterCustomCmd("GenericVessel editor","GenericVessel editor",OpenDlgClbk, NULL);
}
//############################################################################//
//Free the dll on exit
DLLCLBK void ExitModule(HINSTANCE hModule)
{
 if(hlib)FreeLibrary(hlib);
 make_vessel=NULL;
 oapiUnregisterCustomCmd(g_dwCmd);
}   
//############################################################################//
DLLCLBK VESSEL *ovcInit(OBJHANDLE hvessel,int flightmodel){if(inited)return new GenericVessel(hvessel,flightmodel);else return NULL;}
DLLCLBK void ovcExit(VESSEL *vessel){if(vessel)delete(GenericVessel*)vessel;}
//############################################################################//

