//############################################################################//  
// Copyright (C) 2010 Friedrich Kastner-Masilko
// SC3Data.cpp
// SC3 data encapsulation class.
//############################################################################//
#pragma once
#include "SC3Data.h"
#define MAXLEN 4096
#define VECTORFORMAT "(%lf,%lf,%lf)"
#define PAYLOADSECTION "PAYLOAD_%d"
#define UMMUSECTION "UMMU"

#define PAYLOADSECTIONLEN 18

namespace SC3Data
{
	// Helper function xargs converts a list in a string into a character array vector
	// The function allocates the memory for the caller.
	// Return value is the array count (at least one).
	int xargs(char *data, char separator, vector<char *> *dest)
	{
		int count=0;
		char *d=data, *copy;		
		while(*d!=0x00)
		{
			if (*(d++)!=separator) continue;
			*(d-1)=0x00;
			strcpy(copy=new char[strlen(data)+1], data);
			dest->push_back(copy);
			data=d;
			count+=1;
		}
		strcpy(copy=new char[strlen(data)+1], data);
		dest->push_back(copy);
		return ++count;
	}

	Vessel::~Vessel()
	{
		for(vector<Payload*>::iterator i=payloads.begin();i!=payloads.end();i++) delete *i;
		delete crew;
	}

	void Vessel::Init(VESSEL *owner, char *filename, void (__stdcall *make_vessel)(DWORD, DWORD, char *, char *))
	{
		make_vessel((DWORD)(&Config),(DWORD)(&Data),filename, owner->GetName());
		//Decode payloads
		for(int i=0;i<INT_MAX;i++)
		{
			Payload *payload=new Payload(owner);
			if (payload->Init(filename, i))	payloads.push_back(payload);
			else {delete payload; break;}
		}
		//Decode UMmu
		crew=new UMmu(owner);
		if (!crew->Init(filename))
		{
			delete crew;
			crew=NULL;
		}
	}

	void Vessel::clbkSetClassCaps(FILEHANDLE cfg)
	{
		for(vector<Payload*>::iterator i=payloads.begin();i!=payloads.end();i++) (*i)->clbkSetClassCaps(cfg);
	}

	void Vessel::clbkPostStep(double simt,double simdt,double mjd){if (crew) crew->clbkPostStep(simt, simdt, mjd);}

	bool Vessel::clbkLoadStateEx (char *line)
	{
		if (!strnicmp(line,"CURRENT_PAYLOAD",15))
		{
			sscanf(line+15,"%d",&currentPayload);
			currentPayload=min(payloads.size()+1, currentPayload);
			for(int i=0;i<currentPayload;i++) payloads[i]->Clear();
			return true;
		}
		return crew?crew->clbkLoadStateEx(line):false;
	}

	void Vessel::clbkSaveState(FILEHANDLE scn)
	{
		char cbuf[10];
		sprintf(cbuf,"%d",currentPayload);
		oapiWriteScenario_string(scn,"CURRENT_PAYLOAD",cbuf);
		if (crew) crew->clbkSaveState(scn);
	}

	void Vessel::Jettison(){if (currentPayload<payloads.size()) payloads[currentPayload++]->Eject();}
	void Vessel::EVA(int slot){if (crew) crew->EVA(slot);}

	void Vessel::RenderHUD(oapi::Sketchpad *skp){if (crew) crew->RenderHUD(skp);}

	Payload::Payload(VESSEL *owner)
	{
		this->owner=owner;
		module=NULL;
		name=NULL;
	}

	Payload::~Payload()
	{
		for(vector<char *>::iterator i=meshname.begin();i!=meshname.end();i++) delete [] *i;
		delete [] module;
		delete [] name;
	}

	bool Payload::Init(char *ini, int index)
	{
		char section[PAYLOADSECTIONLEN], data[MAXLEN];
		sprintf(section, PAYLOADSECTION, index);

		//Check for meshname for valid payload section, and parse it
		GetPrivateProfileString(section, "MESHNAME", NULL, data, MAXLEN, ini);
		if (data==NULL || strlen(data)==0) return false;
		int meshes=xargs(data, ';', &meshname);
		
		//Reading and parsing offsets
		GetPrivateProfileString(section, "OFF", "(0,0,0)", data, MAXLEN, ini);
		vector<char *> offs;
		xargs(data, ';', &offs);
		for (vector<char *>::iterator i=offs.begin();i!=offs.end();i++,meshes--)
		{
			VECTOR3 v;
			sscanf(*i, VECTORFORMAT, &v.x, &v.y, &v.z);
			off.push_back(v);
			delete [] *i;
		}
		while (meshes-->0) off.push_back(_V(0,0,0));
		
		//Getting the simple values
		GetPrivateProfileString(section, "NAME", meshname[0], data, MAXLEN, ini);
		strcpy(name=new char[strlen(data)+1], data);
		GetPrivateProfileString(section, "MODULE", "Spacecraft\\Spacecraft3", data, MAXLEN, ini);
		strcpy(module=new char[strlen(data)+1], data);
		GetPrivateProfileString(section, "MASS", "0", data, MAXLEN, ini);
		mass=atof(data);
		GetPrivateProfileString(section, "SPEED", "(0,0,0)", data, MAXLEN, ini);
		sscanf(data, VECTORFORMAT, &speed.x, &speed.y, &speed.z);
		GetPrivateProfileString(section, "ROT_SPEED", "(0,0,0)", data, MAXLEN, ini);
		sscanf(data, VECTORFORMAT, &rotation.x, &rotation.y, &rotation.z);

		return true;
	}

	void Payload::clbkSetClassCaps(FILEHANDLE cfg)
	{
		//Load all meshes for the payload
		meshIndex.clear();
		for(int i=0;i<meshname.size();i++)
		{
			int index=owner->AddMesh(oapiLoadMeshGlobal(meshname[i]),&off[i]);
			meshIndex.push_back(index);
			owner->SetMeshVisibilityMode(index,MESHVIS_ALWAYS);
		}
		//Add the masses to the vessel
		owner->SetEmptyMass(owner->GetEmptyMass()+mass);
	}

	void Payload::Eject(void)
	{
		VESSELSTATUS2 status;
		status.flag=0;
		status.version=2;
		owner->GetStatusEx(&status);
		VECTOR3 global;
		owner->GlobalRot(off[0], global); //Using first offset as vessel-relative position
		status.rpos+=global;
		owner->GlobalRot(speed, global); //Speed is also vessel-relative
		status.rvel+=global;
		status.vrot+=rotation; //Rotation should be vessel independent
		status.status=0;
		status.port=0;
		status.dockinfo=NULL;
		status.base=NULL;
		oapiCreateVesselEx(name, module, &status);
		Clear();
	}

	void Payload::Clear(void)
	{
		for(int i=0;i<meshIndex.size();i++) owner->DelMesh(meshIndex[i]); //Delete the meshes
		owner->SetEmptyMass(owner->GetEmptyMass()-mass); //Remove the mass again
	}

	UMmu::UMmu(VESSEL *owner)
	{
		this->owner=owner;
	}

	UMmu::~UMmu(){}

	bool UMmu::Init(char *ini)
	{
		char data[MAXLEN];

		//Check for seat definition for valid UMmu section, and parse it
		GetPrivateProfileString(UMMUSECTION, "MAXSEATS", NULL, data, MAXLEN, ini);
		if (data==NULL || strlen(data)==0) return false;
		maxSeatsAvailable=atoi(data);
		
		if(maxSeatsAvailable!=0)
		{
			ummuSdKInitReturnCode=crew.InitUmmu(owner->GetHandle()); 
			float ummuVersion=crew.GetUserUMmuVersion();
			crew.SetMaxSeatAvailableInShip(maxSeatsAvailable);

			VECTOR3 pos, size;
			GetPrivateProfileString(UMMUSECTION, "AIRLOCK_POSITION", "(0,0,0)", data, MAXLEN, ini);
			sscanf(data, VECTORFORMAT, &pos.x, &pos.y, &pos.z);
			GetPrivateProfileString(UMMUSECTION, "AIRLOCK_SIZE", "(1,1,1)", data, MAXLEN, ini);
			sscanf(data, VECTORFORMAT, &size.x, &size.y, &size.z);
			size/=2;
			VECTOR3 rot=pos-size;
			pos+=size;
			if (rot.x>pos.x) size.x=rot.x,rot.x=pos.x,pos.x=size.x;
			if (rot.y>pos.y) size.y=rot.y,rot.y=pos.y,pos.y=size.y;
			if (rot.z>pos.z) size.z=rot.z,rot.z=pos.z,pos.z=size.z;
			crew.DefineAirLockShape(TRUE, (float)rot.x, (float)pos.x, (float)rot.y, (float)pos.y, (float)rot.z, (float)pos.z);

			GetPrivateProfileString(UMMUSECTION, "EVA_POS", "(0,0,0)", data, MAXLEN, ini);
			sscanf(data, VECTORFORMAT, &pos.x, &pos.y, &pos.z);
			GetPrivateProfileString(UMMUSECTION, "EVA_ROT", "(0,0,0)", data, MAXLEN, ini);
			sscanf(data, VECTORFORMAT, &rot.x, &rot.y, &rot.z);
			crew.SetMembersPosRotOnEVA(pos, rot);
		}

		return true;
	}

	void UMmu::clbkPostStep(double simt,double simdt,double mjd)
	{
		crew.ProcessUniversalMMu();
		crew.WarnUserUMMUNotInstalled("GenericVessel");
	}

	bool UMmu::clbkLoadStateEx(char *line){return crew.LoadAllMembersFromOrbiterScenario(line);}
	void UMmu::clbkSaveState(FILEHANDLE scn){crew.SaveAllMembersInOrbiterScenarios(scn);}
	void UMmu::EVA(int slot){if (slot<maxSeatsAvailable) crew.EvaCrewMember(crew.GetCrewNameBySlotNumber(slot));}

	void UMmu::RenderHUD(oapi::Sketchpad *skp)
	{
		char s[255];
		int cx,cy,l;
		cx=5;
		cy=150;
		l=0;

		if(maxSeatsAvailable)
		{
			sprintf(s,"Crew limit %d",maxSeatsAvailable);
			skp->Text(cx,cy+18*l++,s,strlen(s));

			int onBoard=crew.GetCrewTotalNumber();
			sprintf(s,"Crew on board %d",onBoard);
			skp->Text(cx,cy+18*l++,s,strlen(s));
			sprintf(s,"Alt+num to kick out");
			skp->Text(cx,cy+18*l++,s,strlen(s));
			for(int i=0;i<onBoard;i++)
			{
				sprintf(s,"%d. %s",i,crew.GetCrewNameBySlotNumber(i));
				skp->Text(cx,cy+18*l++,s,strlen(s));
			}
		}
	}
}