
package body Vector_Of_Bytes is

   procedure Finalize (Object : in out Vector_Type) is
      This : Vector_Access := Object'Unchecked_Access;
      Released : Boolean;
   begin
      if Object.RB /= null then
         if Is_Unique(Object.RB.all) then
            Free(Object.RB.Elements);
            Free(Object.RB);
         else
            Release(Object.RB.all, Released);
         end if;
      end if;
      begin
         Free(This);
      exception
         when others => null;
      end;
   end Finalize;

   procedure Copy (Target : in out Vector_Type; Source : Vector_Type)
   is
      Released : Boolean;
   begin
      if Source.RB = null then
         Raise_Exception(Source_Is_Empty'Identity,
                         "Source for Copy command is Empty!");
      end if;
      if Target.RB = Source.RB then
         return;
      end if;
      if Target.RB /= null then
         if Is_Unique(Target.RB.all) then
            Free (Target.RB.Elements);
            Free (Target.RB);
         else
            Release (Target.RB.all, Released);
         end if;
      end if;
      Target.RB := Source.RB;
      Retain (Target.RB.all);
   end Copy;

   procedure Move (Target, Source : in out Vector_Type)
   is
      Released : Boolean;
      Source_Ptr : Vector_Access := Source'Unchecked_Access;
   begin
      if Source.RB = null then
         Raise_Exception(Source_Is_Empty'Identity,
                         "Source for Move command is Empty!");
      end if;
      if Target.RB = Source.RB then
         Source.RB := null;
         Source.Last := No_Index;
         Source.BS := 0;
         return;
      end if;
      if Target.RB /= null then
         if Is_Unique(Target.RB.all) then
            Free (Target.RB.Elements);
            Free (Target.RB);
         else
            Release (Target.RB.all, Released);
         end if;
      end if;
      Target.RB := Source.RB;
      Target.Last := Source.Last;
      Target.BS := Source.BS;
      Source.RB := null;
      Source.Last := No_Index;
      Source.BS := 0;
   end Move;

   function Is_Empty (Vector : Vector_Type) return Boolean
   is
   begin
      return Vector.RB = null;
   end Is_Empty;

   procedure Clean (Vector : in out Vector_Type; Unallocate : Boolean := False)
   is
      Released : Boolean;
      Vector_Ptr : Vector_Access := Vector'Unchecked_Access;
   begin
      if Vector.RB = null then
         return;
      end if;
      if Is_Unique(Vector.RB.all) then
         Free(Vector.RB.Elements);
         Free(Vector.RB);
      else
         Release(Vector.RB.all, Released);
      end if;
      Vector.RB := null;
      Vector.Last := No_Index;
      Vector.BS := 0;
      if Unallocate then
         declare
         begin
            Free(Vector_Ptr);
         exception
            when others => null;
         end;
      end if;
   end Clean;

   function Length (Vector : Vector_Type) return Count_Type
   is
   begin
      if Vector.RB = null then
         return 0;
      end if;
      return Count_Type(Vector.Last - Index_Type'First + 1);
   end Length;

   procedure Create (Vector : in out Vector_Type;
                     Source : Array_Of_Byte_Type)
   is
      Source_Length : Count_Type := Source'Length;
      Released : Boolean;
   begin
      if Source_Length < 1 then
         Raise_Exception(Source_Is_Empty'Identity,
                         "Source array for Create command is Empty!");
      end if;
      if Vector.RB /= null then
         if Is_Unique(Vector.RB.all) then
            if Vector.BS >= Source_Length then
               declare
                  subtype Buffer_Type is Array_Of_Byte_Type(Source'Range);
                  type Buffer_Access is access all Buffer_Type;
                  function To_Pointer is
                    new Ada.Unchecked_Conversion
                      (System.Address, Buffer_Access);
                  Target_Buffer : Buffer_Access :=
                    To_Pointer (Vector.RB.Elements.all'Address);
               begin
                  Target_Buffer.all := Source;
                  Vector.Last := Source'Last;
                  return;
               end;
            else
               Free(Vector.RB.Elements);
               Free(Vector.RB);
               Vector.RB := null;
               Vector.Last := No_Index;
               Vector.BS := 0;
            end if;
         else
            Release(Vector.RB.all, Released);
            Vector.RB := null;
            Vector.Last := No_Index;
            Vector.BS := 0;
         end if;
      end if;
      Vector.RB := new Referenced_Buffer;
      declare
         Target_Buffer : Array_Of_Byte_Access :=
           new Array_Of_Byte_Type
             (Index_Type'First .. Source'Last - Source'First + Size_Delta);
         Target_Index : Index_Type;
      begin
         for I in Source'Range loop
            Target_Index := Index_Type'First + (I - Source'First) + 1;
            Target_Buffer.all(Target_Index) := Source(I);
         end loop;
         Vector.RB.Elements :=
           To_Pointer(Target_Buffer.all(Target_Buffer.all'First)'Address);
         Vector.BS := Target_Buffer.all'Length;
      end;
      Vector.Last := Source'Last;
   end Create;

   procedure Append (Vector : in out Vector_Type;
                     Source : Element_Type;
                     Index  : out Index_Type)
   is
      Released : Boolean;
   begin
      if Vector.RB /= null then
         if Is_Unique(Vector.RB.all) then
            if Vector.BS > Length(Vector) then
               Vector.Last := Vector.Last + 1;
               declare
                  subtype Target_Array_Type is
                    Array_Of_Byte_Type(Index_Type'First .. Vector.Last);
                  type Target_Array_Access is access all Target_Array_Type;
                  function To_Pointer is
                    new Ada.Unchecked_Conversion
                      (System.Address, Target_Array_Access);
                  Target_Array : Target_Array_Access :=
                    To_Pointer(Vector.RB.Elements.all'Address);
               begin
                  Target_Array.all(Vector.Last) := Source;
               end;
               Index := Vector.Last;
               return;
            end if;
            declare
               Target_Buffer : Array_Of_Byte_Access :=
                 new Array_Of_Byte_Type
                   (Index_Type'First .. Vector.Last + Size_Delta);
               subtype Target_Array_Type is
                 Array_Of_Byte_Type(Index_Type'First .. Vector.Last);
               type Target_Array_Access is access all Target_Array_Type;
               function To_Pointer is
                 new Ada.Unchecked_Conversion
                   (System.Address, Target_Array_Access);
               Target_Array : Target_Array_Access :=
                 To_Pointer(Vector.RB.Elements.all'Address);
            begin
               for I in Target_Array.all'Range loop
                  Target_Buffer.all(I) := Target_Array.all(I);
               end loop;
               Vector.Last := Vector.Last + 1;
               Target_Buffer.all(Vector.Last) := Source;
               Free(Vector.RB.Elements);
               Vector.RB.Elements :=
                 To_Pointer(Target_Buffer.all(Index_Type'First)'Address);
               Vector.BS := Target_Buffer.all'Length;
            end;
            Index := Vector.Last;
            return;
         end if;
         declare
            Target_Buffer : Array_Of_Byte_Access :=
              new Array_Of_Byte_Type
                (Index_Type'First .. Vector.Last + Size_Delta);
            subtype Target_Array_Type is
              Array_Of_Byte_Type(Index_Type'First .. Vector.Last);
            type Target_Array_Access is access all Target_Array_Type;
            function To_Pointer is
              new Ada.Unchecked_Conversion
                (System.Address, Target_Array_Access);
            Target_Array : Target_Array_Access :=
              To_Pointer(Vector.RB.Elements.all'Address);
         begin
            for I in Target_Array.all'Range loop
               Target_Buffer.all(I) := Target_Array.all(I);
            end loop;
            Vector.Last := Vector.Last + 1;
            Target_Buffer.all(Vector.Last) := Source;
            Release(Vector.RB.all, Released);
            Vector.RB := new Referenced_Buffer;
            Vector.RB.Elements :=
              To_Pointer(Target_Buffer.all(Index_Type'First)'Address);
            Vector.BS := Target_Buffer.all'Length;
         end;
         Index := Vector.Last;
         return;
      end if;
      Vector.Last := Index_Type'First;
      declare
         Target_Buffer : Array_Of_Byte_Access :=
           new Array_Of_Byte_Type
             (Index_Type'First .. Vector.Last + Size_Delta);
      begin
         Target_Buffer.all(Vector.Last) := Source;
         Vector.RB := new Referenced_Buffer;
         Vector.RB.Elements :=
           To_Pointer(Target_Buffer.all(Index_Type'First)'Address);
         Vector.BS := Target_Buffer.all'Length;
      end;
      Index := Vector.Last;
   end Append;

   procedure Append (Vector : in out Vector_Type;
                     Source : Array_Of_Byte_Type)
   is
      Source_Length : Count_Type := Source'Length;
      Old_Length : Count_Type := Length(Vector);
      New_Length : Count_Type := Old_Length + Source_Length;
      Released : Boolean;
   begin
      if Source_Length < 1 then
         Raise_Exception(Source_Is_Empty'Identity,
                         "Source array for Append command is Empty!");
      end if;
      if Vector.RB /= null then
         if Is_Unique(Vector.RB.all) then
            if Vector.BS >= New_Length then
               declare
                  subtype Target_Array_Type is Array_Of_Byte_Type
                    (Index_Type'First ..
                       Index_Type'First + Index_Type(New_Length - 1));
                  type Target_Array_Access is access all Target_Array_Type;
                  function To_Pointer is
                    new Ada.Unchecked_Conversion
                      (System.Address, Target_Array_Access);
                  Target_Array_Ptr : Target_Array_Access :=
                    To_Pointer(Vector.RB.Elements.all'Address);
                  New_Index : Index_Type;
               begin
                  for I in Source'Range loop
                     New_Index := Vector.Last + (I - Source'First) + 1;
                     Target_Array_Ptr.all(New_Index) := Source(I);
                  end loop;
               end;
               Vector.Last := Vector.Last + Index_Type(Source_Length);
               return;
            end if;
            declare
               New_Buffer : Array_Of_Byte_Access :=
                 new Array_Of_Byte_Type
                   (Index_Type'First ..
                      Index_Type'First + Index_Type(New_Length + Size_Delta));
               subtype Old_Buffer_Type is Array_Of_Byte_Type
                 (Index_Type'First .. Vector.Last);
               type Old_Buffer_Access is access Old_Buffer_Type;
               function To_Pointer is
                 new Ada.Unchecked_Conversion
                   (System.Address, Old_Buffer_Access);
               Old_Buffer_Ptr : Old_Buffer_Access :=
                 To_Pointer(Vector.RB.Elements.all'Address);
               New_Index : Index_Type;
            begin
               for I in Old_Buffer_Ptr.all'Range loop
                  New_Buffer.all(I) := Old_Buffer_Ptr.all(I);
               end loop;
               for I in Source'Range loop
                  New_Index := Vector.Last + (I - Source'First + 1);
                  New_Buffer.all(New_Index) := Source(I);
               end loop;
               Free(Vector.RB.Elements);
               Vector.RB.Elements :=
                 To_Pointer(New_Buffer.all(Index_Type'First)'Address);
               Vector.BS := New_Buffer.all'Length;
            end;
            Vector.Last := Vector.Last + Index_Type(Source_Length);
            return;
         end if;
         Release(Vector.RB.all, Released);
         declare
            New_Buffer : Array_Of_Byte_Access :=
              new Array_Of_Byte_Type
                (Index_Type'First ..
                   Index_Type'First + Index_Type(New_Length + Size_Delta));
            subtype Old_Buffer_Type is Array_Of_Byte_Type
              (Index_Type'First .. Vector.Last);
            type Old_Buffer_Access is access Old_Buffer_Type;
            function To_Pointer is
              new Ada.Unchecked_Conversion
                (System.Address, Old_Buffer_Access);
            Old_Buffer_Ptr : Old_Buffer_Access :=
              To_Pointer(Vector.RB.Elements.all'Address);
            New_Index : Index_Type;
         begin
            for I in Old_Buffer_Ptr.all'Range loop
               New_Buffer.all(I) := Old_Buffer_Ptr.all(I);
            end loop;
            for I in Source'Range loop
               New_Index := Vector.Last + (I - Source'First + 1);
               New_Buffer.all(New_Index) := Source(I);
            end loop;
            Vector.RB := new Referenced_Buffer;
            Vector.RB.Elements :=
              To_Pointer(New_Buffer.all(Index_Type'First)'Address);
            Vector.BS := New_Buffer.all'Length;
         end;
         Vector.Last := Vector.Last + Index_Type(Source_Length);
         return;
      end if;
      Create(Vector, Source);
   end Append;

   procedure Append (Vector : in out Vector_Type;
                     Source : in out Vector_Type)
   is
      Source_Length : Count_Type := Length(Source);
      Old_Length : Count_Type := Length(Vector);
      New_Length : Count_Type := Old_Length + Source_Length;
      Vector_Ptr : Vector_Access := Vector'Unchecked_Access;
      Source_Ptr : Vector_Access := Source'Unchecked_Access;
      Released : Boolean := False;
   begin
      if Source_Length < 1 then
         Raise_Exception(Source_Is_Empty'Identity,
                         "Source vector for Append command is Empty!");
      end if;
      if Old_Length /= 0 then
         if Is_Unique(Vector.RB.all) then
            if Vector.RB = Source.RB then
               if Vector_Ptr /= Source_Ptr then
                  Vector.RB := null;
                  Vector.BS := 0;
                  Vector.Last := No_Index;
                  declare
                     subtype Source_Array_Type is
                       Array_Of_Byte_Type(Index_Type'First .. Source.Last);
                     type Source_Array_Access is access all Source_Array_Type;
                     function To_Pointer is
                       new Ada.Unchecked_Conversion
                         (System.Address, Source_Array_Access);
                     Source_Array_Ptr : Source_Array_Access :=
                       To_Pointer(Source.RB.Elements.all'Address);
                  begin
                     Create(Vector, Source_Array_Ptr.all);
                     Append(Vector, Source_Array_Ptr.all);
                  end;
                  return;
               end if;
            end if;
            declare
               subtype Source_Array_Type is
                 Array_Of_Byte_Type(Index_Type'First .. Source.Last);
               type Source_Array_Access is access all Source_Array_Type;
               function To_Pointer is
                 new Ada.Unchecked_Conversion
                   (System.Address, Source_Array_Access);
               Source_Array_Ptr : Source_Array_Access :=
                 To_Pointer(Source.RB.Elements.all'Address);
            begin
               Append(Vector, Source_Array_Ptr.all);
            end;
            return;
         end if;
         Release(Vector.RB.all, Released);
         declare
            subtype Target_Array_Type is
              Array_Of_Byte_Type(Index_Type'First .. Vector.Last);
            type Target_Array_Access is access all Target_Array_Type;
            function To_Pointer is
              new Ada.Unchecked_Conversion
                (System.Address, Target_Array_Access);
            Target_Array_Ptr : Target_Array_Access :=
              To_Pointer(Vector.RB.Elements.all'Address);
            subtype Source_Array_Type is
              Array_Of_Byte_Type(Index_Type'First .. Source.Last);
            type Source_Array_Access is access all Source_Array_Type;
            function To_Pointer is
              new Ada.Unchecked_Conversion
                (System.Address, Source_Array_Access);
            Source_Array_Ptr : Source_Array_Access :=
              To_Pointer(Source.RB.Elements.all'Address);
         begin
            Vector.RB := null;
            Vector.Last := No_Index;
            Vector.BS := 0;
            Create(Vector, Target_Array_Ptr.all);
            Append(Vector, Source_Array_Ptr.all);
         end;
         return;
      end if;
      Copy(Vector, Source);
   end Append;

   function "=" (Left, Right : Vector_Type) return Boolean
   is
   begin
      if Left.RB = Right.RB then
         return True;
      end if;
      if Length(Left) /= Length(Right) then
         return False;
      end if;
      declare
         subtype Array_Type is Array_Of_Byte_Type
           (Index_Type'First .. Left.Last);
         type Array_Access is access all Array_Type;
         function To_Pointer is
           new Ada.Unchecked_Conversion
             (System.Address, Array_Access);
         Left_Array_Ptr : Array_Access :=
           To_Pointer(Left.RB.Elements.all'Address);
         Right_Array_Ptr : Array_Access :=
           To_Pointer(Right.RB.Elements.all'Address);
      begin
         for I in Array_Type'Range loop
            if Left_Array_Ptr.all(I) /= Right_Array_Ptr.all(I) then
               return False;
            end if;
         end loop;
      end;
      return True;
   end "=";

   function To_Array(Vector : Vector_Type) return Array_Of_Byte_Type
   is
   begin
      if Vector.RB = null then
         Raise_Exception(Source_Is_Empty'Identity,
                         "Source vector for To_Array command is Empty!");
      end if;
      declare
         subtype Constant_Array is Array_Of_Byte_Type
           (Index_Type'First .. Vector.Last);
         type Constant_Array_Access is access constant Constant_Array;
         function To_Pointer is
           new Ada.Unchecked_Conversion
             (System.Address, Constant_Array_Access);
         Constant_Array_Ptr : Constant_Array_Access :=
           To_Pointer(Vector.RB.Elements.all'Address);
      begin
         return Constant_Array_Ptr.all;
      end;
   end To_Array;

   function To_String(Vector : Vector_Type) return String
   is
   begin
      if Vector.RB = null then
         Raise_Exception(Source_Is_Empty'Identity,
                         "Source vector for To_String command is Empty!");
      end if;
      declare
         subtype Constant_Array is String
           (Integer(Index_Type'First) .. Integer(Vector.Last));
         type Constant_Array_Access is access constant Constant_Array;
         function To_Pointer is
           new Ada.Unchecked_Conversion
             (System.Address, Constant_Array_Access);
         Constant_Array_Ptr : Constant_Array_Access :=
           To_Pointer(Vector.RB.Elements.all'Address);
      begin
         return Constant_Array_Ptr.all;
      end;
   end To_String;


   -----------
   -- Write --
   -----------

   procedure Write
     (Stream    : not null access Root_Stream_Type'Class;
      Container : Vector_Type)
   is
      Input_Array : Array_Of_Byte_Type := To_Array(Container);
   begin
      Count_Type'Base'Write (Stream, Length (Container));
      for J in Input_Array'Range loop
         Element_Type'Write (Stream, Input_Array(J));
      end loop;
   end Write;


   ----------
   -- Read --
   ----------

   procedure Read
     (Stream    : not null access Root_Stream_Type'Class;
      Container : out Vector_Type)
   is
      Length : Count_Type'Base;
      Last   : Index_Type'Base := No_Index;
   begin
      Clean (Container);
      Count_Type'Base'Read (Stream, Length);
      declare
         Output_Array : Array_Of_Byte_Type
           (Index_Type'First .. Index_Type'First + Index_Type(Length) - 1);
      begin
         for J in Count_Type range 1 .. Length loop
            Last := Last + 1;
            Element_Type'Read (Stream, Output_Array(Last));
         end loop;
         Create(Container, Output_Array);
      end;
   end Read;

   procedure From_Utf8_To_Utf32 (Utf32 : in out Vector_Type;
                                 Utf8 : Array_Of_Byte_Type;
                                 BOM, BE : Boolean := False)
   is
      Input_length : Count_Type := Utf8'Length;
      Position : Index_Type := Utf8'First;
      Last : Index_Type := Utf8'Last;
      Byte1, Byte2, Byte3, Byte4 : Element_Type;
      Unicode_Symbol, Aux_Symbol : Unicode_Base_Type;
   begin
      if Input_length < 1 then
         Raise_Exception(Source_Is_Empty'Identity,
                         "Input array is Empty!");
      end if;
      Clean(Utf32);
      if BOM then
         if BE then
            Append(Utf32, BOM32BE);
         else
            Append(Utf32, BOM32LE);
         end if;
      end if;
      if Input_length >= 4 then
         declare
            Sub_Array : Array_Of_Byte_Type := Utf8(Position .. Position + 3);
         begin
            if Sub_Array = BOM32LE or else Sub_Array = BOM32BE then
               Raise_Exception(Illegal_Input'Identity,
                               "Illegal input BOM");
            end if;
         end;
      end if;
      if Input_length >= 3 then
         declare
            Sub_Array : Array_Of_Byte_Type := Utf8(Position .. Position + 2);
         begin
            if Sub_Array = BOM8 then
               if not BOM then
                  if BE then
                     Append(Utf32, BOM32BE);
                  else
                     Append(Utf32, BOM32LE);
                  end if;
               end if;
               Position := Position + 3;
               if Position > Utf8'Last then
                  return;
               end if;
            end if;
         end;
      elsif Input_length >= 2 then
         declare
            Sub_Array : Array_Of_Byte_Type := Utf8(Position .. Position + 1);
         begin
            if Sub_Array = BOM16LE or else Sub_Array = BOM16BE then
               Raise_Exception(Illegal_Input'Identity,
                               "Illegal input BOM");
            end if;
         end;
      end if;
      while Position <= Last loop
         Byte1 := Utf8(Position);
         Position := Position + 1;
         if Byte1 < 128 then
            if BE then
               declare
                  Added_Array : Array_Of_Byte_Type := (0, 0, 0, Byte1);
               begin
                  Append(Utf32, Added_Array);
               end;
            else
               declare
                  Added_Array : Array_Of_Byte_Type := (Byte1, 0, 0, 0);
               begin
                  Append(Utf32, Added_Array);
               end;
            end if;
         else
            if (Byte1 and 2#11111000#) = 2#11110000# then
               if Position + 2 > Last then
                  Raise_Exception(Illegal_Input'Identity,
                                  "Illegal length of symbol");
               end if;
               Byte2 := Utf8(Position);
               if (Byte2 and Mask_First) /= Mask_Next then
                  Raise_Exception(Illegal_Input'Identity,
                                  "Illegal second byte of symbol");
               end if;
               Position := Position + 1;
               Byte3 := Utf8(Position);
               if (Byte3 and Mask_First) /= Mask_Next then
                  Raise_Exception(Illegal_Input'Identity,
                                  "Illegal third byte of symbol");
               end if;
               Position := Position + 1;
               Byte4 := Utf8(Position);
               if (Byte4 and Mask_First) /= Mask_Next then
                  Raise_Exception(Illegal_Input'Identity,
                                  "Illegal fourth byte of symbol");
               end if;
               Position := Position + 1;
               Unicode_Symbol := Unicode_Base_Type(Byte4 and Value_Mask);
               Aux_Symbol := Unicode_Base_Type(Byte3 and Value_Mask);
               Aux_Symbol := Shift_Left(Aux_Symbol, 6);
               Unicode_Symbol := Unicode_Symbol or Aux_Symbol;
               Aux_Symbol := Unicode_Base_Type(Byte2 and Value_Mask);
               Aux_Symbol := Shift_Left(Aux_Symbol, 12);
               Unicode_Symbol := Unicode_Symbol or Aux_Symbol;
               Aux_Symbol := Unicode_Base_Type(Byte1 and 2#00000111#);
               Aux_Symbol := Shift_Left(Aux_Symbol, 18);
               Unicode_Symbol := Unicode_Symbol or Aux_Symbol;
            elsif (Byte1 and 2#11110000#) = 2#11100000# then
               if Position + 1 > Last then
                  Raise_Exception(Illegal_Input'Identity,
                                  "Illegal length of symbol");
               end if;
               Byte2 := Utf8(Position);
               if (Byte2 and Mask_First) /= Mask_Next then
                  Raise_Exception(Illegal_Input'Identity,
                                  "Illegal second byte of symbol");
               end if;
               Position := Position + 1;
               Byte3 := Utf8(Position);
               if (Byte3 and Mask_First) /= Mask_Next then
                  Raise_Exception(Illegal_Input'Identity,
                                  "Illegal third byte of symbol");
               end if;
               Position := Position + 1;
               Unicode_Symbol := Unicode_Base_Type(Byte3 and Value_Mask);
               Aux_Symbol := Unicode_Base_Type(Byte2 and Value_Mask);
               Aux_Symbol := Shift_Left(Aux_Symbol, 6);
               Unicode_Symbol := Unicode_Symbol or Aux_Symbol;
               Aux_Symbol := Unicode_Base_Type(Byte1 and 2#00001111#);
               Aux_Symbol := Shift_Left(Aux_Symbol, 12);
               Unicode_Symbol := Unicode_Symbol or Aux_Symbol;
            elsif (Byte1 and 2#11100000#) = 2#11000000# then
               if Position > Last then
                  Raise_Exception(Illegal_Input'Identity,
                                  "Illegal length of symbol");
               end if;
               Byte2 := Utf8(Position);
               if (Byte2 and Mask_First) /= Mask_Next then
                  Raise_Exception(Illegal_Input'Identity,
                                  "Illegal second byte of symbol");
               end if;
               Position := Position + 1;
               Unicode_Symbol := Unicode_Base_Type(Byte2 and Value_Mask);
               Aux_Symbol := Unicode_Base_Type(Byte1 and 2#00011111#);
               Aux_Symbol := Shift_Left(Aux_Symbol, 6);
               Unicode_Symbol := Unicode_Symbol or Aux_Symbol;
            else
               Raise_Exception(Illegal_Input'Identity,
                               "Illegal first byte of symbol");
            end if;
            if Unicode_Symbol = 2#1111_1110_1111_1111# then
               Raise_Exception(Illegal_Input'Identity,
                               "Illegal position of BOM8");
            end if;
            Aux_Symbol := Unicode_Symbol and 16#FF#;
            Byte1 := Element_Type(Aux_Symbol);
            Unicode_Symbol := Shift_Right(Unicode_Symbol, 8);
            Aux_Symbol := Unicode_Symbol and 16#FF#;
            Byte2 := Element_Type(Aux_Symbol);
            Unicode_Symbol := Shift_Right(Unicode_Symbol, 8);
            Aux_Symbol := Unicode_Symbol and 16#FF#;
            Byte3 := Element_Type(Aux_Symbol);
            Unicode_Symbol := Shift_Right(Unicode_Symbol, 8);
            Byte4 := Element_Type(Unicode_Symbol);
            if BE then
               declare
                  Added_Array : Array_Of_Byte_Type :=
                    (Byte4, Byte3, Byte2, Byte1);
               begin
                  Append(Utf32, Added_Array);
               end;
            else
               declare
                  Added_Array : Array_Of_Byte_Type :=
                    (Byte1, Byte2, Byte3, Byte4);
               begin
                  Append(Utf32, Added_Array);
               end;
            end if;
         end if;
      end loop;
   end From_Utf8_To_Utf32;


   procedure Create (Vector : in out Vector_Type;
                     Source : String)
   is
      subtype Bounded_Array is Array_Of_Byte_Type
        (Index_Type(Source'First) .. Index_Type(Source'Last));
      type Bounded_Array_Access is access constant Bounded_Array;
      function To_Pointer is
        new Ada.Unchecked_Conversion (System.Address, Bounded_Array_Access);
      Input_Ptr : Bounded_Array_Access :=
        To_Pointer(Source(Source'First)'Address);
   begin
      Create(Vector, Input_Ptr.all);
   end Create;

   procedure Append (Vector : in out Vector_Type;
                     Source : String)
   is
      subtype Bounded_Array is Array_Of_Byte_Type
        (Index_Type(Source'First) .. Index_Type(Source'Last));
      type Bounded_Array_Access is access constant Bounded_Array;
      function To_Pointer is
        new Ada.Unchecked_Conversion (System.Address, Bounded_Array_Access);
      Input_Ptr : Bounded_Array_Access :=
        To_Pointer(Source(Source'First)'Address);
   begin
      Append(Vector, Input_Ptr.all);
   end Append;

   procedure From_Utf8_To_Utf32 (Utf32 : in out Vector_Type;
                                 Utf8 : String;
                                 BOM, BE : Boolean := False)
   is
      subtype Bounded_Array is Array_Of_Byte_Type
        (Index_Type(Utf8'First) .. Index_Type(Utf8'Last));
      type Bounded_Array_Access is access constant Bounded_Array;
      function To_Pointer is
        new Ada.Unchecked_Conversion (System.Address, Bounded_Array_Access);
      Input_Ptr : Bounded_Array_Access :=
        To_Pointer(Utf8(Utf8'First)'Address);
   begin
      From_Utf8_To_Utf32(Utf32,
                         Input_Ptr.all,
                         BOM,
                         BE);
   end From_Utf8_To_Utf32;


end Vector_Of_Bytes;
