
with System;
with System.Storage_Elements;
with Ada.Finalization;                      use Ada.Finalization;
with Referencing.Types;                     use Referencing.Types;
with Referencing.Types.Operations;          use Referencing.Types.Operations;
with Ada.Unchecked_Deallocation;
with Ada.Unchecked_Conversion;
with Ada.Exceptions;                        use Ada.Exceptions;
with Ada.Streams;                           use Ada.Streams;
with Containers;                            use Containers;
with Interfaces;                            use Interfaces;

package Vector_Of_Bytes is

   Size_Delta : constant := 1023;
   Mask_First : constant := 2#11000000#;
   Mask_Next  : constant := 2#10000000#;
   Value_Mask : constant := 2#00111111#;

   Source_Is_Empty : exception;
   Illegal_Input   : exception;

   subtype Unicode_Base_Type is Unsigned_32;

   type Index_Type is new Integer range 1 .. Integer'Last;
   subtype Extended_Index is
     Index_Type'Base range
       Index_Type'First-1 ..
         Index_Type'Min (Index_Type'Base'Last - 1, Index_Type'Last) + 1;
   No_Index : constant Extended_Index := Extended_Index'First;


   type Element_Type is new System.Storage_Elements.Storage_Element;
   type Elements_Access is access all Element_Type;
   procedure Free is
     new Ada.Unchecked_Deallocation (Element_Type, Elements_Access);
   function To_Pointer is
     new Ada.Unchecked_Conversion (System.Address, Elements_Access);

   type Array_Of_Byte_Type is array
     (Index_Type range <>) of aliased Element_Type;
   type Array_Of_Byte_Access is access all Array_Of_Byte_Type;

   BOM32LE : constant Array_Of_Byte_Type := (16#FF#, 16#FE#, 0, 0);
   BOM32BE : constant Array_Of_Byte_Type := (0, 0, 16#FE#, 16#FF#);
   BOM16LE : constant Array_Of_Byte_Type := (16#FF#, 16#FE#);
   BOM16BE : constant Array_Of_Byte_Type := (16#FE#, 16#FF#);
   BOM8    : constant Array_Of_Byte_Type := (16#EF#, 16#BB#, 16#BF#);

   type Vector_Type is tagged limited private;
   type Vector_Access is access all Vector_Type;

   function Is_Empty (Vector : Vector_Type) return Boolean;
   pragma Inline(Is_Empty);

   function Length (Vector : Vector_Type) return Count_Type;
   pragma Inline(Length);

   procedure Copy (Target : in out Vector_Type; Source : Vector_Type);

   procedure Move (Target, Source : in out Vector_Type);

   procedure Clean (Vector : in out Vector_Type; Unallocate : Boolean := False);

   procedure Create (Vector : in out Vector_Type;
                     Source : Array_Of_Byte_Type);

   procedure Create (Vector : in out Vector_Type;
                     Source : String);

   procedure Append (Vector : in out Vector_Type;
                     Source : Element_Type;
                     Index  : out Index_Type);

   procedure Append (Vector : in out Vector_Type;
                     Source : Array_Of_Byte_Type);

   procedure Append (Vector : in out Vector_Type;
                     Source : String);

   procedure Append (Vector : in out Vector_Type;
                     Source : in out Vector_Type);

   function "=" (Left, Right : Vector_Type) return Boolean;

   function To_Array(Vector : Vector_Type) return Array_Of_Byte_Type;
   pragma Inline(To_Array);

   function To_String(Vector : Vector_Type) return String;
   pragma Inline(To_String);

   procedure From_Utf8_To_Utf32 (Utf32 : in out Vector_Type;
                                 Utf8 : Array_Of_Byte_Type;
                                 BOM, BE : Boolean := False);

   procedure From_Utf8_To_Utf32 (Utf32 : in out Vector_Type;
                                 Utf8 : String;
                                 BOM, BE : Boolean := False);

private

   type Referenced_Buffer is new Referenced with
      record
         Elements : Elements_Access := null;
      end record;
   type Referenced_Buffer_Ptr is access all Referenced_Buffer;
   procedure Free is
      new Ada.Unchecked_Deallocation (Referenced_Buffer, Referenced_Buffer_Ptr);

   type Vector_Type is new Limited_Controlled with record
      RB       : Referenced_Buffer_Ptr := null;
      Last     : Extended_Index := No_Index;
      BS       : Count_Type := 0;
   end record;
   procedure Finalize (Object : in out Vector_Type);
   procedure Write
     (Stream    : not null access Root_Stream_Type'Class;
      Container : Vector_Type);
   for Vector_Type'Write use Write;
   procedure Read
     (Stream    : not null access Root_Stream_Type'Class;
      Container : out Vector_Type);
   for Vector_Type'Read use Read;

   procedure Free is
      new Ada.Unchecked_Deallocation (Vector_Type, Vector_Access);


end Vector_Of_Bytes;
