
/* --------------------------------------------- */
/*  H8-3069F Main function                       */
/*                                               */
/*  CPU    : Renesus H8/3069F 25MHz              */
/*  Memory : ROM 512KB, RAM 16KB E-RAM 2MB       */
/*                (c) KAZ.Imamura                */
/* --------------------------------------------- */

#include "common.h"
#include "sc1602.h"
#include "key.h"
#include "ui.h"
#include "sl811.h"
#include "usb_ms.h"
#include "led.h"
#include "serial.h"

int counter_500us;
unsigned int counter_1ms;

extern unsigned char eram_start;
extern unsigned char eram_end;

const char hex2char[256][3] = {
	"00",	"01",	"02",	"03",	"04",	"05",	"06",	"07",	"08",	"09",	"0A",	"0B",	"0C",	"0D",	"0E",	"0F",
	"10",	"11",	"12",	"13",	"14",	"15",	"16",	"17",	"18",	"19",	"1A",	"1B",	"1C",	"1D",	"1E",	"1F",
	"20",	"21",	"22",	"23",	"24",	"25",	"26",	"27",	"28",	"29",	"2A",	"2B",	"2C",	"2D",	"2E",	"2F",
	"30",	"31",	"32",	"33",	"34",	"35",	"36",	"37",	"38",	"39",	"3A",	"3B",	"3C",	"3D",	"3E",	"3F",
	"40",	"41",	"42",	"43",	"44",	"45",	"46",	"47",	"48",	"49",	"4A",	"4B",	"4C",	"4D",	"4E",	"4F",
	"50",	"51",	"52",	"53",	"54",	"55",	"56",	"57",	"58",	"59",	"5A",	"5B",	"5C",	"5D",	"5E",	"5F",
	"60",	"61",	"62",	"63",	"64",	"65",	"66",	"67",	"68",	"69",	"6A",	"6B",	"6C",	"6D",	"6E",	"6F",
	"70",	"71",	"72",	"73",	"74",	"75",	"76",	"77",	"78",	"79",	"7A",	"7B",	"7C",	"7D",	"7E",	"7F",
	"80",	"81",	"82",	"83",	"84",	"85",	"86",	"87",	"88",	"89",	"8A",	"8B",	"8C",	"8D",	"8E",	"8F",
	"90",	"91",	"92",	"93",	"94",	"95",	"96",	"97",	"98",	"99",	"9A",	"9B",	"9C",	"9D",	"9E",	"9F",
	"A0",	"A1",	"A2",	"A3",	"A4",	"A5",	"A6",	"A7",	"A8",	"A9",	"AA",	"AB",	"AC",	"AD",	"AE",	"AF",
	"B0",	"B1",	"B2",	"B3",	"B4",	"B5",	"B6",	"B7",	"B8",	"B9",	"BA",	"BB",	"BC",	"BD",	"BE",	"BF",
	"C0",	"C1",	"C2",	"C3",	"C4",	"C5",	"C6",	"C7",	"C8",	"C9",	"CA",	"CB",	"CC",	"CD",	"CE",	"CF",
	"D0",	"D1",	"D2",	"D3",	"D4",	"D5",	"D6",	"D7",	"D8",	"D9",	"DA",	"DB",	"DC",	"DD",	"DE",	"DF",
	"E0",	"E1",	"E2",	"E3",	"E4",	"E5",	"E6",	"E7",	"E8",	"E9",	"EA",	"EB",	"EC",	"ED",	"EE",	"EF",
	"F0",	"F1",	"F2",	"F3",	"F4",	"F5",	"F6",	"F7",	"F8",	"F9",	"FA",	"FB",	"FC",	"FD",	"FE",	"FF"
};


// -------------------------------------------
//  Proto type definitions
// -------------------------------------------
void port_initialize(void);
void register_initialize(void);

void int_CMIA0();
void int_CMIA1();
void int_IRQ0();
void int_DEND0();

// Local 
void memfill_word( unsigned int val );

// -------------------------------------------
//  Main routine
// -------------------------------------------
int main(void) {

	BSC.ABWCR.BYTE   = 0xff;	// CS0-CS7:8bitoX
	BSC.ASTCR.BYTE = 0xff;	// CS0-CS7:3Xe[gANZX
	BSC.BCR.BYTE   = 0xc6;	// 

	P1DDR=0xFF;		// Port1 use for BUS
	P2DDR=0x07;		// Port2 use for BUS
	P8DDR=0xEC;		// Port8(4-2) use for CS
	RTCOR=0x03;
	RTMCSR=0x27;
	DRCRB=0x98;
	DRCRA=0x78;

	// uCom initialize
	port_initialize();
	register_initialize();
	
	// Function initialize
	led_initialize();
	sc1602_initialize();
	key_initialize();
	ui_initialize();
	sl811_initialize();
	usbms_initialize();
	fat_initialize();
	serial_initialize();
	
	permit_irq();
	
	while(1) {
		led_process();
		sc1602_process();
		key_process();
		ui_process();
		sl811_process();
		usbms_process();
		fat_process();
		serial_process();
	}
}

// -------------------------------------------
//  Port initialize for start up
// -------------------------------------------
void port_initialize(void) {

	// Port4
	// D7 : (OUT) LED 1
	// D6 : (OUT) LED 0
	// D5 : (OUT) LCD Driver - E
	// D4 : (OUT) LCD Driver - RS
	// D3 : (OUT) LCD Driver - DB7
	// D2 : (OUT) LCD Driver - DB6
	// D1 : (OUT) LCD Driver - DB5
	// D0 : (OUT) LCD Driver - DB4
	P4DDR = 0xFF;
	P4DR.BYTE = 0x00;
	
	// Port5
	// D3 : (IN ) Switch D (BACK)
	// D2 : (IN ) Switch C (OK)
	// D1 : (IN ) Switch B (DOWN)
	// D0 : (IN ) Switch A (UP)
	P5DDR = 0xF0;
	P5PCR.BYTE = 0xFF;
}


// -------------------------------------------
//  Register initialize for start up
// -------------------------------------------
void register_initialize(void) {
	// --------------------------------------
	// DMAC
	// --------------------------------------
	DMAC0.DTCRA.BIT.DTE = 0;		// Disable transfer
	DMAC0.DTCRA.BIT.DTSZ = 1;		// Word size transfer
	DMAC0.DTCRA.BIT.DTIE = 0;		// Disable interrupt
	DMAC0.DTCRA.BIT.SAIDE = 0;		// Source address fix
	DMAC0.DTCRA.BIT.DTS = 0x06;		// Full address mode (w/ Normal mode)
	DMAC0.DTCRB.BIT.DAID = 0;
	DMAC0.DTCRB.BIT.DAIDE = 1;		// Destination address increment
	DMAC0.DTCRB.BIT.TMS = 0;
	DMAC0.DTCRB.BIT.DTS = 0x00;		// Auto request (Burst mode)
	
	// --------------------------------------
	// D/A converter
	// --------------------------------------
	DA.DADR0 = 0x40;
	DA.DACR.BIT.DAOE0 = 1;
	
	// --------------------------------------
	// 8 bit timer ch-0
	// --------------------------------------
	
	// Compare - A register
	//   25MHz CPU : 40ns per clock
	//   Use 8bit timer as 64interval mode. (1 interval = 2.56us)
	TMR0.TCORA = 195; // 499.2 us
	
	// Control register
	TMR0.TCR.BIT.CMIEB	= 0;			// Compare B disable
	TMR0.TCR.BIT.CMIEA	= 1;			// Compare A enable
	TMR0.TCR.BIT.OVIE 	= 0;			// Overflow disable
	TMR0.TCR.BIT.CCLR	= 0x01;			// Comp-A make counte clear
	TMR0.TCR.BIT.CKS	= 0x02;			// Use internal clock. 64 interval.
	TMR0.TCSR.BIT.CMFA  = 0;     		// Interrupt Clear 
	
	
	// --------------------------------------
	// 8 bit timer ch-1
	// --------------------------------------
	
	// Compare - A register
	//   25MHz CPU : 40ns per clock
	//   Use 8bit timer as 8192interval mode. (1 interval = 327.68us)
	TMR1.TCORA = 61; // 19.98848 ms
	
	// Control register
	TMR1.TCR.BIT.CMIEB	= 0;			// Compare B disable
	TMR1.TCR.BIT.CMIEA	= 1;			// Compare A enable
	TMR1.TCR.BIT.OVIE 	= 0;			// Overflow disable
	TMR1.TCR.BIT.CCLR	= 0x01;			// Comp-A make counte clear
	TMR1.TCR.BIT.CKS	= 0x03;			// Use internal clock. 8192 interval.
	TMR1.TCSR.BIT.CMFA  = 0;     		// Interrupt Clear 

}

// -------------------------------------------
//  Interruption handler
// -------------------------------------------
#pragma interrupt (int_CMIA0)
void int_CMIA0() {
	TMR0.TCSR.BIT.CMFA = 0;     	// Interrupt Clear 
	counter_500us++;				// interrupt per 500us.
	
	// 1ms interval 
	if( (counter_500us & 0x01) == 0x01 ) {
		counter_1ms++;
		
		sc1602_1ms_handler();
		key_1ms_handler();
		serial_1ms_handler();
	} else {
		ui_1ms_handler();
		sl811_1ms_handler();
		usbms_1ms_handler();
	}
}

#pragma interrupt (int_CMIA1)
void int_CMIA1() {
	TMR1.TCSR.BIT.CMFA = 0;     	// Interrupt Clear 
	led_20ms_handler();
}

#pragma interrupt (int_IRQ0)
void int_IRQ0() {
	sl811_chip_irq();
}

#pragma interrupt (int_DEND0)
void int_DEND0() {
	
}




void memfill_word( unsigned int val ) {
	unsigned int *addr;
	unsigned int buf;
	
#ifdef SYS_DEBUG_ON
	printf("[SYS] memory fill. (val=%04X) (1ms=%d)\r\n", counter_1ms, val);
#endif
	
	buf = val;
	addr = (unsigned int *)&eram_start;
	
	while( addr < (unsigned int *)&eram_end ) {
		DMAC0.MARA = (void *)&buf;
		DMAC0.MARB = (void *)addr;
		DMAC0.ETCRA = 0x0000;			// 65536 times
		
		DMAC0.DTCRB.BIT.DAID = 0;
		DMAC0.DTCRB.BIT.DAIDE = 1;		// Destination address increment
		DMAC0.DTCRB.BIT.TMS = 0;
		DMAC0.DTCRB.BIT.DTS = 0x00;		// Auto request (Burst mode)
		
		DMAC0.DTCRA.BIT.DTE = 0;		// Disable transfer
		DMAC0.DTCRA.BIT.DTSZ = 1;		// Word size transfer
		DMAC0.DTCRA.BIT.DTIE = 0;		// Disable interrupt
		DMAC0.DTCRA.BIT.SAIDE = 0;		// Source address fix
		DMAC0.DTCRA.BIT.DTS = 0x06;		// Full address mode (w/ Normal mode)
		
		if(DMAC0.DTCRB.BIT.DTME) DMAC0.DTCRB.BIT.DTME = 0;
		DMAC0.DTCRB.BIT.DTME = 1;
		
		if(DMAC0.DTCRA.BIT.DTE) DMAC0.DTCRA.BIT.DTE = 0;
		DMAC0.DTCRA.BIT.DTE = 1;
		
		while( DMAC0.DTCRA.BIT.DTE ) ;
		
		addr += 0x10000;
	}
	
#ifdef SYS_DEBUG_ON
	printf("[SYS] done (1ms=%d)\r\n", counter_1ms);
#endif

}
