
/* --------------------------------------------- */
/*  H8-3069F USB Mass Storage Class              */
/*                                               */
/*  CPU    : Renesus H8/3069F 25MHz              */
/*  Memory : ROM 512KB, RAM 16KB E-RAM 2MB       */
/*                (c) KAZ.Imamura                */
/* --------------------------------------------- */

#include "usb_ms.h"
#include "led.h"

#define MS_DATA_DIR_IN   0x80
#define MS_DATA_DIR_OUT  0x00
#define MS_DATA_DIR_NONE 0x00

#define  CBW_SIZE                      31
#define  CSW_SIZE                      13

#define  SCSI_CMD_REQUEST_SENSE      0x03
#define  SCSI_CMD_TEST_UNIT_READY    0x00
#define  SCSI_CMD_READ_10            0x28
#define  SCSI_CMD_READ_CAPACITY      0x25
#define  SCSI_CMD_WRITE_10           0x2A

#define  SCSI_LEN_REQUEST_SENSE      0x06
#define  SCSI_LEN_TEST_UNIT_READY    0x06
#define  SCSI_LEN_READ_CAPACITY      0x0A

#define SENSE_BUF_SIZE				0x0E
#define TEMP_BUF_SIZE				32

extern unsigned char eram_start;
extern unsigned char eram_end;

// -------------------------------------------
//  Proto type definitions
// -------------------------------------------
// Global
void usbms_1ms_handler(void);
int  usbms_initialize(void);
int  usbms_process(void);
int ui_function_usbms_debug(UI_COMMAND uicmd);
unsigned char usbms_read10(P_READ10_REQ p_req);

// Locals

// -------------------------------------------
//  Variables
// -------------------------------------------
// Global
// Locals
//  +------------+-----+-----+-----+-----+-----+-----+-----+-----+
//  |   Byte     |  7  |  6  |  5  |  4  |  3  |  2  |  1  |  0  |
//  +============+=====+=====+=====+=====+=====+=====+=====+=====+
//  |   0-3      |             dCBWSignature                     |
//  +------------+-----------------------------------------------+
//  |   4-7      |                 dCBWTag                       |
//  +------------+-----------------------------------------------+
//  |   8-11     |         dCBWDataTransferLength                |
//  +------------+-----------------------------------------------+
//  |   12       |               bmCBWFlags                      |
//  +------------+-----------------------+-----------------------+
//  |   13       |      Reserved(0)      |        bCBWLUN        |
//  +------------+-----------------+-----+-----------------------+
//  |   14       |   Reserved(0)   |         bCBWCBLength        |
//  +------------+-----------------+-----------------------------+
//  |   15-30    |                     CBWCB                     |
//  +------------+-----------------------------------------------+
unsigned char CmdBlkWrapper[CBW_SIZE];

//  +------------+-----+-----+-----+-----+-----+-----+-----+-----+
//  |   Byte     |  7  |  6  |  5  |  4  |  3  |  2  |  1  |  0  |
//  +============+=====+=====+=====+=====+=====+=====+=====+=====+
//  |   0-3      |             dCSWSignature                     |
//  +------------+-----------------------------------------------+
//  |   4-7      |                 dCSWTag                       |
//  +------------+-----------------------------------------------+
//  |   8-11     |              dCSWDataResidue                  |
//  +------------+-----------------------------------------------+
//  |   12       |                 bCSWStatus                    |
//  +------------+-----------------------------------------------+
static unsigned char CmdStsWrapper[CSW_SIZE];
static unsigned char SenseBuffer[SENSE_BUF_SIZE];
static unsigned char TempBuffer[TEMP_BUF_SIZE];


static int usbms_wait_timer;
static int usbms_disable_timer;
static unsigned char usbms_proc;
static unsigned char usbms_error;
static unsigned char usbms_reset_request;

static unsigned char max_lun;
static unsigned long max_LBA;
static unsigned long BlockSize;
static unsigned char *p_buffer_addr;

static READ10_REQ read10_param;

enum usbms_error_code {
	USBMS_ERR_NONE,
	USBMS_ERR_TUR_FAIL,
	USBMS_PROCESS_ERROR = 100,
};

enum usbms_process_mode {
	USBMS_READY_WAIT,				// 00
	USBMS_GET_MAX_LUN_01,			// 01
	USBMS_GET_MAX_LUN_02,			// 02
	USBMS_TESTUNIT_READY_01,		// 03
	USBMS_TESTUNIT_READY_02,		// 04
	USBMS_TESTUNIT_READY_03,		// 05
	USBMS_TESTUNIT_READY_04,		// 06
	USBMS_GET_SENSE_INFO_01,		// 07
	USBMS_GET_SENSE_INFO_02,		// 08
	USBMS_GET_SENSE_INFO_03,		// 09
	USBMS_GET_SENSE_INFO_04,		// 10
	USBMS_GET_SENSE_INFO_05,		// 11
	USBMS_GET_SENSE_INFO_06,		// 12
	USBMS_READ_CAPACITY_01,			// 13
	USBMS_READ_CAPACITY_02,			// 14
	USBMS_READ_CAPACITY_03,			// 15
	USBMS_READ_CAPACITY_04,			// 16
	USBMS_READ_CAPACITY_05,			// 17
	USBMS_READ_CAPACITY_06,			// 18
	USBMS_IDLE = 0x80,
	USBMS_READ10_01,				// 0x81
	USBMS_READ10_02,				// 0x82
	USBMS_READ10_03,				// 0x83
	USBMS_READ10_04,				// 0x84
	USBMS_READ10_05,				// 0x85
	USBMS_READ10_06,				// 0x86
	USBMS_READ10_07,				// 0x87
	USBMS_READ10_08,				// 0x88
	USBMS_RESET,
	USBMS_ERROR_STOP = 0xFF,
};

static const DEBUG_VAR_TABLE debug_var_table[] = {
	{	 0,		 1,	"Process number: ",	&usbms_proc,		},
	{	 1,		 1,	"Error Code:     ",	&usbms_error,		},
	{	 2,		 1,	"Max LUN:        ",	&max_lun,			},
	{	 3,		 3,	"max_LBA:        ",	&max_LBA,			},
	{	 4,		 3,	"BlockSize:      ",	&BlockSize,			},
	{	 4,		 3,	"p_buffer_addr:  ",	&p_buffer_addr,		},
	{	-1,		 1,	"0123456789ABCDEF", 0,					},
};


// -------------------------------------------
//  Interrupt handlers
// -------------------------------------------
void usbms_1ms_handler(void) {
	if( usbms_wait_timer )    usbms_wait_timer--;
	if( usbms_disable_timer ) usbms_disable_timer--;
}

// -------------------------------------------
//  Initialize
// -------------------------------------------
int  usbms_initialize(void) {
	usbms_proc = USBMS_READY_WAIT;
	usbms_error = USBMS_ERR_NONE;
	usbms_reset_request = 1;
	usbms_disable_timer = 0;
}

// -------------------------------------------
//  Query for USB Mass strage class status
// -------------------------------------------
unsigned char usbms_status(unsigned char req) {
	
	switch( req ) {
		case CLASS_REQ_RESET:
			if( usbms_proc < USBMS_IDLE ) {			// If initialize in progress
				return CLASS_STS_RESET_IN_PROGRESS;	
			} else if( usbms_proc == USBMS_IDLE || usbms_proc == USBMS_ERROR_STOP ) {	// Reset accepted
				usbms_initialize();
				return CLASS_STS_RESET_IN_PROGRESS;
			} else {								// Class is busy
				return CLASS_STS_NOT_READY;
			}
			break;
			
		default:
			break;
	}
	
	// Status return
	if( usbms_proc < USBMS_IDLE )				return CLASS_STS_RESET_IN_PROGRESS;	
	else if( usbms_proc == USBMS_IDLE ) 		return CLASS_STS_READY;
	else if( usbms_proc == USBMS_ERROR_STOP ) 	return CLASS_STS_ERROR_STOP;
	else 										return CLASS_STS_NOT_READY;
}	

// -------------------------------------------
//  Query for READ10
// -------------------------------------------
unsigned char usbms_read10(P_READ10_REQ p_req) {
	if( usbms_proc != USBMS_IDLE )	return CLASS_REQ_BUSY;
	
	read10_param.logical_block_address	= p_req->logical_block_address;
	read10_param.transfer_length		= p_req->transfer_length;
	read10_param.p_buffer				= p_req->p_buffer;
	usbms_proc = USBMS_READ10_01;
	return CLASS_REQ_ACCEPTED;
}

// -------------------------------------------
//  Main process
// -------------------------------------------
int  usbms_process(void) {
	unsigned char status, i, pktsts;
	unsigned int tmp;
	SETUP_FORMAT	setup_fmt;
	
	if(usbms_disable_timer) return 0;
	
	switch( usbms_proc )
	{
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Waiting for device class ready                                            *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USBMS_READY_WAIT:
			if( usbms_reset_request == 1 ) {
				usbms_proc = USBMS_RESET;
				usbms_reset_request = 0;
				p_buffer_addr = &eram_start;
			}
			led_mode_set(0, LED_MODE_BLINK_300MS);
			
			status = sl811_status(CLASS_REQ_NONE);
			if( status == CLASS_STS_READY ) {
				// Initialize
				for(i=0; i<CBW_SIZE; i++) CmdBlkWrapper[i] = 0;
				for(i=0; i<CSW_SIZE; i++) CmdStsWrapper[i] = 0;
				for(i=0; i<SENSE_BUF_SIZE; i++) SenseBuffer[i] = 0;
				for(tmp=0; tmp<TEMP_BUF_SIZE; tmp++) TempBuffer[tmp] = 0;
				usbms_wait_timer = 0;
				usbms_disable_timer = 0;
				usbms_error = USBMS_ERR_NONE;
				usbms_reset_request = 0;
				max_LBA = 0;
				BlockSize = 0;
				
				// Start !
				usbms_proc = USBMS_GET_MAX_LUN_01;
//				usbms_proc = USBMS_TESTUNIT_READY_01;
			} else if( status == CLASS_STS_RESET_IN_PROGRESS ) {
					usbms_disable_timer = 10;   // 100ms Wait
			} else {
					usbms_disable_timer = 100;  // 1S Wait
			}
			break;
		
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Get max number of LUN                                                     *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USBMS_GET_MAX_LUN_01:
			setup_fmt.bmRequestType 	= DIR_DEV2HOST | USB_REQUEST_TYPE_CLASS| RECEIPIENT_INTERFACE;
			setup_fmt.bRequest 			= REQ_TYPE_GET_MAX_LUN;
			setup_fmt.wValue 			= 0;
			setup_fmt.wIndex 			= 0;
			setup_fmt.wLength 			= 1;
			
#ifdef MSCLASS_DEBUG_ON
			printf("[MSC] GetMaxLUN\r\n");
#endif
			if( sl811_ctrl_receive(&setup_fmt) == CLASS_REQ_ACCEPTED )
				usbms_proc = USBMS_GET_MAX_LUN_02;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
				
			break;
			
		case USBMS_GET_MAX_LUN_02:
			status = sl811_status(CLASS_REQ_NONE);
			if( status == CLASS_STS_READY ) {
				usbms_proc = USBMS_TESTUNIT_READY_01;
//				usbms_proc = USBMS_GET_SENSE_INFO_01;
//				usbms_proc = USBMS_READ_CAPACITY_01;
//				usbms_proc = USBMS_MS_RESET_01;
				sl811_buffer_copy( &max_lun, 1);
			} else if( status == CLASS_STS_ERROR_STOP ) {
				pktsts = IsPacketStatus();
				if( pktsts & SL811HS_HOSTSTATUS_BIT_STALL ) {
					// not multiple LUN case
					sl811_status(CLASS_REQ_RECOVERY);
				} else {
					usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
					usbms_proc = USBMS_ERROR_STOP;
				}
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
				
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Test Unit Ready                                                           *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USBMS_TESTUNIT_READY_01:
			for(i=0; i<CBW_SIZE; i++) CmdBlkWrapper[i] = 0;
//			CmdBlkWrapper[ 0] = 0x43;	CmdBlkWrapper[ 1] = 0x42;	CmdBlkWrapper[ 2] = 0x53;	CmdBlkWrapper[ 3] = 0x55;
			CmdBlkWrapper[ 3] = 0x43;	CmdBlkWrapper[ 2] = 0x42;	CmdBlkWrapper[ 1] = 0x53;	CmdBlkWrapper[ 0] = 0x55;
			CmdBlkWrapper[ 8] = 0x00;
			CmdBlkWrapper[12] = MS_DATA_DIR_NONE;
			CmdBlkWrapper[14] = SCSI_LEN_TEST_UNIT_READY;
			CmdBlkWrapper[15] = SCSI_CMD_TEST_UNIT_READY;
			
#ifdef MSCLASS_DEBUG_ON
			printf("[MSC] TUR\r\n");
#endif
			if( sl811_bulk_out_request(CmdBlkWrapper, CBW_SIZE) == CLASS_REQ_ACCEPTED )
				usbms_proc = USBMS_TESTUNIT_READY_02;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
			break;
			
		case USBMS_TESTUNIT_READY_02:
			status = sl811_status(CLASS_REQ_NONE);
			if( status == CLASS_STS_READY ) {
				usbms_proc = USBMS_TESTUNIT_READY_03;
			} else if( status == CLASS_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
			
		case USBMS_TESTUNIT_READY_03:
			if( sl811_bulk_in_request(CmdStsWrapper, CSW_SIZE) == CLASS_REQ_ACCEPTED )
				usbms_proc = USBMS_TESTUNIT_READY_04;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
			break;
			
		case USBMS_TESTUNIT_READY_04:
			status = sl811_status(CLASS_REQ_NONE);
			if( status == CLASS_STS_READY ) {
				if( CmdStsWrapper[12] 	!= 0 ) {
					usbms_proc  = USBMS_GET_SENSE_INFO_01;
				} else {
					usbms_proc  = USBMS_GET_SENSE_INFO_01;
//					usbms_proc = USBMS_IDLE;
				}
			} else if( status == CLASS_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
			
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Get Sense Info                                                            *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USBMS_GET_SENSE_INFO_01:
			for(i=0; i<SENSE_BUF_SIZE; i++) SenseBuffer[i] = 0;
			for(i=0; i<CBW_SIZE; i++) CmdBlkWrapper[i] = 0;
//			CmdBlkWrapper[ 0] = 0x43;	CmdBlkWrapper[ 1] = 0x42;	CmdBlkWrapper[ 2] = 0x53;	CmdBlkWrapper[ 3] = 0x55;
			CmdBlkWrapper[ 3] = 0x43;	CmdBlkWrapper[ 2] = 0x42;	CmdBlkWrapper[ 1] = 0x53;	CmdBlkWrapper[ 0] = 0x55;
			CmdBlkWrapper[ 8] = SENSE_BUF_SIZE;
			CmdBlkWrapper[12] = MS_DATA_DIR_IN;
			CmdBlkWrapper[14] = SCSI_LEN_REQUEST_SENSE;		// 0x06
			CmdBlkWrapper[15] = SCSI_CMD_REQUEST_SENSE;		// 0x03
			CmdBlkWrapper[19] = SENSE_BUF_SIZE;				// 0x0E
			
#ifdef MSCLASS_DEBUG_ON
			printf("[MSC] GetSense\r\n");
#endif
			if( sl811_bulk_out_request(CmdBlkWrapper, CBW_SIZE) == CLASS_REQ_ACCEPTED )
				usbms_proc = USBMS_GET_SENSE_INFO_02;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
			break;
			
		case USBMS_GET_SENSE_INFO_02:
			status = sl811_status(CLASS_REQ_NONE);
			if( status == CLASS_STS_READY ) {
				usbms_proc = USBMS_GET_SENSE_INFO_03;
			} else if( status == CLASS_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
			
		case USBMS_GET_SENSE_INFO_03:
			if( sl811_bulk_in_request(SenseBuffer, SENSE_BUF_SIZE) == CLASS_REQ_ACCEPTED )
				usbms_proc = USBMS_GET_SENSE_INFO_04;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
			break;
			
		case USBMS_GET_SENSE_INFO_04:
			status = sl811_status(CLASS_REQ_NONE);
			if( status == CLASS_STS_READY ) {
				usbms_proc = USBMS_GET_SENSE_INFO_05;
			} else if( status == CLASS_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
			
		case USBMS_GET_SENSE_INFO_05:
			if( sl811_bulk_in_request(CmdStsWrapper, CSW_SIZE) == CLASS_REQ_ACCEPTED )
				usbms_proc = USBMS_GET_SENSE_INFO_06;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
			break;
			
		case USBMS_GET_SENSE_INFO_06:
			status = sl811_status(CLASS_REQ_NONE);
			if( status == CLASS_STS_READY ) {
				usbms_proc = USBMS_READ_CAPACITY_01;
//				usbms_proc = USBMS_IDLE;
			} else if( status == CLASS_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
			
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Read Capacity                                                             *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USBMS_READ_CAPACITY_01:
			for(i=0; i<CBW_SIZE; i++) CmdBlkWrapper[i] = 0;
//			CmdBlkWrapper[ 0] = 0x43;	CmdBlkWrapper[ 1] = 0x42;	CmdBlkWrapper[ 2] = 0x53;	CmdBlkWrapper[ 3] = 0x55;
			CmdBlkWrapper[ 3] = 0x43;	CmdBlkWrapper[ 2] = 0x42;	CmdBlkWrapper[ 1] = 0x53;	CmdBlkWrapper[ 0] = 0x55;
			CmdBlkWrapper[ 8] = 0x08;
			CmdBlkWrapper[12] = MS_DATA_DIR_NONE;
			CmdBlkWrapper[14] = SCSI_LEN_READ_CAPACITY;
			CmdBlkWrapper[15] = SCSI_CMD_READ_CAPACITY;
			
#ifdef MSCLASS_DEBUG_ON
			printf("[MSC] ReadCapa\r\n");
#endif
			if( sl811_bulk_out_request(CmdBlkWrapper, CBW_SIZE) == CLASS_REQ_ACCEPTED )
				usbms_proc = USBMS_READ_CAPACITY_02;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
			break;
			
		case USBMS_READ_CAPACITY_02:
			status = sl811_status(CLASS_REQ_NONE);
			if( status == CLASS_STS_READY ) {
				usbms_proc = USBMS_READ_CAPACITY_03;
			} else if( status == CLASS_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
			
		case USBMS_READ_CAPACITY_03:
			if( sl811_bulk_in_request(TempBuffer, 8) == CLASS_REQ_ACCEPTED )
				usbms_proc = USBMS_READ_CAPACITY_04;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
			break;
			
		case USBMS_READ_CAPACITY_04:
			status = sl811_status(CLASS_REQ_NONE);
			if( status == CLASS_STS_READY ) {
				max_LBA     = TempBuffer[0] * 0x1000000 +
				              TempBuffer[1] * 0x10000   +
				              TempBuffer[2] * 0x100   +
				              TempBuffer[3] ;
				BlockSize   = TempBuffer[4] * 0x1000000 +
				              TempBuffer[5] * 0x10000   +
				              TempBuffer[6] * 0x100   +
				              TempBuffer[7] ;
				usbms_proc = USBMS_READ_CAPACITY_05;
			} else if( status == CLASS_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
			

		case USBMS_READ_CAPACITY_05:
			if( sl811_bulk_in_request(CmdStsWrapper, CSW_SIZE) == CLASS_REQ_ACCEPTED )
				usbms_proc = USBMS_READ_CAPACITY_06;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
			break;
			
		case USBMS_READ_CAPACITY_06:
			status = sl811_status(CLASS_REQ_NONE);
			if( status == CLASS_STS_READY ) {
				if( CmdStsWrapper[12] 	!= 0 ) {
					usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
					usbms_proc  = USBMS_ERROR_STOP;
				} else {
					usbms_proc = USBMS_IDLE;
//					usbms_proc = USBMS_READ10_01;
				}
			} else if( status == CLASS_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
			
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Idle                                                                      *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USBMS_IDLE:
			led_mode_set(0, LED_MODE_BLINK_1000MS);
			if( usbms_reset_request == 1 ) {
				usbms_proc = USBMS_RESET;
				usbms_reset_request = 0;
			}
			break;
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Read Sequence                                                             *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USBMS_READ10_01:
			for(i=0; i<CBW_SIZE; i++) CmdBlkWrapper[i] = 0;
			CmdBlkWrapper[ 3] = 0x43;	CmdBlkWrapper[ 2] = 0x42;	CmdBlkWrapper[ 1] = 0x53;	CmdBlkWrapper[ 0] = 0x55;
//			CmdBlkWrapper[ 8] = 0x00;	CmdBlkWrapper[ 9] = 0x02;
			CmdBlkWrapper[ 8] = ( (read10_param.transfer_length * BlockSize ) >>  0) & 0xFF;		// Length
			CmdBlkWrapper[ 9] = ( (read10_param.transfer_length * BlockSize ) >>  8) & 0xFF;		// Length
			CmdBlkWrapper[10] = ( (read10_param.transfer_length * BlockSize ) >> 16) & 0xFF;		// Length
			CmdBlkWrapper[11] = ( (read10_param.transfer_length * BlockSize ) >> 24) & 0xFF;		// Length
			CmdBlkWrapper[12] = MS_DATA_DIR_IN;
			CmdBlkWrapper[13] = 0x00;													// LUN
			CmdBlkWrapper[14] = 0x0A;
			CmdBlkWrapper[15] = SCSI_CMD_READ_10;
			CmdBlkWrapper[17] = (read10_param.logical_block_address >> 24) & 0xFF;
			CmdBlkWrapper[18] = (read10_param.logical_block_address >> 16) & 0xFF;
			CmdBlkWrapper[19] = (read10_param.logical_block_address >>  8) & 0xFF;
			CmdBlkWrapper[20] = (read10_param.logical_block_address >>  0) & 0xFF;
			CmdBlkWrapper[22] = (read10_param.transfer_length >> 8) & 0xFF;
			CmdBlkWrapper[23] = (read10_param.transfer_length >> 0) & 0xFF;
			
#ifdef MSCLASS_DEBUG_ON
			printf("[MSC] READ10\r\n");
#endif
			if( sl811_bulk_out_request(CmdBlkWrapper, CBW_SIZE) == CLASS_REQ_ACCEPTED )
				usbms_proc = USBMS_READ10_02;
			else
				usbms_disable_timer = 10;   // 10ms Wait for retry
			break;
			
		case USBMS_READ10_02:
			status = sl811_status(CLASS_REQ_NONE);
			if( status == CLASS_STS_READY ) {
				usbms_proc = USBMS_READ10_03;
			} else if( status == CLASS_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				//usbms_disable_timer = 10;   // 10ms Wait for retry
			}
			break;
			
		case USBMS_READ10_03:
			if( sl811_bulk_in_request(read10_param.p_buffer, 512) == CLASS_REQ_ACCEPTED )
				usbms_proc = USBMS_READ10_04;
			else
				usbms_disable_timer = 10;   // 10ms Wait for retry
			break;
			
		case USBMS_READ10_04:
			status = sl811_status(CLASS_REQ_NONE);
			if( status == CLASS_STS_READY ) {
				usbms_proc = USBMS_READ10_05;
			} else if( status == CLASS_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				//usbms_disable_timer = 10;   // 10ms Wait for retry
			}
			break;
			
		case USBMS_READ10_05:
			if( sl811_bulk_in_request(CmdStsWrapper, CSW_SIZE) == CLASS_REQ_ACCEPTED )
				usbms_proc = USBMS_READ10_06;
			else
				usbms_disable_timer = 10;   // 10ms Wait for retry
			break;
			
		case USBMS_READ10_06:
			status = sl811_status(CLASS_REQ_NONE);
			if( status == CLASS_STS_READY ) {
				usbms_proc = USBMS_IDLE;
			} else if( status == CLASS_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				//usbms_disable_timer = 10;   // 10ms Wait for retry
			}
			break;
			

		case USBMS_RESET:
			if( sl811_status(CLASS_REQ_RESET) == CLASS_STS_RESET_IN_PROGRESS ) {
				usbms_proc = USBMS_READY_WAIT;
			} else {
				usbms_disable_timer = 10;   // 10ms Wait for retry
			}
			break;
		
		case USBMS_ERROR_STOP:
			if( usbms_reset_request == 1 ) {
				usbms_proc = USBMS_RESET;
				usbms_reset_request = 0;
			}
			break;
	}
}

// -------------------------------------------
//  UI Function - USB-MS debug
// -------------------------------------------
int ui_function_usbms_debug(UI_COMMAND uicmd) {
	static unsigned char current_index;
	static unsigned char ok_press;
	int ret_val;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
		break;
		
	case UI_CMD_KEY_PRESS_OK:
		if( uicmd.param == OFF_EDGE )	ok_press = 0;
		else							ok_press = 1;
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		sc1602_set_buffer( 0, debug_var_table[current_index].display );
		switch(debug_var_table[current_index].type) {
			case 1:
				sc1602_set_buffer_variable1 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 2:
				sc1602_set_buffer_variable2 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 3:
				sc1602_set_buffer_variable3 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 4:
				sc1602_set_buffer_dump (1, debug_var_table[current_index].p_variable );
				break;
		}
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		ok_press = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( ok_press == 0 ) {
			if( debug_var_table[current_index+1].num !=  -1 ) current_index++;
		} else {
			usbms_reset_request = 1;
		}
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}

