﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Progressive.PecaStarter.Model;
using Progressive.PecaStarter.Model.Internal;
using Progressive.PecaStarter.ViewModel.Page;

namespace Progressive.PecaStarter.ViewModel.Command
{
    public class BroadcastCommand : ChannelCommandBase
    {
        public BroadcastCommand(Peercast peercast, Logger logger, IList<YellowPages> yellowPageses, MainWindowViewModel viewModel)
            : base(peercast, logger, yellowPageses, viewModel) { }

        #region ICommand メンバー
        protected override bool CanExecute()
        {
            if (peercast.IsBroadcast)
            {
                return false;
            }
            var ypvm = viewModel.YellowPagesesViewModel.SelectedYellowPages;
            if (ypvm == null)
            {
                return false;
            }
            if (!ypvm.IsAccepted) return false;
            var esvm = viewModel.ExternalSourceViewModel;
            if (string.IsNullOrEmpty(esvm.StreamUrl)) return false;
            if (!TryLength(esvm.StreamUrl)) return false;
            if (string.IsNullOrEmpty(esvm.Name.Value)) return false;
            if (!TryLength(esvm.Name.Value)) return false;
            if (!TryLength(esvm.Description.Value)) return false;
            if (!TryLength(esvm.ContactUrl.Value)) return false;
            if (!TryLength(esvm.Comment.Value)) return false;
            var yp = yellowPageses.Single(a => a.Name == ypvm.Name);
            if (yp is PeercastYellowPages)
            {
                if (!TryLength(ypvm.Prefix + esvm.Genre.Value)) return false;
                return true;
            }
            else
            {
                if (!TryLength(esvm.Genre.Value)) return false;
                foreach (var param in (yp as WebApiYellowPages).BroadcastParameters)
                {
                    if (WebApiDefine.ExcludeParameters.Contains(param))
                    {
                        continue;
                    }
                    if (!TryLength(ypvm[param])) return false;
                }
                return true;
            }
        }
        protected override async Task Execute()
        {
            viewModel.Save();
            var ypvm = viewModel.YellowPagesesViewModel.SelectedYellowPages;
            var yp = yellowPageses.Single(a => a.Name == ypvm.Name);
            if (yp.CanGetNoticeHash())
            {
                viewModel.Feedback = "規約の更新を確認中...";
                if (ypvm.AcceptedHash != await yp.GetNoticeHashAsync())
                {
                    ypvm.IsAccepted = false;
                    viewModel.Feedback = "中止";
                    viewModel.NotifyAlert("イエローページの規約が更新されています。規約を再確認してください。");
                    return;
                }
            }
            viewModel.Feedback = "通信中...";
            if (yp is PeercastYellowPages)
            {
                var result = await PeercastYpExecute(yp as PeercastYellowPages, ypvm, viewModel.ExternalSourceViewModel);
                if (result != Result.Success)
                {
                    viewModel.Feedback = "中止";
                    viewModel.NotifyAlert("チャンネルの作成時にエラーが発生しました。\n原因: " + GetErrorMessage(result));
                    return;
                }
            }
            else
            {
                var info = await WebApiYpExecute(yp as WebApiYellowPages, ypvm, viewModel.ExternalSourceViewModel);
                if (info.Result != Result.Success)
                {
                    viewModel.Feedback = "中止";
                    viewModel.NotifyAlert("チャンネルの作成時にエラーが発生しました。\n原因: " + GetErrorMessage(info));
                    return;
                }
            }
            if (viewModel.SettingsViewModel.Logging)
            {
                var esvm = viewModel.ExternalSourceViewModel;
                logger.Name = esvm.Name.Value;
                logger.StartAt = DateTime.Now;
                logger.insert("0", "0", esvm.Genre.Value, esvm.Description.Value, esvm.Comment.Value);
            }
            if (yp is WebApiYellowPages || viewModel.SettingsViewModel.Logging)
            {
                viewModel.BeginTimer();
            }
            viewModel.Feedback = "チャンネルを作成しました";
        }
        #endregion
        private async Task<Result> PeercastYpExecute(PeercastYellowPages pyp, YellowPagesViewModel yp, ExternalSourceViewModel es)
        {
            return (await peercast.Broadcast(pyp.Host, es.StreamUrl, es.Name.Value, yp.Prefix + es.Genre.Value, es.Description.Value, "WMV", es.ContactUrl.Value, es.Comment.Value, "", "", "", "", "")).Result;
        }
        private async Task<ResultInfo<string>> WebApiYpExecute(WebApiYellowPages wyp, YellowPagesViewModel ypvm, ExternalSourceViewModel esvm)
        {
            var info = await peercast.Broadcast("", esvm.StreamUrl, esvm.Name.Value, esvm.Genre.Value, esvm.Description.Value, "WMV", esvm.ContactUrl.Value, esvm.Comment.Value, "", "", "", "", "");
            if (info.Result != Result.Success)
            {
                return ResultInfo.Create(info.Result, "");
            }
            var wypInfo = await wyp.Broadcast(GetWebApiParameters(wyp.BroadcastParameters, info.Value.Item1, ypvm, esvm, "http://localhost:7144/pls/" + info.Value.Item2));
            if (wypInfo.Result != Result.Success)
            {
                await peercast.Stop();
                return wypInfo;
            }
            return wypInfo;
        }
    }
}
