﻿using System;
using System.Collections.Generic;
using System.ComponentModel;

namespace Progressive.PecaStarter.ViewModel
{
    public class ViewModelBase : INotifyPropertyChanged, IDisposable
    {
        private IDictionary<Tuple<INotifyPropertyChanged, string, string>, PropertyChangedEventHandler> relations;

        #region INotifyPropertyChanged メンバー
        public event PropertyChangedEventHandler PropertyChanged;
        #endregion

        public ViewModelBase()
        {
            relations = null;
        }

        protected void NotifyPropertyChanged(string name)
        {
            if (GetType().GetProperty(name) == null)
            {
                throw new ArgumentException();
            }
            if (PropertyChanged == null)
            {
                return;
            }
            PropertyChanged(this, new PropertyChangedEventArgs(name));
        }

        protected void AddRelation(INotifyPropertyChanged source, string destinationPropertyName)
        {
            AddRelation(source, null, destinationPropertyName);
        }
        protected void AddRelation(string sourcePropertyName, string destinationPropertyName)
        {
            AddRelation(this, sourcePropertyName, destinationPropertyName);
        }
        protected void AddRelation(INotifyPropertyChanged source, string sourcePropertyName, string destinationPropertyName)
        {
            if (sourcePropertyName != null && source.GetType().GetProperty(sourcePropertyName) == null)
            {
                throw new ArgumentException();
            }
            if (GetType().GetProperty(destinationPropertyName) == null)
            {
                throw new ArgumentException();
            }
            if (relations == null)
            {
                relations = new Dictionary<Tuple<INotifyPropertyChanged, string, string>, PropertyChangedEventHandler>();
            }
            var tuple = Tuple.Create(source, sourcePropertyName, destinationPropertyName);
            if (relations.ContainsKey(tuple))
            {
                return;
            }
            PropertyChangedEventHandler handler = (sender, e) =>
            {
                if (sourcePropertyName == null || e.PropertyName == sourcePropertyName)
                {
                    NotifyPropertyChanged(destinationPropertyName);
                }
            };
            source.PropertyChanged += handler;
            relations.Add(tuple, handler);
        }
        protected void RemoveRelation(INotifyPropertyChanged source, string destinationPropertyName)
        {
            RemoveRelation(this, null, destinationPropertyName);
        }
        protected void RemoveRelation(string sourcePropertyName, string destinationPropertyName)
        {
            RemoveRelation(this, sourcePropertyName, destinationPropertyName);
        }
        protected void RemoveRelation(INotifyPropertyChanged source, string sourcePropertyName, string destinationPropertyName)
        {
            var tuple = Tuple.Create(source, sourcePropertyName, destinationPropertyName);
            tuple.Item1.PropertyChanged -= relations[tuple];
            relations.Remove(tuple);
            if (relations.Count == 0)
            {
                relations = null;
            }
        }
        protected void RemoveAllRelation()
        {
            if (relations != null)
            {
                foreach (var relation in relations)
                {
                    relation.Key.Item1.PropertyChanged -= relation.Value;
                }
                relations.Clear();
                relations = null;
            }
        }

        #region IDisposable メンバー

        public virtual void Dispose()
        {
            RemoveAllRelation();
        }

        #endregion
    }
}
