/*
 * Created on 2005/07/01
 * 
 * Copyright (C) 2005- Monami software, LP.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Monami software, LP. - Initial API and implementation
 */
package jp.pizzafactory.debug.core;

import java.io.File;
import java.io.IOException;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import jp.pizzafactory.core.PathConverter;

import org.eclipse.cdt.debug.mi.core.IMIConstants;
import org.eclipse.cdt.debug.mi.core.IMITTY;
import org.eclipse.cdt.debug.mi.core.MIException;
import org.eclipse.cdt.debug.mi.core.MIProcess;
import org.eclipse.cdt.debug.mi.core.MIProcessAdapter;
import org.eclipse.cdt.debug.mi.core.MISession;
import org.eclipse.cdt.debug.mi.core.cdi.Session;
import org.eclipse.cdt.debug.mi.core.command.MIVersion;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

/**
 * The main plugin class to be used in the desktop.
 */
public class CorePlugin extends AbstractUIPlugin {
	//The shared instance.
	private static CorePlugin plugin;
	//Resource bundle.
	private ResourceBundle resourceBundle;

	/**
	 * The plug-in identifier of the Java core support
	 * (value <code>"jp.pizzafactory.debug.core"</code>).
	 */
	public static final String PLUGIN_ID = "jp.pizzafactory.debug.core" ; //$NON-NLS-1$

	// GDB init command file
	private static final String GDBINIT = ".gdbinit"; //$NON-NLS-1$

	// GDB command
	private static final String GDB = "gdb"; //$NON-NLS-1$

	/**
	 * The constructor.
	 */
	public CorePlugin() {
		super();
		plugin = this;
		try {
			resourceBundle = ResourceBundle.getBundle("jp.pizzafactory.debug.core.CorePluginResources"); //$NON-NLS-1$//
		} catch (MissingResourceException x) {
			resourceBundle = null;
		}
	}

	/**
	 * This method is called upon plug-in activation
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
	}

	/**
	 * This method is called when the plug-in is stopped
	 */
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
	}

	/**
	 * Returns the shared instance.
	 */
	public static CorePlugin getDefault() {
		return plugin;
	}

	/**
	 * Returns the string from the plugin's resource bundle,
	 * or 'key' if not found.
	 */
	public static String getResourceString(String key) {
		ResourceBundle bundle = CorePlugin.getDefault().getResourceBundle();
		try {
			return (bundle != null) ? bundle.getString(key) : key;
		} catch (MissingResourceException e) {
			return key;
		}
	}

	/**
	 * Returns the plugin's resource bundle,
	 */
	public ResourceBundle getResourceBundle() {
		return resourceBundle;
	}

	/**
	 * Convenience method which returns the unique identifier of this plugin.
	 */
	public static String getUniqueIdentifier() {
		if (getDefault() == null) {
			// If the default instance is not yet initialized,
			// return a static identifier. This identifier must
			// match the plugin id defined in plugin.xml
			return PLUGIN_ID;
		}
		return getDefault().getBundle().getSymbolicName();
	}

	/**
	 * Method createMISession.
	 * @param Process
	 * @param PTY
	 * @param int
	 * @param int
	 * @throws MIException
	 * @return MISession
	 */
	public MISession createMISession(MIProcess process, IMITTY pty, int timeout, int type, int launchTimeout) throws MIException {
		return new MISession(process, pty, type, timeout, launchTimeout, MIVersion.MI1, new NullProgressMonitor());
	}

	/**
	 * Method createMISession.
	 * @param Process
	 * @param PTY
	 * @param type
	 * @throws MIException
	 * @return MISession
	 */
	public MISession createMISession(MIProcess process, IMITTY pty, int type) throws MIException {
		CorePlugin miPlugin = getDefault();
		Preferences prefs = miPlugin.getPluginPreferences();
		int timeout = prefs.getInt(IMIConstants.PREF_REQUEST_TIMEOUT);
		int launchTimeout = prefs.getInt(IMIConstants.PREF_REQUEST_LAUNCH_TIMEOUT);
		return createMISession(process, pty, timeout, type, launchTimeout);
	}

	/**
	 * Method createCSession.
	 * @param program
	 * @param pid
	 * @return ICDISession
	 * @throws IOException
	 */
	private MISession _createCSession(String gdb, String miVersion, File program, File cwd, IProgressMonitor monitor) throws IOException, MIException {

		if (gdb == null || gdb.length() == 0) {
			gdb =  GDB;
		}

		if (monitor == null) {
			monitor = new NullProgressMonitor();
		}

		String[] args;
		if (program == null) {
			args = new String[] {gdb, "--cd="+cwd.getAbsolutePath(), "-nx", "--quiet", "-nw", "-i", miVersion}; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
		} else {
			String programPath = PathConverter.asShort(program.getAbsolutePath());
			args = new String[] {gdb, "--cd="+cwd.getAbsolutePath(), "-nx", "--quiet", "-nw", "-i", miVersion, programPath}; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
		}
		MIProcess pgdb = new MIProcessAdapter(args, monitor);
		MISession session;
		try {
			session = createMISession(pgdb, null, MISession.ATTACH);
		} catch (MIException e) {
			pgdb.destroy();
			throw e;
		}
		//@@@ We have to manually set the suspended state when we attach
		session.getMIInferior().setSuspended();
		session.getMIInferior().update();
		
		return session;
	}
	
	public HalfBakedSession createCSession(String gdb, String miVersion, File program, File cwd, IProgressMonitor monitor) throws IOException, MIException {
		MISession miSession = _createCSession(gdb, miVersion, program, cwd, monitor);
		Session session = new SingleThreadSession(miSession, true);
		HalfBakedSession hbSession = new HalfBakedSession(session);
		return hbSession;
	}
	
	public HalfBakedSession createMultiThreadCSession(String gdb, String miVersion, File program, File cwd, IProgressMonitor monitor) throws IOException, MIException {
		MISession miSession = _createCSession(gdb, miVersion, program, cwd, monitor);
		Session session = new Session(miSession);
		HalfBakedSession hbSession = new HalfBakedSession(session);
		return hbSession;
	}
}
