//
//  BookmarkFolderWindow.m
//  GalapagosReader
//
//  Created by Yoshitaka Sakamaki on 平成22/04/15.
//  Copyright 2010 MyHOME. All rights reserved.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//

#import "BookmarkFolderWindow.h"
#import "Database.h"


NSString * MA_Bookmark_TagName = @"TagName";
NSString * MA_Bookmark_Count = @"BookmarkCount";
NSString * MA_Bookmark_Title = @"Title";
NSString * MA_Bookmark_Date = @"Date";
NSString * MA_Bookmark_URL = @"URL";
NSString * MA_Bookmark_Comment = @"Comment";

static NSArray * folders = nil;
static NSMutableArray * bookmarkTagsArray = nil;
static NSMutableArray * bookmarksArray = nil;
static int tagsFolderRowIndex;
static int bookmarkFolderRowIndex;
static int categoryMark;

@interface BookmarkFolderWindow  (Private)
- (void) handleBookmarkListRefresh:(NSNotification *)nc;
- (void) handleBookmarkListEnterKeyDown:(NSNotification *)nc;
- (void) handleBookmarkListSelected:(NSNotification *)nc;
- (void)categoryMarkup:(id)sender;
@end

@implementation BookmarkFolderWindow

/* init
 * Initialize the class
 */
-(id)init
{
	if ((self = [super init]) != nil)
	{
		link = nil;
		bookmarkFolderWindow = nil;
		tagsFolderRowIndex = 0;
		bookmarkFolderRowIndex = 0;
		//タグリスト選択時に呼び出される。
		//１つのタグに対応するブックマークリストを表示させるためのリフレッシュメソッド。
		NSNotificationCenter * nc = [NSNotificationCenter defaultCenter];
		[nc addObserver:self selector:@selector(handleBookmarkListRefresh:) 
				   name:@"MA_Notify_BookmarkListRefresh" object:nil];
		
		[nc addObserver:self selector:@selector(handleBookmarkListEnterKeyDown:) 
				   name:@"MA_Notify_BookmarkListEnterKeyDown" object:nil];
		
		[nc addObserver:self selector:@selector(handleBookmarkListSelected:) 
				   name:@"MA_Notify_BookmarkListSelected" object:nil];

	}
	
	return self;
}

- (void)awakeFromNib
{
	[bookmarkTable setTarget:self];
	[bookmarkTable setDoubleAction:@selector(actionDoubleClick:)];
	
	//検索メニューのセットアップ
	NSMenu *cellMenu = [[[NSMenu alloc] init] autorelease];
    NSMenuItem *item;
	
    item = [[[NSMenuItem alloc] initWithTitle:@"全体から検索"
                                       action:@selector(categoryMarkup:)
                                keyEquivalent:@""] autorelease];
    [item setTarget:self];
    [item setTag:0];
    [cellMenu insertItem:item atIndex:0];
	
    item = [[[NSMenuItem alloc] initWithTitle:@"タイトルから検索"
                                       action:@selector(categoryMarkup:)
                                keyEquivalent:@""] autorelease];
    [item setTarget:self];
    [item setTag:1];
    [cellMenu insertItem:item atIndex:1];
	
    item = [[[NSMenuItem alloc] initWithTitle:@"コメントから検索"
                                       action:@selector(categoryMarkup:)
                                keyEquivalent:@""] autorelease];
    [item setTarget:self];
    [item setTag:2];
    [cellMenu insertItem:item atIndex:2];
	
    item = [[[NSMenuItem alloc] initWithTitle:@"URLから検索"
                                       action:@selector(categoryMarkup:)
                                keyEquivalent:@""] autorelease];
    [item setTarget:self];
    [item setTag:3];
    [cellMenu insertItem:item atIndex:3];
	
    id searchCell = [searchField cell];
    [searchCell setSearchMenuTemplate:cellMenu];

}

-(void)showBookmarkFolderSheet:(NSWindow*)window didEndSelector:(SEL)endSelector delegate:(id)theDelegate
{
	if (!bookmarkFolderWindow)
		[NSBundle loadNibNamed:@"BookmarkFolder" owner:self];
	
	handler = endSelector;
	delegate = theDelegate;
	
	if ([[self window] firstResponder] == tagsTable)
	{
		//タグ側のテーブルをリロードさせる。
		[tagsTable reloadTagsFolder];
		
		//ブックマーク側のテーブルをリロードさせる。
		[bookmarkTable reloadBookmarksFolder];		
	} else {
		//タグ側のテーブルをリロードさせる。
		[tagsTable reloadTagsFolder];
	}
	
	
	//コメント＆URLフィールドの値を表示させる。
	[self handleBookmarkListSelected:nil];		
	
	[NSApp beginSheet:bookmarkFolderWindow modalForWindow:window modalDelegate:nil didEndSelector:nil contextInfo:nil];

}

/* link
 * ブラウザに表示するために必要なURLを返す
 */
-(NSString *)link
{
	return link;
}

- (void) handleBookmarkListRefresh:(NSNotification *)nc
{
		
	if (tagsFolderRowIndex >=0)
		[bookmarkTable reloadBookmarksFolder];

}

/* handleBookmarkListEnterKeyDown
 * ブックマークテーブルの列に対して、Enterキーを押下すると呼び出される
 */
- (void) handleBookmarkListEnterKeyDown:(NSNotification *)nc
{
	if( [bookmarkTable selectedRow]>=0 )
	{
		NSDictionary * bookmark = [bookmarksArray objectAtIndex:[bookmarkTable selectedRow]];
		if (link)
			[link release];
		
		link = [bookmark objectForKey:MA_Bookmark_URL];
		[link retain];
		
		//NSLog(@"%@", link);
		[NSApp endSheet:bookmarkFolderWindow];
		[bookmarkFolderWindow orderOut:self];
		
		//AppControllerにWebブラウザを開くよう伝える。
		[[NSRunLoop currentRunLoop] performSelector:handler 
											 target:delegate 
										   argument:self 
											  order:0 
											  modes:[NSArray arrayWithObjects:NSDefaultRunLoopMode, nil]];
		
	}
}


/* handleBookmarkListSelected
 * 一件のブックマーク情報を表示する
 */
- (void) handleBookmarkListSelected:(NSNotification *)nc
{
	if( [bookmarkTable selectedRow]>=0 )
	{
		NSDictionary * bookmark = [bookmarksArray objectAtIndex:bookmarkFolderRowIndex];
		[urlField setStringValue:[bookmark objectForKey:MA_Bookmark_URL]];
		[commentField setStringValue:[bookmark objectForKey:MA_Bookmark_Comment]];		
	}
}

/* actionDoubleClick
 * ブックマークテーブルの列をダブルクリックした際に呼び出される
 */
- (IBAction) actionDoubleClick:(id)sender
{
	// ヘッダクリック時以外。
	if( [bookmarkTable clickedRow]>=0 )
	{
		NSDictionary * bookmark = [bookmarksArray objectAtIndex:[bookmarkTable clickedRow]];
		if (link)
			[link release];
		
		link = [bookmark objectForKey:MA_Bookmark_URL];
		[link retain];
		
		//NSLog(@"%@", link);
		[NSApp endSheet:bookmarkFolderWindow];
		[bookmarkFolderWindow orderOut:self];
		
		//AppControllerにWebブラウザを開くよう伝える。
		[[NSRunLoop currentRunLoop] performSelector:handler 
											 target:delegate 
										   argument:self 
											  order:0 
											  modes:[NSArray arrayWithObjects:NSDefaultRunLoopMode, nil]];
		
	}
}

-(IBAction)closeBookmarkFolder:(id)sender
{	
	[NSApp endSheet:bookmarkFolderWindow];
	[bookmarkFolderWindow orderOut:self];
}

/* keyDown
 * 閉じるボタン上でEnterキー押下の動作を定義している。
 */
- (void)keyDown:(NSEvent *)theEvent
{
	unichar unicodeKey;
	unicodeKey = [ [ theEvent characters ] characterAtIndex:0 ];
	
	switch (unicodeKey) {
		case NSCarriageReturnCharacter:
		case NSEnterCharacter:
			if ([[self window] firstResponder] == closeButton) {
				[self closeBookmarkFolder:self];
			}
			return;
			
		case NSRightArrowFunctionKey:
			if ([[self window] firstResponder] == closeButton) {
				[[self window] makeFirstResponder:[closeButton nextKeyView]];
			}
			return;
			
		case NSLeftArrowFunctionKey:
			if ([[self window] firstResponder] == closeButton) {
				[[self window] makeFirstResponder:[closeButton previousKeyView]];
			}			
			return;
			
		case 'f':
		case 'F':
			[[self window] makeFirstResponder:searchField];
			return;
			
		default:
			break;
	}
	[super keyDown:theEvent];	
}

/* categoryMarkup
 * 検索メニュー選択時に呼ばれる
 */
- (void)categoryMarkup:(id)sender
{
    //int saveState = [sender state];
    //チェックオフ
	NSMenu *menu = [sender menu];
	[[menu itemWithTag:categoryMark] setState:NSOffState];
	
    //チェックオン
	[sender setState:NSOnState];
	categoryMark = [sender tag];
	switch ([sender tag]) {
		case 0:
			[[searchField cell] setPlaceholderString:@"全体から検索 （'F'）"];
			break;
		case 1:
			[[searchField cell] setPlaceholderString:@"タイトルから検索 （'F'）"];
			break;
		case 2:
			[[searchField cell] setPlaceholderString:@"コメントから検索 （'F'）"];
			break;
		case 3:
			[[searchField cell] setPlaceholderString:@"URLから検索 （'F'）"];
			break;
		default:
			abort();
			break;
	}
}

// searchFieldのデリゲート。
- (void)controlTextDidEndEditing:(NSNotification *)aNotification
{
	[self updateFilter];
	[self handleBookmarkListSelected:nil];
	if (0 != [bookmarksArray count])
		[[self window] makeFirstResponder:bookmarkTable];
}

/* updateFilter
 * 検索カテゴリを指定して実行を依頼する。
 */
- (void)updateFilter
{
	if (0 < [[searchField stringValue] length])
	{
		switch (categoryMark) {
			case MA_Search_ALL:
				[bookmarkTable reloadBookmarksFolder:[searchField stringValue]
													:MA_Search_ALL];
				break;
				
			case MA_Search_Title:
				[bookmarkTable reloadBookmarksFolder:[searchField stringValue]
													:MA_Search_Title];
				break;
				
			case MA_Search_Comment:
				[bookmarkTable reloadBookmarksFolder:[searchField stringValue]
													:MA_Search_Comment];
				break;
				
			case MA_Search_URL:
				[bookmarkTable reloadBookmarksFolder:[searchField stringValue]
													:MA_Search_URL];
				break;
				
			default:
				break;
		}		
	}	
}

-(void)dealloc
{
	[link release];
	[bookmarksArray release];
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	[super dealloc];
}
@end



@implementation TagsTableView
/* init
 * Initialize the class
 */
-(id)init
{
	if ((self = [super init]) != nil)
	{
	}
	
	return self;
}

-(void)reloadTagsFolder
{
	if (folders)
		[folders release];
	folders = [[Database sharedDatabase] arrayOfAllFoldersHatena];
	[folders retain];

	if (bookmarkTagsArray)
		[bookmarkTagsArray release];
	bookmarkTagsArray = [[NSMutableArray array] retain];
	
	for (BookmarkFolder * folder in folders)
	{
		NSArray *objArr = [NSArray arrayWithObjects:
						   [folder name],
						   [NSNumber numberWithInt:[folder bookmarkCount]],
						   nil];					
		NSArray *keyArr = [NSArray arrayWithObjects:
						   MA_Bookmark_TagName,
						   MA_Bookmark_Count,
						   nil];
		NSDictionary *tagsDict = 
		[[NSDictionary alloc] initWithObjects:objArr forKeys:keyArr];
		[bookmarkTagsArray addObject:tagsDict];
		[tagsDict release];
	}
	
	// 初期表示はタグ名の昇順に並べてセットする
	NSSortDescriptor *sortdesc = [[NSSortDescriptor alloc] initWithKey:MA_Bookmark_TagName
															 ascending:YES
															  selector:@selector(compare:)];
	
	[bookmarkTagsArray sortUsingDescriptors:[NSArray arrayWithObject:sortdesc]];
	[sortdesc release];
	
	// 表示画面をリフレッシュ
	[self reloadData];
}

/* numberOfRowsInTableView [datasource]
 * Datasource for the table view. Return the total number of rows we'll display which
 * is equivalent to the number of articles in the current folder.
 */
-(int)numberOfRowsInTableView:(NSTableView *)aTableView
{
	return [folders count];
}

/* objectValueForTableColumn [datasource]
 * Called by the table view to obtain the object at the specified column and row. This is
 * called often so it needs to be fast.
 */
-(id)tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	NSParameterAssert(rowIndex >= 0 && rowIndex < (int)[folders count]);
	NSString * identifier = [aTableColumn identifier];
	if ([identifier isEqualToString:MA_Bookmark_TagName])
	{
		return [[bookmarkTagsArray objectAtIndex:rowIndex] objectForKey:MA_Bookmark_TagName];
	}
	else if ([identifier isEqualToString:MA_Bookmark_Count])
	{
		return [[bookmarkTagsArray objectAtIndex:rowIndex] objectForKey:MA_Bookmark_Count];
	}
	else
	{
		[NSException raise:@"TagsTableView unknown table column identifier exception" format:@"Unknown table column identifier: %@", identifier];
	}
	
    return nil;
}

- (void)tableView:(NSTableView *)tableView sortDescriptorsDidChange:(NSArray *)oldDescriptors
{
	[bookmarkTagsArray sortUsingDescriptors:[tableView sortDescriptors]];
	[tableView reloadData];
} 

- (BOOL)tableView:(NSTableView *)aTableView shouldSelectRow:(int)rowIndex
{	
	//NSLog(@"%d", [aTableView selectedRow]);
	tagsFolderRowIndex = rowIndex;
	[[NSNotificationCenter defaultCenter] postNotificationName:@"MA_Notify_BookmarkListRefresh" object:nil];

	//コメント＆URLフィールドの値をリロードする。
	[[NSNotificationCenter defaultCenter] postNotificationName:@"MA_Notify_BookmarkListSelected" object:nil];

	// CPU率があがらないように少しsleep
	[NSThread sleepForTimeInterval:0.02f];
	return YES;
}

/* keyDown
 * 以下のキーに対して動作を割り付けている
 * ・左右のカーソルキー
 * ・エンターキー
 */
- (void)keyDown:(NSEvent *)theEvent
{
	unichar unicodeKey;
	unicodeKey = [[theEvent characters] characterAtIndex:0];

	switch (unicodeKey) {
		case NSCarriageReturnCharacter:
		case NSEnterCharacter:
			[[self window] makeFirstResponder:[self nextKeyView]];
			return;
			
		case NSRightArrowFunctionKey:
			[[self window] makeFirstResponder:[self nextKeyView]];
			return;
		
		case NSPageUpFunctionKey:
		case NSPageDownFunctionKey:
		case NSUpArrowFunctionKey:
		case NSDownArrowFunctionKey:
		case '\x1b'://esc
		case '\t'://tab
		case '\x19'://shift-tab
			[super keyDown:theEvent];
			return;
			
		case 'f':
		case 'F':
			[[self window] keyDown:theEvent];
			return;
			
		default:
			break;
	}
}

-(void)dealloc
{
	[bookmarkTagsArray release];
	[folders release];
	[super dealloc];
}

@end


@implementation BookmarkTableView
static NSDictionary * tagFolderDict;
/* init
 * Initialize the class
 */
-(id)init
{
	if ((self = [super init]) != nil)
	{
		tagFolderDict = nil;
	}
	
	return self;
}

-(void)reloadBookmarksFolder
{
	if (tagFolderDict)
		[tagFolderDict release];
	
	if (bookmarksArray)
		[bookmarksArray release];
	bookmarksArray = [[NSMutableArray array] retain];

	tagFolderDict = [[bookmarkTagsArray objectAtIndex:tagsFolderRowIndex] retain];
	for (BookmarkFolder * folder in folders)
	{
		if ([[folder name] isEqualToString:[tagFolderDict objectForKey:MA_Bookmark_TagName]])
		{
			for (BookmarkTag * tag in [folder bookmarkTags])
			{
				for (Bookmark * bookmark in [tag bookmarks])
				{
					[bookmark retain];
					NSArray *objArr = [NSArray arrayWithObjects:
									   [bookmark bookmarkTitle],
									   [bookmark tagBookmarkDate],
									   [bookmark bookmarkUrl],
									   [bookmark bookmarkComment],
									   nil];					
					NSArray *keyArr = [NSArray arrayWithObjects:
									   MA_Bookmark_Title, 
									   MA_Bookmark_Date,
									   MA_Bookmark_URL,
									   MA_Bookmark_Comment,
									   nil];
					NSDictionary *bookmarkDict = 
					[[NSDictionary alloc] initWithObjects:objArr forKeys:keyArr];
					[bookmarksArray addObject:bookmarkDict];
					[bookmarkDict release];
					[bookmark release];
				}
			}			
		}
	}
		
	// 初期表示は日付の降順に並べてセットする
	NSSortDescriptor *sortdesc = [[NSSortDescriptor alloc] initWithKey:MA_Bookmark_Date
															 ascending:NO
															  selector:@selector(compare:)];
	
	[bookmarksArray sortUsingDescriptors:[NSArray arrayWithObject:sortdesc]];
	[sortdesc release];
	
	// 表示画面をリフレッシュ
	[self reloadData];
}

/* reloadBookmarksFolder
 * 検索実行時に呼び出されるリロードメソッド。
 */
-(void)reloadBookmarksFolder:(NSString*)searchString
							:(SearchBookmarkKind)searchKind
{	
	if (bookmarksArray)
		[bookmarksArray release];
	bookmarksArray = [[NSMutableArray array] retain];
	
	for (BookmarkFolder * folder in folders)
	{
		for (BookmarkTag * tag in [folder bookmarkTags])
		{
			for (Bookmark * bookmark in [tag bookmarks])
			{
				NSRange rng;
				switch (searchKind) {
					case MA_Search_ALL:
						rng = [[bookmark searchString] rangeOfString:searchString
															 options:NSCaseInsensitiveSearch];
						break;
						
					case MA_Search_Title:
						rng = [[bookmark bookmarkTitle] rangeOfString:searchString
															  options:NSCaseInsensitiveSearch];
						break;
						
					case MA_Search_Comment:
						rng = [[bookmark bookmarkComment] rangeOfString:searchString
																options:NSCaseInsensitiveSearch];
						break;
						
					case MA_Search_URL:
						rng = [[bookmark bookmarkUrl] rangeOfString:searchString
															options:NSCaseInsensitiveSearch];
						break;
						
					default:
						break;
				}
				
				if (NSNotFound != rng.location)
				{
					NSArray *objArr = [NSArray arrayWithObjects:
									   [bookmark bookmarkTitle],
									   [bookmark tagBookmarkDate],
									   [bookmark bookmarkUrl],
									   [bookmark bookmarkComment],
									   nil];					
					NSArray *keyArr = [NSArray arrayWithObjects:
									   MA_Bookmark_Title, 
									   MA_Bookmark_Date,
									   MA_Bookmark_URL,
									   MA_Bookmark_Comment,
									   nil];
					NSDictionary *bookmarkDict = 
					[[NSDictionary alloc] initWithObjects:objArr forKeys:keyArr];
					
					//追加した要素のURLと比較し、同一であれば配列に追加しないようにする。
					BOOL findFlag = NO;
					for (NSDictionary * arrayDict in bookmarksArray)
					{
						if (YES == [[arrayDict objectForKey:MA_Bookmark_URL] 
								   isEqualToString:[bookmarkDict objectForKey:MA_Bookmark_URL]])
						{
							findFlag = YES;
							break;
						}
					}
					
					if (NO == findFlag)
						[bookmarksArray addObject:bookmarkDict];
					
					[bookmarkDict release];
				}
			}
		}			
	}
	
	// 初期表示は日付の降順に並べてセットする
	NSSortDescriptor *sortdesc = [[NSSortDescriptor alloc] initWithKey:MA_Bookmark_Date
															 ascending:NO
															  selector:@selector(compare:)];
	
	[bookmarksArray sortUsingDescriptors:[NSArray arrayWithObject:sortdesc]];
	[sortdesc release];
	
	// 表示画面をリフレッシュ
	[self reloadData];
	
}

/* numberOfRowsInTableView [datasource]
 * Datasource for the table view. Return the total number of rows we'll display which
 * is equivalent to the number of articles in the current folder.
 */
-(int)numberOfRowsInTableView:(NSTableView *)aTableView
{
	//return [[tagFolderDict objectForKey:MA_Bookmark_Count] intValue];
	//NSLog(@"numberOfRowsInTableView = %d",[bookmarksArray count]);
	return [bookmarksArray count];
}

/* objectValueForTableColumn [datasource]
 * Called by the table view to obtain the object at the specified column and row. This is
 * called often so it needs to be fast.
 */
-(id)tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	/*
	NSLog(@"rowIndex = %d",rowIndex);
	NSLog(@"bookmarksArray = %d",[bookmarksArray count]);
	NSLog(@"%@",[[bookmarksArray objectAtIndex:rowIndex] objectForKey:MA_Bookmark_Title]);
	NSLog(@"%@",[[bookmarksArray objectAtIndex:rowIndex] objectForKey:MA_Bookmark_Date]);
	 */
	NSParameterAssert(rowIndex >= 0 && rowIndex < (int)[bookmarksArray count]);
	NSString * identifier = [aTableColumn identifier];
	if ([identifier isEqualToString:MA_Bookmark_Title])
	{
		return 	[[bookmarksArray objectAtIndex:rowIndex] objectForKey:MA_Bookmark_Title];
	}
	else if ([identifier isEqualToString:MA_Bookmark_Date])
	{
		return 	[[bookmarksArray objectAtIndex:rowIndex] objectForKey:MA_Bookmark_Date];
	}		
	else
	{
		[NSException raise:@"BookmarkTableView unknown table column identifier exception" format:@"Unknown table column identifier: %@", identifier];
	}
	
    return nil;
}

- (void)tableView:(NSTableView *)tableView sortDescriptorsDidChange:(NSArray *)oldDescriptors
{
	[bookmarksArray sortUsingDescriptors:[tableView sortDescriptors]];
	[tableView reloadData];
} 

- (BOOL)tableView:(NSTableView *)aTableView shouldSelectRow:(int)rowIndex
{	
	bookmarkFolderRowIndex = rowIndex;
	// コメント＆URLフィールドをリロードさせる。
	[[NSNotificationCenter defaultCenter] postNotificationName:@"MA_Notify_BookmarkListSelected" object:nil];
	
	// CPU率があがらないように少しsleep
	[NSThread sleepForTimeInterval:0.02f];

	return YES;
}


/* keyDown
 * 行に対するEnterキー押下時の動作を定義している。
 */
- (void)keyDown:(NSEvent *)theEvent
{
	unichar unicodeKey;
	unicodeKey = [[theEvent characters] characterAtIndex:0];

	switch (unicodeKey) {
		case NSCarriageReturnCharacter:
		case NSEnterCharacter:
			[[NSNotificationCenter defaultCenter] postNotificationName:@"MA_Notify_BookmarkListEnterKeyDown" object:nil];
			return;
			
		case NSLeftArrowFunctionKey:
		case '\x19'://shift-tab
			[[self window] makeFirstResponder:[self previousKeyView]];
			[[NSNotificationCenter defaultCenter] postNotificationName:@"MA_Notify_BookmarkListRefresh" object:nil];
			[[NSNotificationCenter defaultCenter] postNotificationName:@"MA_Notify_BookmarkListSelected" object:nil];

			return;

		case 'f':
		case 'F':
			[[self window] keyDown:theEvent];
			return;
			
		case NSPageUpFunctionKey:
		case NSPageDownFunctionKey:
		case NSUpArrowFunctionKey:
		case NSDownArrowFunctionKey:
		case '\x1b'://esc
		case '\t'://tab
			[super keyDown:theEvent];
			return;
			
		default:
			break;
	}	
}

-(void)dealloc
{
	[tagFolderDict release];
	[super dealloc];
}
@end
