//
//  FoldersTreeHatena.m
//  GalapagosReader
//
//  Created by poi-poi-capsule on 平成22/03/10.
//  Copyright 2010 MyHOME. All rights reserved.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//

#import "FoldersTreeHatena.h"

@interface FoldersTreeHatena (Private)
-(void)setFolderListFont;
-(NSArray *)archiveState;
-(void)unarchiveState:(NSArray *)stateArray;
-(void)reloadDatabase:(NSArray *)stateArray;
-(BOOL)loadTree:(NSArray *)listOfFolders rootNode:(TreeNode *)node;
-(void)setManualSortOrderForNode:(TreeNode *)node;
-(void)handleDoubleClick:(id)sender;
-(void)handleAutoSortFoldersTreeChange:(NSNotification *)nc;
-(void)handleFolderAdded:(NSNotification *)nc;
-(void)handleFolderNameChange:(NSNotification *)nc;
-(void)handleFolderUpdate:(NSNotification *)nc;
-(void)handleFolderDeleted:(NSNotification *)nc;
-(void)handleShowFolderImagesChange:(NSNotification *)nc;
-(void)handleFolderFontChange:(NSNotification *)nc;
-(void)reloadFolderItem:(id)node reloadChildren:(BOOL)flag;
-(void)expandToParent:(TreeNode *)node;
-(BOOL)copyTableSelection:(NSArray *)items toPasteboard:(NSPasteboard *)pboard;
-(BOOL)moveFolders:(NSArray *)array;
-(void)enableFoldersRenaming:(id)sender;
-(void)enableFoldersRenamingAfterDelay;
@end

@implementation FoldersTreeHatena


-(id)initWithFrame:(NSRect)frameRect
{
	if ((self = [super initWithFrame:frameRect]) != nil)
	{
	}
	return self;
}

/* saveFolderSettings
 * Preserve the expanded/collapsed and selection state of the folders list
 * into the user's preferences.
 */
-(void)saveFolderSettings
{
	[[Preferences standardPreferences] setArray:[self archiveState] forKey:MAPref_FolderStatesHatena];
}

-(void)initialiseFoldersTree
{
	// Want tooltips
	[outlineView setEnableTooltips:YES];
	
	// Set the menu for the popup button
	[outlineView setMenu:[[NSApp delegate] folderMenu]];
	
	blockSelectionHandler = YES;
	[self reloadDatabase:[[Preferences standardPreferences] arrayForKey:MAPref_FolderStatesHatena]];
	blockSelectionHandler = NO;
	
	// Register for notifications
	NSNotificationCenter * nc = [NSNotificationCenter defaultCenter];
	[nc addObserver:self selector:@selector(handleFolderUpdate:) name:@"MA_Notify_FoldersUpdated_Hatena" object:nil];
	[nc addObserver:self selector:@selector(handleFolderNameChange:) name:@"MA_Notify_FolderNameChanged_Hatena" object:nil];
	[nc addObserver:self selector:@selector(handleFolderAdded:) name:@"MA_Notify_FolderAdded_Hatena" object:nil];
	[nc addObserver:self selector:@selector(handleFolderDeleted:) name:@"MA_Notify_FolderDeleted_Hatena" object:nil];
	[nc addObserver:self selector:@selector(autoCollapseFolder:) name:@"MA_Notify_AutoCollapseFolder_Hatena" object:nil];
	[nc addObserver:self selector:@selector(handleFolderFontChange:) name:@"MA_Notify_FolderFontChange_Hatena" object:nil];
	[nc addObserver:self selector:@selector(handleShowFolderImagesChange:) name:@"MA_Notify_ShowFolderImages_Hatena" object:nil];
	[nc addObserver:self selector:@selector(handleAutoSortFoldersTreeChange:) name:@"MA_Notify_AutoSortFoldersTreeChange_Hatena" object:nil];
}

-(void)awakeFromNib
{
	NSTableColumn * tableColumn;
	ImageAndTextCell * imageAndTextCell;
	
	// Our folders have images next to them.
	tableColumn = [outlineView tableColumnWithIdentifier:@"folderColumns"];
	imageAndTextCell = [[[ImageAndTextCell alloc] init] autorelease];
	[imageAndTextCell setEditable:YES];
	[tableColumn setDataCell:imageAndTextCell];
	
	// Folder image
	//folderErrorImage = [NSImage imageNamed:@"folderError.tiff"];
	//refreshProgressImage = [NSImage imageNamed:@"refreshProgress.tiff"];
	
	// Create and set whatever font we're using for the folders
	[self setFolderListFont];
	
	// Set background colour
	[outlineView setBackgroundColor:[NSColor colorWithCalibratedRed:0.84 green:0.87 blue:0.90 alpha:1.00]];
	
	// Allow double-click a node to edit the node
	[outlineView setAction:@selector(handleSingleClick:)];
	[outlineView setDoubleAction:@selector(handleDoubleClick:)];
	[outlineView setTarget:self];
	
	// Initially size the outline view column to be the correct width
	[outlineView sizeLastColumnToFit];
	
	// Don't resize the column when items are expanded as this messes up
	// the placement of the unread count button.
	[outlineView setAutoresizesOutlineColumn:NO];
	
	// Register for dragging
	[outlineView registerForDraggedTypes:[NSArray arrayWithObjects:MA_PBoardType_FolderList, MA_PBoardType_RSSSource, @"WebURLsWithTitlesPboardType", NSStringPboardType, nil]]; 
	[outlineView setVerticalMotionCanBeginDrag:YES];
	
	// Make sure selected row is visible
	[outlineView scrollRowToVisible:[outlineView selectedRow]];
}

-(void)outlineView:(NSOutlineView *)olv willDisplayCell:(NSCell *)cell forTableColumn:(NSTableColumn *)tableColumn item:(id)item 
{
	if ([[tableColumn identifier] isEqualToString:@"folderColumns"]) 
	{
		TreeNode * node = (TreeNode *)item;
		Folder * folder = [node folder];
		ImageAndTextCell * realCell = (ImageAndTextCell *)cell;
		
		// Set the colour of the text in the cell
		int rowIndex = [olv rowForItem:item];
		NSColor * textColor;
		
		if (IsUnsubscribed(folder))
			textColor = [NSColor grayColor];
		else if ([olv editedRow] == rowIndex)
			textColor = [NSColor blackColor];
		else if ([olv selectedRow] == rowIndex)
			textColor = [NSColor whiteColor];
		else
			textColor = [NSColor blackColor];
		[realCell setTextColor:textColor];
		
		[realCell setItem:item];
		
		// Use the auxiliary position of the feed item to show
		// the refresh icon if the feed is being refreshed, or an
		// error icon if the feed failed to refresh last time.
		if (IsUpdating(folder))
		{
			[realCell setAuxiliaryImage:nil];
			[realCell setInProgress:YES];
		}
		/*
		else if (IsError(folder))
		{
			[realCell setAuxiliaryImage:folderErrorImage];
			[realCell setInProgress:NO];
		}
		 */
		else
		{
			[realCell setAuxiliaryImage:nil];
			[realCell setInProgress:NO];
		}
		
		if (IsSmartFolder(folder))  // Because if the search results contain unread articles we don't want the smart folder name to be bold.
		{
			[realCell clearCount];
			[realCell setFont:cellFont];
		}
		else if ([folder unreadCount])
		{
			[realCell setFont:boldCellFont];
			[realCell setCount:[folder unreadCount]];
			[realCell setCountBackgroundColour:[NSColor colorForControlTint:[NSColor currentControlTint]]];
		}
		else if ([folder childUnreadCount] && ![olv isItemExpanded:item])
		{
			[realCell setFont:boldCellFont];
			[realCell setCount:[folder childUnreadCount]];
			[realCell setCountBackgroundColour:[NSColor colorForControlTint:[NSColor currentControlTint]]];
		}
		else
		{
			[realCell clearCount];
			[realCell setFont:cellFont];
		}
		
		// Only show folder images if the user prefers them.
		Preferences * prefs = [Preferences standardPreferences];
		[realCell setImage:([prefs showFolderImages] ? [folder image] : [folder standardImage])];
	}
}


/* loadTree
 * Recursive routine that populates the folder list
 */
-(BOOL)loadTree:(NSArray *)listOfFolders rootNode:(TreeNode *)node
{
	// saka
	Folder * folder;
	if ([[Preferences standardPreferences] foldersTreeSortMethod] != MA_FolderSort_Manual)
	{
		for (folder in listOfFolders)
		{
			//saka add
			if (MA_Hatena_Folder != [folder type])
				continue;			
			
			int itemId = [folder itemId];
			NSArray * listOfSubFolders = [[Database sharedDatabase] arrayOfFolders:itemId];
			int count = [listOfSubFolders count];
			TreeNode * subNode;
			
			subNode = [[TreeNode alloc] init:node atIndex:-1 folder:folder canHaveChildren:(count > 0)];
			if (count)
				[self loadTree:listOfSubFolders rootNode:subNode];
			
			[subNode release];
		}
	}
	else
	{
		NSArray * listOfFolderIds = [listOfFolders valueForKey:@"itemId"];
		int index = 0;
		int nextChildId = (node == rootNode) ? [[Database sharedDatabase] firstFolderId] : [[node folder] firstChildId];
		while (nextChildId > 0)
		{
			unsigned int listIndex = [listOfFolderIds indexOfObject:[NSNumber numberWithInt:nextChildId]];
			if (listIndex == NSNotFound)
			{
				NSLog(@"Cannot find child with id %i for folder with id %i", nextChildId, [node nodeId]);
				return NO;
			}
			folder = [listOfFolders objectAtIndex:listIndex];
			NSArray * listOfSubFolders = [[Database sharedDatabase] arrayOfFolders:nextChildId];
			int count = [listOfSubFolders count];
			TreeNode * subNode;
			
			subNode = [[TreeNode alloc] init:node atIndex:index folder:folder canHaveChildren:(count > 0)];
			if (count)
			{
				if (![self loadTree:listOfSubFolders rootNode:subNode])
				{
					[subNode release];
					return NO;
				}
			}
			[subNode release];
			nextChildId = [folder nextSiblingId];
			++index;
		}
		if (index < (int)[listOfFolders count])
		{
			NSLog(@"Missing children for folder with id %i", nextChildId, [node nodeId]);
			return NO;
		}
	}
	return YES;
}

/* renameFolder
 * Begin in-place editing of the selected folder name.
 */
-(void)renameFolder:(int)folderId
{
	//saka TODO:はてブのフォルダ名置換apiを呼ばないといけない。
	TreeNode * node = [rootNode nodeFromID:folderId];
	int rowIndex = [outlineView rowForItem:node];
	
	if (rowIndex != -1)
	{
		[outlineView selectRowIndexes:[NSIndexSet indexSetWithIndex:(unsigned int)rowIndex] byExtendingSelection:NO];
		[outlineView editColumn:[outlineView columnWithIdentifier:@"folderColumns"] row:rowIndex withEvent:nil select:YES];
	}
}

/* handleAutoSortFoldersTreeChange
 * Respond to the notification when the preference is changed for automatically or manually sorting the folders tree.
 */
-(void)handleAutoSortFoldersTreeChange:(NSNotification *)nc
{
	int selectedFolderId = [self actualSelection];
	
	if ([[Preferences standardPreferences] foldersTreeSortMethod] == MA_FolderSort_Manual)
	{
		[[Database sharedDatabase] beginTransaction];
		[self setManualSortOrderForNode:rootNode];
		[[Database sharedDatabase] commitTransaction];
	}
	
	blockSelectionHandler = YES;
	[self reloadDatabase:[[Preferences standardPreferences] arrayForKey:MAPref_FolderStatesHatena]];
	blockSelectionHandler = NO;
	
	// Make sure selected folder is visible
	[self selectFolder:selectedFolderId];
}

/* reloadDatabase
 * Refresh the folders tree and restore the specified archived state
 */
-(void)reloadDatabase:(NSArray *)stateArray
{
	[rootNode removeChildren];
	if (![self loadTree:[[Database sharedDatabase] arrayOfFolders:MA_Root_Folder] rootNode:rootNode])
	{
		[[Preferences standardPreferences] setFoldersTreeSortMethod:MA_FolderSort_ByName];
		[rootNode removeChildren];
		[self loadTree:[[Database sharedDatabase] arrayOfFolders:MA_Root_Folder] rootNode:rootNode];
	}
	[outlineView reloadData];
	[self unarchiveState:stateArray];
}

@end
