//
//  RefreshManagerGoogle.m
//  GalapagosReader
//
//  Created by Yoshitaka Sakamaki on 2010/02/04.
//  Copyright 2010. All rights reserved.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//

#import "RefreshManagerGoogle.h" 

#define MAX_ARTICLE_COUNT 160

// Singleton
static RefreshManagerGoogle * _refreshManagerGoogle = nil;
static NSArray * unReadCountList = nil;

// Refresh types
typedef enum {
	MA_Refresh_NilType = -1,
	MA_Refresh_Feed,
	MA_Refresh_FavIcon,
	MA_Refresh_Star
} RefreshTypes;


NSString *LABEL_DELIMITER = @" ー ";
NSArray *folderArrayLocal = nil;

// Private functions
@interface RefreshManagerGoogle (Private)

// sync folders function
-(void)syncFoldersAndFeedURLsLocal:(NSDictionary *)googleFolder;
-(void)syncFoldersAndFeedURLsGoogle:(NSArray *)folderArrayGoogle :(Folder *)localFolder;
-(NSString *) getAddLabel:(const int)parentId;
-(NSArray *) getDismantleLabel:(NSString *)label;
-(int) getLastFolderId;
-(BOOL) findFolder:(NSString *)googleLabel
				   :(int*)parentId;
-(int) createFolder:(NSArray *)labelArray;
-(void) localFolderRefresh;
-(void) pumpSubscriptionRefreshGoogle:(Folder *)folder;
-(void)addConnection:(AsyncConnectionGoogle *)conn;
-(void)folderRefreshCompleted:(AsyncConnectionGoogle *)connector;
-(void)setFolderUpdatingFlag:(Folder *)folder flag:(BOOL)theFlag;
-(void)refreshFavIcon:(Folder *)folder;
-(void)getCredentialsForFolder;
-(void)setFolderErrorFlag:(Folder *)folder flag:(BOOL)theFlag;
-(void)pumpFolderIconRefresh:(Folder *)folder;
-(void)refreshFeed:(Folder *)folder fromURL:(NSURL *)url withLog:(ActivityItem *)aItem;
-(void)beginRefreshTimer;
-(void)refreshPumper:(NSTimer *)aTimer;
-(void)removeConnection:(AsyncConnectionGoogle *)conn;
-(NSString *)getRedirectURL:(NSData *)data;
-(BOOL)findFeedUrlToLocal:(NSDictionary *)googleFolder;
-(BOOL)findFeedUrlToGoogle:(NSArray *)folderArrayGoogle :(Folder *)localFolder;
-(void)refreshSubscriptions:(NSArray *)foldersArray ignoringSubscriptionStatus:(BOOL)ignoreSubStatus;
-(BOOL)isRefreshingFolder:(Folder *)folder ofType:(RefreshTypes)type;
-(void)connRequest:(NSURL *)url 
			folder:(Folder *)folder 
			 aItem:(ActivityItem*)aItem
	   endSelector:(SEL)endSelector;

-(void)connRequest:(NSURL *)url 
			method:(NSString*)method
			  body:(NSString*)body
	   endSelector:(SEL)endSelector;

-(void)getStarItemListCompleted:(AsyncConnectionGoogle *)connector;
-(void)pumpStarItemListRefresh:(Folder *)folder;
-(void)pumpStarItemFeedRefresh:(Folder *)folder;
-(void)postReadItemCompleted:(AsyncConnectionGoogle *)connector;
@end

// Single refresh item type
@interface RefreshItemGoogle : NSObject {
	Folder * folder;
	RefreshTypes type;
}

// Accessor functions
-(void)setFolder:(Folder *)newFolder;
-(void)setType:(RefreshTypes)newType;
-(Folder *)folder;
-(RefreshTypes)type;
@end

@implementation RefreshItemGoogle

/* init
 * Initialises an empty RefreshItemGoogle with default values.
 */
-(id)init
{
	if ((self = [super init]) != nil)
	{
		[self setFolder:nil];
		[self setType:MA_Refresh_NilType];
	}
	return self;
}

/* setFolder
 */
-(void)setFolder:(Folder *)newFolder
{
	[newFolder retain];
	[folder release];
	folder = newFolder;
}

/* folder
 */
-(Folder *)folder
{
	return folder;
}

/* setType
 */
-(void)setType:(RefreshTypes)newType
{
	type = newType;
}

/* type
 */
-(RefreshTypes)type
{
	return type;
}

/* dealloc
 * Clean up behind ourselves.
 */
-(void)dealloc
{
	[folder release];
	[super dealloc];
}
@end


@implementation RefreshManagerGoogle
-(id)init
{
	if ((self = [super init]) != nil)
	{
		updateStarItem = YES;		
	}
	return self;
}

+(RefreshManagerGoogle *)sharedManager
{
	if (!_refreshManagerGoogle){
		_refreshManagerGoogle = [[RefreshManagerGoogle alloc] init];
	}
	return _refreshManagerGoogle;
}


- (void)syncFoldersAndFeedURLs
{	
	//google側のFeedListを取得する。
	NSArray *feedListGoogle = [[EditManagerGoogle sharedManager] getFeedList];
	
	//ネットワーク異常ならnilが返ってくる。
	if (nil == feedListGoogle)
		return;
	
	// DBから最新のフォルダ情報を取得する。
	[self localFolderRefresh];
	
	for (NSDictionary *googleFolder in feedListGoogle)
	{
		[self syncFoldersAndFeedURLsLocal:googleFolder];
	}
	
	for (Folder *localFolder in folderArrayLocal)
	{
		[self syncFoldersAndFeedURLsGoogle:feedListGoogle :localFolder];
	}
}

/*
 * GoogleReader側の購読リストを元にアプリケーション側の購読リストを反映する。
 */
- (void)syncFoldersAndFeedURLsLocal:(NSDictionary *)googleFolder
{
	//FeedURLを比較
	if (YES == [self findFeedUrlToLocal:googleFolder])
	{
		return;
	}
	
	Database * db = [Database sharedDatabase];
	
	// Feedの位置はルート（トップ）指定か？
	if (nil == [googleFolder objectForKey:@"label"])
	{
		// DB登録
		[db addRSSFolderGoogle:[googleFolder objectForKey:@"title"]
				   underParent:MA_Root_Folder
					afterChild:[self getLastFolderId] + 1
			   subscriptionURL:[[[NSMutableString alloc] initWithString:[googleFolder objectForKey:@"id"]] autorelease]
		 ];
		
		[self localFolderRefresh];
		
	} else {
		// ラベル名→フォルダ名に変換
		NSArray *labelArray = [self getDismantleLabel:[googleFolder objectForKey:@"label"]];
		
		//label数＝１
		if (1 == [labelArray count])
		{
			int folderId = MA_Root_Folder;
			if (YES == [self findFolder:[labelArray objectAtIndex:0] 
									   :&folderId]
				)
			{
				
				// ローカルに同名のフォルダが見つかった
				[db addRSSFolderGoogle:[googleFolder objectForKey:@"title"]
						   underParent:folderId//memo みつかったID
							afterChild:[self getLastFolderId] + 1
					   subscriptionURL:[[[NSMutableString alloc] initWithString:[googleFolder objectForKey:@"id"]] autorelease]
				 ];
				
				[self localFolderRefresh];
				
			} else {
				
				// 購読フィードの登録
				[db addRSSFolderGoogle:[googleFolder objectForKey:@"title"]
						   underParent:[self createFolder:labelArray] //フォルダ作成。
							afterChild:[self getLastFolderId] + 1
					   subscriptionURL:[[[NSMutableString alloc] initWithString:[googleFolder objectForKey:@"id"]] autorelease]
				 ];
				
				[self localFolderRefresh];
				
			}
		} else { //label数＝２以上
			
			// 購読フィードの登録
			[db addRSSFolderGoogle:[googleFolder objectForKey:@"title"]
					   underParent:[self createFolder:labelArray] //フォルダ作成。
						afterChild:[self getLastFolderId] + 1
				   subscriptionURL:[[[NSMutableString alloc] initWithString:[googleFolder objectForKey:@"id"]] autorelease]
			 ];
			[self localFolderRefresh];
		}
	}
}

/*
 * ローカル側のFeedURLを比較し、比較結果を返却する。
 */
- (BOOL)findFeedUrlToLocal:(NSDictionary *)googleFolder
{
	for (Folder *localFolder in folderArrayLocal)
	{
		if (MA_RSS_Folder != [localFolder type])
		{
			continue;
		}
		
		// FeedURL名を比較
		if ([[googleFolder objectForKey:@"id"] hasSuffix:[localFolder feedURL]])
		{
			//一致したら登録は行わない。
			return YES;
		}
	}
	return NO;
}

/*
 * Google側のFeedURLを比較し、比較結果を返却する。
 */
- (BOOL)findFeedUrlToGoogle:(NSArray *)folderArrayGoogle :(Folder *)localFolder
{
	for (NSMutableDictionary *googleFolder in folderArrayGoogle)
	{
		// FeedURL名を比較
		if ([[googleFolder objectForKey:@"id"] hasSuffix:[localFolder feedURL]])
		{
			//TODO:同一URL、フォルダの所属が違うというパターンもありえる。
			//そういう場合は、Google側に合わせるべき。
			return YES;
		}
	}
	return NO;
}

/*
 * アプリケーション側の購読リストを元にgoogle側の購読リストに反映させる
 */
- (void)syncFoldersAndFeedURLsGoogle:(NSArray *)folderArrayGoogle :(Folder *)localFolder
{
	if (MA_Google_NOT_SYNC == [localFolder googleSyncFlag])
		return;
	
	if ( ! IsRSSFolder(localFolder))
		return;

	if ([self findFeedUrlToGoogle:folderArrayGoogle :localFolder])
		return;
	
	EditManagerGoogle *edit = [EditManagerGoogle sharedManager];
	// Feedの位置はルート（トップ）指定か？
	if (MA_Root_Folder == [localFolder parentId])
	{
		[edit subscribedFeed:[localFolder name]
							:[localFolder feedURL]
							:nil
		 ];				
	} else {
		/* Feedが所属しているフォルダ〜ルートフォルダまでさかのぼって連結した文字列を返却する
		 * その時、フォルダ間は" ー "（大文字ハイフンを半角スペースで囲む）で連結させることとする。
		 */
		NSString *label = [self getAddLabel:[localFolder parentId]];
		[edit subscribedFeed:[localFolder name]
							:[localFolder feedURL]
							:label
		 ];
	}	
}

/*
 * " ー "でフォルダ名を連結し、返却する。
 */
- (NSString *) getAddLabel:(const int)parentId
{
	NSMutableString *result = [[[NSMutableString alloc] init] autorelease];
	for (Folder *localFolder in folderArrayLocal)
	{
		if (MA_Group_Folder != [localFolder type])
		{
			continue;
		}
		
		if ([localFolder itemId] == parentId)
		{
			if (MA_Root_Folder == [localFolder parentId])
			{
				[result insertString:[localFolder name]
							 atIndex:0];
				break;
			} else {
				[result insertString:[localFolder name]
							 atIndex:0];
				
				[result insertString:LABEL_DELIMITER
							 atIndex:0];
				
				// さらに親のフォルダを探す
				[result insertString:[self getAddLabel:[localFolder parentId]]
							 atIndex:0];
			}
		}
	}
	return result;
}


/*
 * " ー "で連結された文字列(半角スペース、全角ハイフン、半角スペース)をデリミタとし、
 * 分解した文字列を配列オブジェクトで返却する。
 */
- (NSArray *) getDismantleLabel:(NSString *)label
{
	NSMutableArray *result = [[[NSMutableArray alloc]init]autorelease];
	for (id loopItem in [label componentsSeparatedByString:LABEL_DELIMITER])
	{
		[result addObject:loopItem];
	}
	
	return result;
}

- (int)getLastFolderId
{
	int folderId = 0;
	int tempId = 0;
	
	for (Folder *localFolder in folderArrayLocal)
	{
		tempId = [localFolder itemId];
		if (folderId < tempId)
		{
			folderId = tempId;
		}
	}
	
	return folderId;
}

- (BOOL) findFolder:(NSString *)googleLabel
				   :(int*)parentId
{
	BOOL find_flg = NO;
	
	for (Folder * localFolder in folderArrayLocal)
	{
		if ( NO == IsGroupFolder(localFolder))
		{
			continue;
		}
		
		// フォルダ名とparentIdを比較して該当のlabel（フォルダ）がローカルに存在するかを調べる。
		if (
			(YES == [googleLabel isEqualToString:[localFolder name]])
			&&
			([localFolder parentId] == *parentId)
			)
		{
			// 一致したフォルダIDを保存しておく。
			*parentId = [localFolder itemId];
			find_flg = YES;
			break;
		}
	}
	return find_flg;
}

/*
 * 引数の文字列で指定されているアプリケーション側のフォルダ名を検索し、なければ作成する。
 */
- (int) createFolder:(NSArray *)labelArray
{
	int parentId = MA_Root_Folder;

	// 分解したラベル（要素番号０がトップ）を使ってフォルダの階層を作る。
	for (id loopItem in labelArray)
	{
		if (NO == [self findFolder:loopItem 
								  :&parentId]
			)
		{
			// Create the new folder in the database
			Database * db = [Database sharedDatabase];
			[db beginTransaction];
			parentId = [db addFolder:parentId
						  afterChild:-1 
						  folderName:loopItem  
								type:MA_Group_Folder 
					  canAppendIndex:NO];
			[db commitTransaction];
			
			[self localFolderRefresh];
			
			if (parentId != -1)
				[(AppController *)[NSApp delegate] selectFolder:parentId];
			
		}
	}
	return parentId;
}

/*
 * ローカルフォルダの情報を再取得する。
 */
- (void) localFolderRefresh
{
	// フォルダ追加等で常にデータが変化する為、毎回Databaseクラスから取得するようにする。
	folderArrayLocal = [[Database sharedDatabase] arrayOfAllFolders];
}

/*
 * 記事の取得
 */
-(void)pumpSubscriptionRefreshGoogle:(Folder *)folder
{
	
	// GoogleReaderと同期していないFeedはスキップ。
	if (MA_Google_NOT_SYNC == [folder googleSyncFlag])
		return;
	
	for (NSDictionary *unReadDict in unReadCountList)
	{
		// 未読リストに該当するFeedか判定する。
		if (NO == [[unReadDict objectForKey:@"id"] hasSuffix:[folder feedURL]])
			continue;
		
		int unReadCount = 0;

		// TODO:一応、全件取得しとく。
		unReadCount = [[unReadDict objectForKey:@"count"] intValue];

		NSString * name = [[folder name] isEqualToString:[Database untitledFeedFolderName]] ? [folder feedURL] : [folder name];
		ActivityItem * aItem = [[ActivityLog defaultLog] itemByName:name];
		// Seed the activity log for this feed.
		[aItem clearDetails];
		[aItem setStatus:NSLocalizedString(@"Retrieving articles", nil)];
		
		// Mark the folder as being refreshed. The updating status is not
		// persistent so we set this directly on the folder rather than
		// through the database.
		[self setFolderUpdatingFlag:folder flag:YES];
		
		// Additional detail for the log
		[aItem appendDetail:[NSString stringWithFormat:NSLocalizedString(@"Connecting to %@", nil), [folder feedURL]]];		

		//TODO:518400000=7日前までさかのぼって取得ってこと。
		/*
		NSURL* url = [NSURL URLWithString:[NSString
										   stringWithFormat:
										   @"http://www.google.com/reader/atom/%@?n=%d&ot=518400000"
										   ,[[unReadDict objectForKey:@"id"] stringByURLEncoding:NSUTF8StringEncoding]
										   ,unReadCount]];
		 */
		
		NSURL* url = [[EditManagerGoogle sharedManager] getUnreadFeedURL:
					  [[unReadDict objectForKey:@"id"] stringByURLEncoding:NSUTF8StringEncoding]
																		:unReadCount];
		
		[self connRequest:url
				   folder:folder 
					aItem:aItem
			  endSelector:@selector(folderRefreshCompleted:)];
		
	}
}

-(void)connRequest:(NSURL *)url 
			folder:(Folder *)folder 
			 aItem:(ActivityItem*)aItem
	   endSelector:(SEL)endSelector
{
	AsyncConnectionGoogle * conn;
	@try
	{
		conn = [[AsyncConnectionGoogle alloc] init];
		[conn setHttpHeaders:[[EditManagerGoogle sharedManager] createHttpHeader]];
		
		if ([conn beginLoadDataFromURL:url
							  username:[folder username]
							  password:[folder password]
							  delegate:self
						   contextData:folder
								   log:aItem
						didEndSelector:endSelector])
			[self addConnection:conn];
	}
	@finally
	{
		[conn release];
	}			
}

-(void)connRequest:(NSURL *)url 
			method:(NSString*)method
			  body:(NSString*)body
	   endSelector:(SEL)endSelector
{
	AsyncConnectionGoogle * conn;
	@try
	{
		conn = [[AsyncConnectionGoogle alloc] init];
		[conn setHttpHeaders:[[EditManagerGoogle sharedManager] createHttpHeader]];
		
		if ([conn beginLoadDataFromURL:url
							  delegate:self
						didEndSelector:endSelector
								method:method
								  body:body])
			[self addConnection:conn];
	}
	@finally
	{
		[conn release];
	}			
}

/* refreshSubscriptions
 * Add the folders specified in the foldersArray to the refreshArray.
 */
-(void)refreshSubscriptions:(NSArray *)foldersArray ignoringSubscriptionStatus:(BOOL)ignoreSubStatus
{
	// GoogleReaderから未読記事リストの取得し、新着記事を取得する
	unReadCountList = [[[EditManagerGoogle sharedManager] getUnReadCount] copy];
	if (nil == unReadCountList)
		return;
	
	statusMessageDuringRefresh = NSLocalizedString(@"Refreshing subscriptions...", nil);
	for (Folder * folder in foldersArray)
	{
		if (IsGroupFolder(folder))
			[self refreshSubscriptions:[[Database sharedDatabase] arrayOfFolders:[folder itemId]] ignoringSubscriptionStatus:NO];
		else if (IsRSSFolder(folder))
		{
			if (!IsUnsubscribed(folder) || ignoreSubStatus)
			{
				if (![self isRefreshingFolder:folder ofType:MA_Refresh_Feed])
				{
					RefreshItemGoogle * newItem = [[RefreshItemGoogle alloc] init];
					[newItem setFolder:folder];
					[newItem setType:MA_Refresh_Feed];
					[refreshArray addObject:newItem];
					[newItem release];
				}
			}
		}
	}
	[self beginRefreshTimer];
}

/* isRefreshingFolder
 * Returns whether refreshArray has an queue refresh for the specified folder
 * and refresh type.
 */
-(BOOL)isRefreshingFolder:(Folder *)folder ofType:(RefreshTypes)type
{
	for (RefreshItemGoogle * item in refreshArray)
	{
		if ([item folder] == folder && [item type] == type)
			return YES;
	}
	return NO;
}

-(void)refreshPumper:(NSTimer *)aTimer
{
	while (([connectionsArray count] < maximumConnections) && ([refreshArray count] > 0))
	{
		RefreshItemGoogle * item = [refreshArray objectAtIndex:0];
		switch ([item type])
		{
			case MA_Refresh_NilType:
				NSAssert(false, @"Uninitialised RefreshItem in refreshArray");
				break;
				
			case MA_Refresh_Feed:
				[self pumpSubscriptionRefreshGoogle:[item folder]];
				break;
				
			case MA_Refresh_FavIcon:
				[self pumpFolderIconRefresh:[item folder]];
				break;
				
			case MA_Refresh_Star:
				[self pumpStarItemListRefresh:[item folder]];
				break;

		}
		[refreshArray removeObjectAtIndex:0];
	}
	
	if ([connectionsArray count] == 0 && [refreshArray count] == 0 && hasStarted)
	{
		[pumpTimer invalidate];
		[pumpTimer release];
		pumpTimer = nil;
		[[NSNotificationCenter defaultCenter] postNotificationName:@"MA_Notify_RefreshStatus" object:nil];
		
		hasStarted = NO;
	}
}

-(void)subscriptionFeed:(Folder *)folder
{
	// Feedタイトル取得
	EditManagerGoogle *edit = [EditManagerGoogle sharedManager];
	NSString *title = [edit getFeedTitle:[folder feedURL]];
	if (title == nil){
		// 取得失敗
		return;
	}
	// 取得したタイトルを保持させておく
	[folder setFeedDescription:title];
	
	// ラベル編集の為にローカルフォルダをリフレッシュしておく。
	[self localFolderRefresh];
	// ラベル取得
	NSString *label = [self getAddLabel:[folder parentId]];
	
	// GoogleReaderへ購読を通知する。
	[edit subscribedFeed:title :[folder feedURL] :label ];
	
	// Feed Refresh
	[self refreshSubscriptions:[NSArray arrayWithObject:folder] ignoringSubscriptionStatus:NO];
	
	return;
}

/* folderRefreshCompleted
 * Called when a folder refresh completed.
 */
-(void)folderRefreshCompleted:(AsyncConnectionGoogle *)connector
{
	Folder * folder = (Folder *)[connector contextData];
	int folderId = [folder itemId];
	Database * db = [Database sharedDatabase];
	
	[self setFolderUpdatingFlag:folder flag:NO];
	if ([connector status] == MA_Connect_NeedCredentials)
	{
		if (![authQueue containsObject:folder])
			[authQueue addObject:folder];
		[self getCredentialsForFolder];
	}
	else if ([connector status] == MA_Connect_PermanentRedirect)
		// 今のところ通り得ないルート。
	{
		// We got a permanent redirect from the feed so change the feed URL
		// to the new location.
		[db setFolderFeedURL:folderId newFeedURL:[connector URLString]];
		[[connector aItem] appendDetail:[NSString stringWithFormat:NSLocalizedString(@"Feed URL updated to %@", nil), [connector URLString]]];
		return;
	}
	else if ([connector status] == MA_Connect_Stopped)
	{
		// Stopping the connection isn't an error, so clear any
		// existing error flag.
		[self setFolderErrorFlag:folder flag:NO];
		
		// Set the last update date for this folder.
		[db setFolderLastUpdate:folderId lastUpdate:[NSDate date]];
		
		// If this folder also requires an image refresh, add that
		if ([folder flags] & MA_FFlag_CheckForImage)
			[self refreshFavIcon:folder];
	}
	else if ([connector status] == MA_Connect_URLIsGone)
	{
		// We got HTTP 410 which means the feed has been intentionally
		// removed so unsubscribe the feed.
		[db setFolderFlag:folderId flagToSet:MA_FFlag_Unsubscribed];
		[[NSNotificationCenter defaultCenter] postNotificationName:@"MA_Notify_FoldersUpdated" object:[NSNumber numberWithInt:folderId]];
	}
	//saka 通信障害のルート
	else if ([connector status] == MA_Connect_Failed)
	{
		// Mark the feed as failed
		[self setFolderErrorFlag:folder flag:YES];
	}
	else if ([connector status] == MA_Connect_Succeeded)
	{
		NSData * receivedData = [connector receivedData];
		
		// Check whether this is an HTML redirect. If so, create a new connection using
		// the redirect.
		NSString * redirectURL = [self getRedirectURL:receivedData];
		if (redirectURL != nil)
		{
			if ([redirectURL isEqualToString:[connector URLString]])
			{
				// To prevent an infinite loop, don't redirect to the same URL.
				[[connector aItem] appendDetail:[NSString stringWithFormat:NSLocalizedString(@"Improper infinitely looping URL redirect to %@", nil), [connector URLString]]];
			}
			else
			{
				[self refreshFeed:folder fromURL:[NSURL URLWithString:redirectURL] withLog:[connector aItem]];
				[self removeConnection:connector];
				return;
			}
		}
		
		// Remember the last modified date
		NSString * lastModifiedString = [[connector responseHeaders] valueForKey:@"Last-Modified"];
		if (lastModifiedString != nil)
			[db setFolderLastUpdateString:folderId lastUpdateString:lastModifiedString];
		
		// Empty data feed is OK if we got HTTP 200
		int newArticlesFromFeed = 0;	
		RichXMLParser * newFeed = [[RichXMLParser alloc] init];
		if ([receivedData length] > 0)
		{
			Preferences * standardPreferences = [Preferences standardPreferences];
			if ([standardPreferences shouldSaveFeedSource])
			{
				NSString * feedSourcePath = [folder feedSourceFilePath];
				
				if ([standardPreferences boolForKey:MAPref_ShouldSaveFeedSourceBackup])
				{
					BOOL isDirectory = YES;
					NSFileManager * defaultManager = [NSFileManager defaultManager];
					if ([defaultManager fileExistsAtPath:feedSourcePath isDirectory:&isDirectory] && !isDirectory)
					{
						NSString * backupPath = [feedSourcePath stringByAppendingPathExtension:@"bak"];
						if (![defaultManager fileExistsAtPath:backupPath] || [defaultManager removeItemAtPath:backupPath error:NULL]) // Remove any old backup first
						{
							[defaultManager moveItemAtPath:feedSourcePath toPath:backupPath error:NULL];
						}
					}
				}
				
				[receivedData writeToFile:feedSourcePath options:NSAtomicWrite error:NULL];
			}
			
			// Create a new rich XML parser instance that will take care of
			// parsing the XML data we just got.
			if (newFeed == nil || ![newFeed parseRichXML:receivedData])
			{
				// Mark the feed as failed
				[self setFolderErrorFlag:folder flag:YES];
				[[connector aItem] setStatus:NSLocalizedString(@"Error parsing XML data in feed", nil)];
				[newFeed release];
				[self removeConnection:connector];
				return;
			}
			
			// Log number of bytes we received
			[[connector aItem] appendDetail:[NSString stringWithFormat:NSLocalizedString(@"%ld bytes received", nil), [receivedData length]]];
			
			// Extract the latest title and description
			NSString * feedTitle = [newFeed title];
			NSString * feedDescription = [newFeed description];
			NSString * feedLink = [newFeed link];
			
			// Synthesize feed link if it is missing
			if (feedLink == nil || [feedLink isBlank])
				feedLink = [[folder feedURL] baseURL];
			
			// We'll be collecting articles into this array
			NSMutableArray * articleArray = [NSMutableArray array];
			NSMutableArray * articleGuidArray = [NSMutableArray array];
			

			int skipArticleCount = 0;
			
			for (FeedItem * newsItem in [newFeed items])
			{
				NSDate * articleDate = [newsItem date];
				
				/*
				 * 既読処理の信頼性を上げるため、以下の条件の場合、サーバへ既読を通知する。
				 * ・既に取得した記事かつ、既読済みの場合
				 */
				NSMutableString * articleGuid = [NSMutableString stringWithFormat:[newsItem guid]];
				[articleGuid replaceOccurrencesOfString:@"tag:google.com,2005:reader/item/"
											 withString:@""
												options:NSLiteralSearch
												  range:NSMakeRange(0, [articleGuid length])];
				/*
				Article * article = [folder articleFromGuid:articleGuid];
				if (nil != article && YES == [article isRead])
				{
					[[EditManagerGoogle sharedManager] notifyMarkArticleRead:articleGuid
																			:[folder feedURL]
																			:YES];
					//除外する記事をカウントする。
					skipArticleCount++;
					continue;
				}
				 */

				
				// This is a horrible hack for horrible feeds that contain more than one item with the same guid.
				// Bad feeds! I'm talking to you, WordPress Trac.
				NSUInteger articleIndex = [articleGuidArray indexOfObject:articleGuid];
				if (articleIndex != NSNotFound)
				{
					if (articleDate == nil)
						continue; // Skip this duplicate article
					
					Article * existingArticle = [articleArray objectAtIndex:articleIndex];
					if ([articleDate compare:[existingArticle date]] == NSOrderedDescending)
					{
						// This article is later, so use it instead
						[articleArray removeObjectAtIndex:articleIndex];
						[articleGuidArray removeObjectAtIndex:articleIndex];
					}
					else
						continue; // Skip this duplicate article
				}
				[articleGuidArray addObject:articleGuid];
				
				if (articleDate == nil)
					articleDate = [NSDate date];
				
				Article * article = [[Article alloc] initWithGuid:articleGuid];
				[article setFolderId:folderId];
				[article setAuthor:[newsItem author]];
				[article setBody:[newsItem description]];
				[article setTitle:[newsItem title]];
				[article setLink:[newsItem link]];
				[article setDate:articleDate];
				[article setEnclosure:[newsItem enclosure]];
				if ([[article enclosure] isNotEqualTo:@""])
				{
					[article setHasEnclosure:YES];
				}
				[articleArray addObject:article];
				[article release];
			}
			
			// Here's where we add the articles to the database
			if (([articleArray count] - skipArticleCount) > 0u)
			{
				NSArray * guidHistory = [db guidHistoryForFolderId:folderId];
				
				[folder clearCache];
				// Should we wrap the entire loop or just individual article updates?
				[db beginTransaction];
				for (Article * article in articleArray)
				{
					if ([db createArticle:folderId article:article guidHistory:guidHistory] && ([article status] == MA_MsgStatus_New))
						++newArticlesFromFeed;
				}
				[db commitTransaction];				
			}
			
			[db beginTransaction];
			
			// A notify is only needed if we added any new articles.
			if ([[folder name] hasPrefix:[Database untitledFeedFolderName]] && ![feedTitle isBlank])
			{
				// If there's an existing feed with this title, make ours unique
				// BUGBUG: This duplicates logic in database.m so consider moving it there.
				NSString * oldFeedTitle = feedTitle;
				unsigned int index = 1;
				
				while (([db folderFromName:feedTitle]) != nil)
					feedTitle = [NSString stringWithFormat:@"%@ (%i)", oldFeedTitle, index++];
				
				[[connector aItem] setName:feedTitle];
				[db setFolderName:folderId newName:feedTitle];
			}
			if (feedDescription != nil)
				[db setFolderDescription:folderId newDescription:feedDescription];
			
			if (feedLink!= nil)
				[db setFolderHomePage:folderId newHomePage:feedLink];
			
			[db commitTransaction];
			
			// Let interested callers know that the folder has changed.
			[[NSNotificationCenter defaultCenter] postNotificationName:@"MA_Notify_FoldersUpdated" object:[NSNumber numberWithInt:folderId]];
		}
		
		// Mark the feed as succeeded
		[self setFolderErrorFlag:folder flag:NO];
		
		// Set the last update date for this folder.
		[db setFolderLastUpdate:folderId lastUpdate:[NSDate date]];
		
		// Send status to the activity log
		if (newArticlesFromFeed == 0)
			[[connector aItem] setStatus:NSLocalizedString(@"No new articles available", nil)];
		else
		{
			NSString * logText = [NSString stringWithFormat:NSLocalizedString(@"%d new articles retrieved", nil), newArticlesFromFeed];
			[[connector aItem] setStatus:logText];
		}
		
		// Done with this connection
		[newFeed release];
		
		// If this folder also requires an image refresh, add that
		if ([folder flags] & MA_FFlag_CheckForImage)
			[self refreshFavIcon:folder];
		
		// Add to count of new articles so far
		countOfNewArticles += newArticlesFromFeed;
	}
	[self removeConnection:connector];
}

/* refreshSubscriptions
 * Add the folders specified in the foldersArray to the refreshArray.
 */
-(void)refreshStarItem:(Folder *)folder
{
	RefreshItemGoogle * newItem = [[RefreshItemGoogle alloc] init];
	[newItem setFolder:folder];
	[newItem setType:MA_Refresh_Star];
	[refreshArray addObject:newItem];
	[newItem release];			
	[self beginRefreshTimer];
}

/*
 * スターリストの取得
 */
-(void)pumpStarItemListRefresh:(Folder *)folder
{
	
	NSString * name = [[folder name] isEqualToString:[Database untitledFeedFolderName]] ? [folder feedURL] : [folder name];
	ActivityItem * aItem = [[ActivityLog defaultLog] itemByName:name];
	// Seed the activity log for this feed.
	[aItem clearDetails];
	[aItem setStatus:NSLocalizedString(@"Retrieving articles", nil)];
	
	// Mark the folder as being refreshed. The updating status is not
	// persistent so we set this directly on the folder rather than
	// through the database.
	[self setFolderUpdatingFlag:folder flag:YES];
	
	// Additional detail for the log
	[aItem appendDetail:[NSString stringWithFormat:NSLocalizedString(@"Connecting to %@", nil), [folder feedURL]]];		
	
	[self connRequest:[[EditManagerGoogle sharedManager] getStarListURL]
			   folder:folder
				aItem:aItem
		  endSelector:@selector(getStarItemListCompleted:)];
}

/*
 * スターリストの取得完了処理
 */
-(void)getStarItemListCompleted:(AsyncConnectionGoogle *)connector
{
	Folder * folder = (Folder *)[connector contextData];
	int folderId = [folder itemId];
	Database * db = [Database sharedDatabase];
	
	[self setFolderUpdatingFlag:folder flag:NO];
	
	if ([connector status] == MA_Connect_Stopped)
	{
		// Stopping the connection isn't an error, so clear any
		// existing error flag.
		[self setFolderErrorFlag:folder flag:NO];
		
		// Set the last update date for this folder.
		[db setFolderLastUpdate:folderId lastUpdate:[NSDate date]];
		
		// If this folder also requires an image refresh, add that
		if ([folder flags] & MA_FFlag_CheckForImage)
			[self refreshFavIcon:folder];
	}
	else if ([connector status] == MA_Connect_URLIsGone)
	{
		// We got HTTP 410 which means the feed has been intentionally
		// removed so unsubscribe the feed.
		[db setFolderFlag:folderId flagToSet:MA_FFlag_Unsubscribed];
		[[NSNotificationCenter defaultCenter] postNotificationName:@"MA_Notify_FoldersUpdated" object:[NSNumber numberWithInt:folderId]];
	}
	//saka 通信障害のルート
	else if ([connector status] == MA_Connect_Failed)
	{
		// Mark the feed as failed
		[self setFolderErrorFlag:folder flag:YES];
	}
	else if ([connector status] == MA_Connect_Succeeded)
	{
		NSData * receivedData = [connector receivedData];
		DDXMLDocument *doc = [[[DDXMLDocument alloc] initWithData:receivedData
														  options:0 error:nil] autorelease];
		DDXMLElement *root = [doc rootElement];
		
		
		NSArray *feedListCount = [root nodesForXPath:@"//object[1]/list[1]/object" error:nil];
		//NSLog(@"%d", [feedListCount count]);
		
		int offset = 1;
		NSArray * articlesArray = [folder articles];
		
		if ([articlesArray count] != [feedListCount count]) {
			
			//ローカル側とグーグル側のフィード数が合わなければ即、スター記事を取得しなおす
			[self pumpStarItemFeedRefresh:folder];
			
		} else {
			int roopCnt1=0;
			for (;roopCnt1<[feedListCount count];roopCnt1++)
			{
				BOOL findFlg1 = NO;
				// id
				NSArray *feedListId = [root nodesForXPath:[NSString stringWithFormat:@"//object[1]/list[1]/object[%d]/number[1]",
														   roopCnt1+offset] error:nil];
				
				NSString *tempGuid = [[feedListId objectAtIndex:0] stringValue];
				
				//NSLog(@"%@", [NSString stringWithFormat:@"%016qx", [tempGuid longLongValue]]);
				
				for (Article * article in articlesArray)
				{
					if ([[article guid] isEqualToString:[NSString stringWithFormat:@"%@%016qx", MA_StarItem_String, [tempGuid longLongValue]]]) {
						findFlg1 = YES;
						break;
					}
				}
				
				if (findFlg1 == YES)
					continue;
				
				//未取得のスター記事を取得する
				[self pumpStarItemFeedRefresh:folder];
				break;
			}
		}
	}
	[self removeConnection:connector];
}

/*
 * スター記事の取得
 */
-(void)pumpStarItemFeedRefresh:(Folder *)folder
{
	
	NSString * name = [[folder name] isEqualToString:[Database untitledFeedFolderName]] ? [folder feedURL] : [folder name];
	ActivityItem * aItem = [[ActivityLog defaultLog] itemByName:name];
	// Seed the activity log for this feed.
	[aItem clearDetails];
	[aItem setStatus:NSLocalizedString(@"Retrieving articles", nil)];
	
	// Mark the folder as being refreshed. The updating status is not
	// persistent so we set this directly on the folder rather than
	// through the database.
	[self setFolderUpdatingFlag:folder flag:YES];
	
	// Additional detail for the log
	[aItem appendDetail:[NSString stringWithFormat:NSLocalizedString(@"Connecting to %@", nil), [folder feedURL]]];		
		
	[self connRequest:[[EditManagerGoogle sharedManager] getStarItemURL]
			   folder:folder
				aItem:aItem
		  endSelector:@selector(getStarItemFeedCompleted:)];
}

-(void)getStarItemFeedCompleted:(AsyncConnectionGoogle *)connector
{
	Folder * folder = (Folder *)[connector contextData];
	int folderId = [folder itemId];
	Database * db = [Database sharedDatabase];
	
	[self setFolderUpdatingFlag:folder flag:NO];
	
	if ([connector status] == MA_Connect_Stopped)
	{
		// Stopping the connection isn't an error, so clear any
		// existing error flag.
		[self setFolderErrorFlag:folder flag:NO];
		
		// Set the last update date for this folder.
		[db setFolderLastUpdate:folderId lastUpdate:[NSDate date]];
		
		// If this folder also requires an image refresh, add that
		if ([folder flags] & MA_FFlag_CheckForImage)
			[self refreshFavIcon:folder];
	}
	else if ([connector status] == MA_Connect_URLIsGone)
	{
		// We got HTTP 410 which means the feed has been intentionally
		// removed so unsubscribe the feed.
		[db setFolderFlag:folderId flagToSet:MA_FFlag_Unsubscribed];
		[[NSNotificationCenter defaultCenter] postNotificationName:@"MA_Notify_FoldersUpdated" object:[NSNumber numberWithInt:folderId]];
	}
	//saka 通信障害のルート
	else if ([connector status] == MA_Connect_Failed)
	{
		// Mark the feed as failed
		[self setFolderErrorFlag:folder flag:YES];
	}
	else if ([connector status] == MA_Connect_Succeeded)
	{
		NSData * receivedData = [connector receivedData];
		Database * db = [Database sharedDatabase];

		RichXMLParser * newFeed = [[RichXMLParser alloc] init];
		if ([receivedData length] > 0)
		{		
			// Create a new rich XML parser instance that will take care of
			// parsing the XML data we just got.
			if (newFeed == nil || ![newFeed parseRichXML:receivedData])
			{
				// Mark the feed as failed
				[self setFolderErrorFlag:folder flag:YES];
				[newFeed release];
				return;
			}
			
			// Extract the latest title and description
			NSString * feedTitle = [newFeed title];
			NSString * feedDescription = [newFeed description];
			
			// We'll be collecting articles into this array
			NSMutableArray * articleArray = [NSMutableArray array];
			NSMutableArray * articleGuidArray = [NSMutableArray array];
			
			// Parse off items.
			
			for (FeedItem * newsItem in [newFeed items])
			{
				NSDate * articleDate = [newsItem date];
				
				//16桁のID部分のみを切り出す
				NSMutableString * articleGuid = [NSMutableString stringWithFormat:[newsItem guid]];
				[articleGuid replaceOccurrencesOfString:@"tag:google.com,2005:reader/item/"
											 withString:@""
												options:NSLiteralSearch
												  range:NSMakeRange(0, [articleGuid length])];
				
				//既存のレコードと重複させないよう、guidの先頭に"star:"を付与する。
				NSString * starArticleGuid = [NSString stringWithFormat:@"%@%@",
											  MA_StarItem_String, articleGuid];
				
				// Bad feeds! I'm talking to you, WordPress Trac.
				NSUInteger articleIndex = [articleGuidArray indexOfObject:starArticleGuid];
				if (articleIndex != NSNotFound)
				{
					if (articleDate == nil)
						continue; // Skip this duplicate article
					
					Article * existingArticle = [articleArray objectAtIndex:articleIndex];
					if ([articleDate compare:[existingArticle date]] == NSOrderedDescending)
					{
						// This article is later, so use it instead
						[articleArray removeObjectAtIndex:articleIndex];
						[articleGuidArray removeObjectAtIndex:articleIndex];
					}
					else
						continue; // Skip this duplicate article
				}
				[articleGuidArray addObject:starArticleGuid];
				
				if (articleDate == nil)
					articleDate = [NSDate date];
				
				Article * article = [[Article alloc] initWithGuid:starArticleGuid];
				[article setFolderId:folderId];
				[article setAuthor:[newsItem author]];
				[article setBody:[newsItem description]];
				[article setTitle:[newsItem title]];
				[article setLink:[newsItem link]];
				[article setDate:articleDate];
				[article setEnclosure:[newsItem enclosure]];
				[article markRead:YES];
				[article markFlagged:YES];
				if ([[article enclosure] isNotEqualTo:@""])
				{
					[article setHasEnclosure:YES];
				}
				[articleArray addObject:article];
				[article release];
				
				//guidに対応するレコードがあれば、フラグをONにする。
				[db updateStarItem:folderId :articleGuid];
			}
			
			// Here's where we add the articles to the database
			if ([articleArray count] > 0u)
			{
				//サーバ側のスター付きアイテムに無いローカル側のアイテムは削除。
				[db beginTransaction];
				BOOL findFlag;
				NSArray * localArticles = [folder articles];
				for (Article * localArticle in localArticles)
				{
					findFlag = NO;
					for (Article * serverArticle in articleArray)
					{
						if ([[localArticle guid] isEqualToString:[serverArticle guid]])
						{
							findFlag = YES;
							break;
						}
					}
					if (NO == findFlag)
					{
						//削除
						[db markArticleFlagged:[folder itemId]
										  guid:[localArticle guid]
									 isFlagged:NO];
					}
				}
				
				NSArray * guidHistory = [db guidHistoryForFolderId:folderId];
				
				[folder clearCache];
				// Should we wrap the entire loop or just individual article updates?
				for (Article * article in articleArray)
				{
					//DBにぶち込む。
					[db createArticle:folderId article:article guidHistory:guidHistory];
				}
				[db commitTransaction];				
			} else {
				//ローカル側のアイテムを全削除。
				[db beginTransaction];
				NSArray * localArticles = [folder articles];
				for (Article * localArticle in localArticles)
				{
					[db markArticleFlagged:[folder itemId]
									  guid:[localArticle guid]
								 isFlagged:NO];
				}
				[db commitTransaction];							
			}
			
			[db beginTransaction];
			
			// A notify is only needed if we added any new articles.
			if ([[folder name] hasPrefix:[Database untitledFeedFolderName]] && ![feedTitle isBlank])
			{
				// If there's an existing feed with this title, make ours unique
				// BUGBUG: This duplicates logic in database.m so consider moving it there.
				NSString * oldFeedTitle = feedTitle;
				unsigned int index = 1;
				
				while (([db folderFromName:feedTitle]) != nil)
					feedTitle = [NSString stringWithFormat:@"%@ (%i)", oldFeedTitle, index++];
				
				[db setFolderName:folderId newName:feedTitle];
			}
			if (feedDescription != nil)
				[db setFolderDescription:folderId newDescription:feedDescription];
			
			[db commitTransaction];
			
			NSString * continuation = [newFeed continuation];
			if ( ! [continuation isEqualToString:@""])
			{
				// 記事を継続して取得する
				NSLog(@"TODO: Continuation の処理まだ");
			}
		}
		
		// Done with this connection
		[newFeed release];
	}
	[self removeConnection:connector];
}

/* syncReadItem
 * 既読リストの取得
 */
-(void)syncReadItem
{
	[self connRequest:[[EditManagerGoogle sharedManager] getReadItemListURL]
			   method:nil
				 body:nil
		  endSelector:@selector(getReadItemListCompleted:)];
	
}

/* getReadItemListCompleted
 * 既読状態の同期処理を行う
 */
#define LIMIT 200
-(void)getReadItemListCompleted:(AsyncConnectionGoogle *)connector
{
	
	int feedCount = 0;
	Database *db = [Database sharedDatabase];
	NSMutableString *body = [[[NSMutableString alloc] init] autorelease];
	NSData * receivedData = [connector receivedData];
	//NSLog(@"%@", [[NSString alloc] initWithData:receivedData encoding:NSUTF8StringEncoding]);
	NSArray * readItemList = [[EditManagerGoogle sharedManager] getReadItemList:receivedData];
	for (NSDictionary * readFeedDict in readItemList)
	{
		BOOL findFlag = NO;
		for (Folder * folder in [db arrayOfAllFolders])
		{
			if (![folder isRSSFolder])
				continue;
			
			Article * article = [folder articleFromGuid:[readFeedDict objectForKey:@"id"]];
			//NSLog(@"%@", readFeedId);
			
			if (article != nil && NO == [article isRead]) {
				[db markArticleRead:[folder itemId] guid:[article guid] isRead:YES notifyFlag:NO];
				findFlag = YES;
				break;
			}
		}
		
		if (YES == findFlag)
			continue;
		
		//ローカル側に記事が存在しない場合
		[body appendFormat:@"i=tag:google.com,2005:reader/item/%@&", [readFeedDict objectForKey:@"id"]];
		
		//
		if (++feedCount == LIMIT)
		{
			//既読記事取得
			[self connRequest:[[EditManagerGoogle sharedManager] getReadItemURL]
					   method:@"POST"
						 body:[[EditManagerGoogle sharedManager] getReadItemBody:body]
				  endSelector:@selector(postReadItemCompleted:)];
			
			body = [[[NSMutableString alloc] init] autorelease];
			feedCount = 0;
		}
	}
	
	if (feedCount > 0)
	{
		//既読記事取得
		[self connRequest:[[EditManagerGoogle sharedManager] getReadItemURL]
				   method:@"POST"
					 body:[[EditManagerGoogle sharedManager] getReadItemBody:body]
			  endSelector:@selector(postReadItemCompleted:)];
	}
	
	[self removeConnection:connector];
}

/* postReadItemCompleted
 */
-(void)postReadItemCompleted:(AsyncConnectionGoogle *)connector
{
	Database * db = [Database sharedDatabase];
	
	if ([connector status] == MA_Connect_Succeeded)
	{
		NSData * receivedData = [connector receivedData];
		
		RichXMLParser * newFeed = [[RichXMLParser alloc] init];
		if ([receivedData length] > 0)
		{		
			// Create a new rich XML parser instance that will take care of
			// parsing the XML data we just got.
			if (newFeed == nil || ![newFeed parseRichXML:receivedData])
			{
				[newFeed release];
				return;
			}
			
			// Extract the latest title and description
			NSString * feedTitle = [newFeed title];
			NSString * feedDescription = [newFeed description];
			NSMutableString * feedURL = [NSMutableString stringWithFormat:[newFeed feedId]];
			if ([feedURL isEqualToString:@""])
				return;
			
			[feedURL replaceOccurrencesOfString:@"tag:google.com,2005:reader/feed/"
									 withString:@""
										options:NSLiteralSearch
										  range:NSMakeRange(0, [feedURL length])];
				
			Folder * folder = [db folderFromFeedURL:feedURL];
			int folderId = [folder itemId];
						
			// We'll be collecting articles into this array
			NSMutableArray * articleArray = [NSMutableArray array];
			NSMutableArray * articleGuidArray = [NSMutableArray array];
			
			// Parse off items.
			
			for (FeedItem * newsItem in [newFeed items])
			{
				NSDate * articleDate = [newsItem date];
				
				//16桁のID部分のみを切り出す
				NSMutableString * articleGuid = [NSMutableString stringWithFormat:[newsItem guid]];
				[articleGuid replaceOccurrencesOfString:@"tag:google.com,2005:reader/item/"
											 withString:@""
												options:NSLiteralSearch
												  range:NSMakeRange(0, [articleGuid length])];
				
				// Bad feeds! I'm talking to you, WordPress Trac.
				NSUInteger articleIndex = [articleGuidArray indexOfObject:articleGuid];
				if (articleIndex != NSNotFound)
				{
					if (articleDate == nil)
						continue; // Skip this duplicate article
					
					Article * existingArticle = [articleArray objectAtIndex:articleIndex];
					if ([articleDate compare:[existingArticle date]] == NSOrderedDescending)
					{
						// This article is later, so use it instead
						[articleArray removeObjectAtIndex:articleIndex];
						[articleGuidArray removeObjectAtIndex:articleIndex];
					}
					else
						continue; // Skip this duplicate article
				}
				[articleGuidArray addObject:articleGuid];
				
				if (articleDate == nil)
					articleDate = [NSDate date];
				
				Article * article = [[Article alloc] initWithGuid:articleGuid];
				[article setFolderId:folderId];
				[article setAuthor:[newsItem author]];
				[article setBody:[newsItem description]];
				[article setTitle:[newsItem title]];
				[article setLink:[newsItem link]];
				[article setDate:articleDate];
				[article setEnclosure:[newsItem enclosure]];
				[article markRead:YES];
				[article markFlagged:NO];
				if ([[article enclosure] isNotEqualTo:@""])
				{
					[article setHasEnclosure:YES];
				}
				[articleArray addObject:article];
				[article release];				
			}
			// Here's where we add the articles to the database
			if ([articleArray count] > 0u)
			{
				//サーバ側のスター付きアイテムに無いローカル側のアイテムは削除。
				[db beginTransaction];
				NSArray * guidHistory = [db guidHistoryForFolderId:folderId];
				
				[folder clearCache];
				// Should we wrap the entire loop or just individual article updates?
				for (Article * article in articleArray)
				{
					//DBにぶち込む。
					[db createArticle:folderId article:article guidHistory:guidHistory];
				}
				[db commitTransaction];				
			} else {
				//ローカル側のアイテムを全削除。
				[db beginTransaction];
				NSArray * localArticles = [folder articles];
				for (Article * localArticle in localArticles)
				{
					[db markArticleFlagged:[folder itemId]
									  guid:[localArticle guid]
								 isFlagged:NO];
				}
				[db commitTransaction];							
			}
			
			[db beginTransaction];
			
			// A notify is only needed if we added any new articles.
			if ([[folder name] hasPrefix:[Database untitledFeedFolderName]] && ![feedTitle isBlank])
			{
				// If there's an existing feed with this title, make ours unique
				// BUGBUG: This duplicates logic in database.m so consider moving it there.
				NSString * oldFeedTitle = feedTitle;
				unsigned int index = 1;
				
				while (([db folderFromName:feedTitle]) != nil)
					feedTitle = [NSString stringWithFormat:@"%@ (%i)", oldFeedTitle, index++];
				
				[db setFolderName:folderId newName:feedTitle];
			}
			if (feedDescription != nil)
				[db setFolderDescription:folderId newDescription:feedDescription];
			
			[db commitTransaction];
			
		}
		// Done with this connection
		[newFeed release];		
	}
	[self removeConnection:connector];
}

-(void)dealloc
{
	[super dealloc];
}

@end
