////////////////////////////////////////////////////////////////////////////////
// CMesh.cpp
///////////////////////////////////////////////////////////////////////////////

#include "CMesh.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "../Mesh/MeshDef.h"
#include "../MQOLib/CMQOLoader.h"
#include "../Mesh/ConvMesh.h"
#include "../Mesh/CTextureManager.h"
#include "../Matrix/Calc.h"

#include <float.h>

CMesh::CMesh()
{
	memset( &mInfo, 0, sizeof(MeshInfo) );
	mMaterial = NULL;
	mObject   = NULL;
}

CMesh::~CMesh()
{
}

void CMesh::Release()
{
	// eNX`̊J
	if( mMaterial ){
		int material_num = mInfo.MaterialNum;
		for( int i = 0 ; i < material_num ; i++ ){
			char* name = mMaterial[i].TextureName;
			gTextureManager.Release( name );
		}
	}
	strcpy( mInfo.Name, "" );
	if( mMaterial ){
		delete [] mMaterial;
		mMaterial = NULL;
		mInfo.MaterialNum = 0;
	}
	if( mObject ){
		for( int i = 0 ; i < mInfo.ObjectNum ; i++ ){
			MeshObject* pObject = &mObject[i];
			strcpy( pObject->ObjectName, "" );

			if( pObject->Vertex ){
				delete [] pObject->Vertex;
				pObject->Vertex = NULL;
				pObject->VertexNum = 0;
			}

			if( pObject->Face ){
				for( int j = 0 ; j < pObject->FaceNum ; j++ ){
					MeshObjectFace* pFace = &pObject->Face[j];

					if( pFace->VertexIndex ){
						delete [] pFace->VertexIndex;
						pFace->VertexIndex = NULL;
						pFace->VertexIndexNum = 0;
					}
				}
			}
		}

		delete [] mObject;
		mObject = NULL;
		mInfo.ObjectNum = 0;
	}
}

bool CMesh::LoadFromMQO( char* filename, float scale )
{
	CMQOLoader MQOLoadr;
	CMQOData MQOData;
	bool result = MQOLoadr.Load( &MQOData, filename );
	if( !result )return false;

	result = ConvMesh( &MQOData, this, scale );
	if( !result ){
		Release();
	}
	
	MQOData.Release();
	
	ClearMatrix();
	// eNX`̓ǂݍ
	if( mMaterial ){
		int material_num = mInfo.MaterialNum;
		for( int i = 0 ; i < material_num ; i++ ){
			char* name = mMaterial[i].TextureName;
			gTextureManager.LoadTexture( name, name );
		}
	}
	return result;
}

void CMesh::Draw()
{
	{
		glPolygonMode( GL_FRONT, GL_FILL );
		glPolygonMode( GL_BACK, GL_POINT );
		glCullFace(GL_FRONT);
		glShadeModel(GL_SMOOTH);

		glPushMatrix();
		glMultMatrixf( &mMatrix.m[0][0] );
	}
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_LIGHTING);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	int n = mInfo.ObjectNum;
	for( int i = 0 ; i < n ; i++ ){
		MeshObject* pObject = &mObject[i];
		glPushMatrix();
		glMultMatrixf( &pObject->Matrix.m[0][0] );

		MeshVertex* pv_top = pObject->Vertex;
		int face_num = pObject->FaceNum;
		for( int fi = 0 ; fi < face_num ; fi++ ){
			
			if( mMaterial ){
				int material_index = pObject->Face[fi].Material;
				
				if( material_index >= 0 ){
					MeshMaterial* pMaterial = &mMaterial[material_index];
					glMaterialfv( GL_FRONT_AND_BACK, GL_DIFFUSE, pMaterial->Diffuse );
					
					glMaterialfv( GL_FRONT_AND_BACK, GL_SPECULAR, pMaterial->Speculer );
					glMaterialf( GL_FRONT_AND_BACK,  GL_SHININESS, pMaterial->Power );
					
					glMaterialfv( GL_FRONT_AND_BACK, GL_AMBIENT, pMaterial->Ambient );
					glMaterialfv( GL_FRONT_AND_BACK, GL_EMISSION, pMaterial->Emissive );
					
					char* texname =  pMaterial->TextureName;
					gTextureManager.BindTexture( texname );
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);//GL_NEAREST
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );//GL_LINEAR);
				}
			}
			
			int* pIndex = pObject->Face[fi].VertexIndex;

			glBegin ( (GLenum)GetObjectFacePrimitiveType( i, fi ) );

			int index_num = pObject->Face[fi].VertexIndexNum;
			for( int ii = 0 ; ii < index_num ; ii++ ){

				MeshVertex* pv = &pv_top[ pIndex[ii] ];
				glTexCoord2f( pv->u, pv->v );
				glNormal3f( pv->nx, pv->ny, pv->nz );
				glVertex3f( pv->x, pv->y, pv->z );
			}
			glEnd();
		}
		glPopMatrix();
	}

	glPopMatrix();
	
	glDisable(GL_BLEND);
}

void CMesh::ClearMatrix()
{
	for( int i = 0 ; i < mInfo.ObjectNum ; i++ ){
		mObject[i].Matrix.Identity();
	}
	mMatrix.Identity();
}

void CMesh::SetMatrix( char* Name, CMatrix* pMatrix )
{
	if( !Name ){
		mMatrix = *pMatrix;
	}else{
		for( int i = 0 ; i < mInfo.ObjectNum ; i++ ){
			char* name = mObject[i].ObjectName;
			if( strcmp( Name, name ) == 0 ){
				mObject[i].Matrix = *pMatrix;
				break;
			}
		}
	}
}

void CMesh::SetName( const char* name )
{
	strcpy( mInfo.Name, name );
}

bool CMesh::CreateMaterial( int n )
{
	try{
		if( mMaterial )throw "CMesh::CreateMaterial";
		mMaterial = new MeshMaterial[ n ];

		if( !mMaterial )throw "CMesh::CreateMaterial";
		memset( mMaterial, 0, sizeof(MeshMaterial)*n );
		
		mInfo.MaterialNum = n;
		memset( mMaterial, 0, sizeof(MeshMaterial)*n );
	}catch( char* str ){
		//printf( "ERROR:[%s]\n", str);
		return false;
	}
	return true;
}

bool CMesh::SetMaterial( int idx, const MeshMaterial* material )
{
	try{
		if( idx < 0 || idx >= mInfo.MaterialNum )throw "CMesh::SetMaterial";
		memcpy( &mMaterial[idx], material, sizeof(MeshMaterial) );
	}catch( char* str ){
		//printf( "ERROR:[%s]\n", str);
		return false;
	}
	return true;
}


bool CMesh::CreateObject( int n )
{
	try{
		if( mObject )throw "CMesh::CreateObject";
		mObject = new MeshObject[n];

		if( !mObject )throw "CMesh::CreateObject";
		memset( mObject, 0, sizeof(MeshObject)*n );
		
		mInfo.ObjectNum = n;
		memset( mObject, 0, sizeof(MeshObject)*n );
	}catch( char* str ){
		//printf( "ERROR:[%s]\n", str);
		return false;
	}
	return true;
}

bool CMesh::CreateObjectVertex( int idx, int num )
{
	try{
		if( idx < 0 || idx >= mInfo.ObjectNum )throw "CMesh::CreateObjectVertex";
		MeshObject* pObject = &mObject[idx];
		if( pObject->Vertex )throw "CMesh::CreateObjectVertex";
		pObject->Vertex = new MeshVertex[num];
		
		if( !pObject->Vertex )throw "CMesh::CreateObjectVertex";
		memset( pObject->Vertex, 0, sizeof(MeshVertex)*num );

		pObject->VertexNum = num;
		memset( pObject->Vertex, 0, sizeof(MeshVertex)*num );
	}catch( char* str ){
		//printf( "ERROR:[%s]\n", str);
		return false;
	}
	return true;
}

bool CMesh::CreateObjectFace( int idx, int num )
{
	try{
		if( idx < 0 || idx >= mInfo.ObjectNum )throw "CMesh::CreateObjectFace";
		MeshObject* pObject = &mObject[idx];
		if( pObject->Face )throw "CMesh::CreateObjectFace";
		pObject->Face = new MeshObjectFace[num];
		
		if( !pObject->Face )throw "CMesh::CreateObjectFace";
		memset( pObject->Face, 0, sizeof(MeshObjectFace)*num );

		pObject->FaceNum = num;
		memset( pObject->Face, 0, sizeof(MeshObjectFace)*num );
		for( int i = 0 ; i < num ; i++ ){
			pObject->Face[i].PrimitiveType = GL_TRIANGLES;
		}
	}catch( char* str ){
		//printf( "ERROR:[%s]\n", str);
		return false;
	}
	return true;
}

bool CMesh::CreateObjectFaceVertexIndex( int idx, int fidx, int num )
{
	try{
		if( idx < 0 || idx >= mInfo.ObjectNum )throw "CMesh::CreateObjectFaceVertexIndex";
		MeshObject* pObject = &mObject[idx];
		if( fidx < 0 || fidx >= pObject->FaceNum )throw "CMesh::CreateObjectFaceVertexIndex";
		MeshObjectFace* pFace = &pObject->Face[fidx];
		if( pFace->VertexIndex )throw "CMesh::CreateObjectFaceVertexIndex";
		pFace->VertexIndex = new int[num];

		if( !pFace->VertexIndex )throw "CMesh::CreateObjectFaceVertexIndex";
		memset( pFace->VertexIndex, 0, sizeof(int)*num );

		pFace->VertexIndexNum = num;
	}catch( char* str ){
		//printf( "ERROR:[%s]\n", str);
		return false;
	}
	return true;
}

bool CMesh::SetObjectName( int indx, const char* name )
{
	strcpy( mObject[indx].ObjectName, name );
	return true;
}

bool CMesh::SetObjectVertex( int idx, int n, const MeshVertex* vertex )
{
	try{
		if( idx < 0 || idx >= mInfo.ObjectNum )throw "CMesh::SetObjectVertex";
		MeshObject* pObject = &mObject[idx];
		if( !pObject->Vertex )throw "CMesh::SetObjectVertex";
		memcpy( &pObject->Vertex[n], vertex, sizeof(MeshVertex) );
	}catch( char* str ){
		//printf( "ERROR:[%s]\n", str);
		return false;
	}
	return true;
}

bool CMesh::SetObjectFaceMaterial( int idx, int fidx, int material_id )
{
	try{
		if( material_id < 0 || material_id >= mInfo.MaterialNum )throw "CMesh::SetObjectFaceMaterial";
		if( idx < 0 || idx >= mInfo.ObjectNum )throw "CMesh::SetObjectFaceMaterial";
		MeshObject* pObject = &mObject[idx];
		if( fidx < 0 || fidx >= pObject->FaceNum )throw "CMesh::SetObjectFaceMaterial";
		MeshObjectFace* pFace = &pObject->Face[fidx];

		pFace->Material = material_id;
	}catch( char* str ){
		//printf( "ERROR:[%s]\n", str);
		return false;
	}
	return true;

}

bool CMesh::SetObjectFaceVertexIndex( int idx, int fidx, int n, const int* index )
{
	try{
		if( idx < 0 || idx >= mInfo.ObjectNum )throw "CMesh::SetObjectFaceVertexIndex";
		MeshObject* pObject = &mObject[idx];
		if( fidx < 0 || fidx >= pObject->FaceNum )throw "CMesh::SetObjectFaceVertexIndex";
		MeshObjectFace* pFace = &pObject->Face[fidx];

		if( !pFace->VertexIndex )throw "CMesh::SetObjectFaceVertexIndex";
		if( n < 0 || n >= pFace->VertexIndexNum )throw "CMesh::SetObjectFaceVertexIndex";
		
		if( n < 4 ){
			n = n;
		}
		memcpy( &pFace->VertexIndex[n], index, sizeof(int) );
	}catch( char* str ){
		//printf( "ERROR:[%s]\n", str);
		return false;
	}
	return true;
}

bool CMesh::SetObjectFacePrimitiveType( int idx, int fidx, int type )
{
	try{
		if( idx < 0 || idx >= mInfo.ObjectNum )throw "CMesh::SetObjectFaceVertexIndex";
		MeshObject* pObject = &mObject[idx];
		if( fidx < 0 || fidx >= pObject->FaceNum )throw "CMesh::SetObjectFaceVertexIndex";
		MeshObjectFace* pFace = &pObject->Face[fidx];
		
		pFace->PrimitiveType = type;

	}catch( char* str ){
		//printf( "ERROR:[%s]\n", str);
		return false;
	}
	return true;
}

bool CMesh::SetObjectCenter( int idx, CVector3* pos, float radius )
{
	try{
		if( idx < 0 || idx >= mInfo.ObjectNum )throw "CMesh::SetObjectCenter";
		MeshObject* pObject = &mObject[idx];
		
		pObject->CenterPos = *pos;
		pObject->Radius    = radius;
	}catch( char* str ){
		//printf( "ERROR:[%s]\n", str);
		return false;
	}
	return true;
}

int CMesh::GetMaterialNum()
{
	return mInfo.MaterialNum;
}

const MeshMaterial* CMesh::GetMaterial( int id )
{
	if( id < 0 || id >= mInfo.MaterialNum )return NULL;
	return &mMaterial[id];
}

int CMesh::GetObjectNum()
{
	return mInfo.ObjectNum;
}

const char* CMesh::GetObjectName( int indx )
{
	if( indx < 0 || indx >= mInfo.ObjectNum )return NULL;
	MeshObject* pObject = &mObject[indx];
	return pObject->ObjectName;
}

int CMesh::GetObjectVertexNum( int idx )
{
	if( idx < 0 || idx >= mInfo.ObjectNum )return -1;
	MeshObject* pObject = &mObject[idx];

	return pObject->VertexNum;
}

const MeshVertex* CMesh::GetObjectVertex( int idx )
{
	if( idx < 0 || idx >= mInfo.ObjectNum )return NULL;
	MeshObject* pObject = &mObject[idx];

	return pObject->Vertex;
}

int CMesh::GetObjectFaceNum( int idx )
{
	if( idx < 0 || idx >= mInfo.ObjectNum )return -1;
	MeshObject* pObject = &mObject[idx];

	return pObject->FaceNum;
}

int CMesh::GetObjectFaceMaterial( int idx, int fidx )
{
	if( idx < 0 || idx >= mInfo.ObjectNum )return -1;
	MeshObject* pObject = &mObject[idx];
	if( fidx < 0 || fidx >= pObject->FaceNum )return -1;
	MeshObjectFace* pFace = &pObject->Face[fidx];

	return pFace->Material;
}

int CMesh::GetObjectFaceVertexIndexNum( int idx, int fidx )
{
	if( idx < 0 || idx >= mInfo.ObjectNum )return -1;
	MeshObject* pObject = &mObject[idx];
	if( fidx < 0 || fidx >= pObject->FaceNum )return -1;
	MeshObjectFace* pFace = &pObject->Face[fidx];

	return pFace->VertexIndexNum;
}

const int* CMesh::GetObjectFaceVertexIndex( int idx, int fidx )
{
	if( idx < 0 || idx >= mInfo.ObjectNum )return NULL;
	MeshObject* pObject = &mObject[idx];
	if( fidx < 0 || fidx >= pObject->FaceNum )return NULL;
	MeshObjectFace* pFace = &pObject->Face[fidx];

	return pFace->VertexIndex;
}

int CMesh::GetObjectFacePrimitiveType( int idx, int fidx )
{
	if( idx < 0 || idx >= mInfo.ObjectNum )return 0;
	MeshObject* pObject = &mObject[idx];
	if( fidx < 0 || fidx >= pObject->FaceNum )return 0;
	MeshObjectFace* pFace = &pObject->Face[fidx];
	
	return pFace->PrimitiveType;
}

const CVector3* CMesh::GetObjectCenter( int idx )
{
	if( idx < 0 || idx >= mInfo.ObjectNum )return NULL;
	MeshObject* pObject = &mObject[idx];
	return &(pObject->CenterPos);
}

float CMesh::GetObjectRadius( int idx )
{
	if( idx < 0 || idx >= mInfo.ObjectNum )return NULL;
	MeshObject* pObject = &mObject[idx];
	return pObject->Radius;
}

CMatrix* CMesh::GetObjectMatrix( int idx )
{
	if( idx < 0 )return &mMatrix;
	if( idx >= mInfo.ObjectNum )return NULL;
	MeshObject* pObject = &mObject[idx];
	return &(pObject->Matrix);
}

void CMesh::DebugPrint()
{
	for( int i = 0 ; i < mInfo.MaterialNum ; i++ ){
		//printf( "MaterilaName [%s]\n", mMaterial[i].MaterilaName );
	}
	for( int i = 0 ; i < mInfo.ObjectNum ; i++ ){
		//printf( "ObjectName [%s]\n", mObject[i].ObjectName );
		for( int j = 0 ; j < mObject[i].VertexNum ; j++ ){
			//printf( "(%.4f, %.4f, %.4f)", mObject[i].Vertex[j].x, mObject[i].Vertex[j].y, mObject[i].Vertex[j].z);
		}

		//printf( "\n{\n" );
		for( int j = 0 ; j < mObject[i].FaceNum ; j++ ){
			for( int k = 0 ; k < mObject[i].Face[j].VertexIndexNum ; k++ ){
				//printf( " %d", mObject[i].Face[j].VertexIndex[k] );
			}
		}
		//printf( "}\n" );
	}
	//printf( "\n" );
}


bool CMesh::CheckHit( CVector3* p1, CVector3* p2, CVector3* out )
{
	if( !mObject )return false;
	
	bool Hit = false;
	CMatrix mtx_root = mMatrix;
	CMatrix mtx_root_inv = mtx_root;
	mtx_root_inv.Inverse();
	// MeshWnɕϊ
	CVector3 p1_root = mtx_root_inv * (*p1);
	CVector3 p2_root = mtx_root_inv * (*p2);
	
	float dist = FLT_MAX;

	for( int i = 0 ; i <  mInfo.ObjectNum ; i++ ){
		MeshObject* pObject = &mObject[i];
		//
		CMatrix mtx_obj = pObject->Matrix;
		CMatrix mtx_obj_inv = mtx_obj;
		mtx_obj_inv.Inverse();
		// ObjectWnɕϊ
		CVector3 p_local[2];
		p_local[0] = mtx_obj_inv * p1_root;
		p_local[1] = mtx_obj_inv * p2_root;
		
		// ܂̂Ɛ̌
		{
			CVector3 center = pObject->CenterPos;
			CVector3 v1 = p_local[0] - center;
			CVector3 v2 = p_local[1] - center;
			float radius = pObject->Radius;
			{// xyʂŃ`FbN
				float pt[3][2];
				pt[0][0] = v1.x;pt[0][1] = v1.y;
				pt[1][0] = v2.x;pt[1][1] = v2.y;
				if( !CheckHitObjectCircle( pt[0], pt[1], radius ) )continue;
			}
			{// xzʂŃ`FbN
				float pt[3][2];
				pt[0][0] = v1.x;pt[0][1] = v1.z;
				pt[1][0] = v2.x;pt[1][1] = v2.z;
				if( !CheckHitObjectCircle( pt[0], pt[1], radius ) )continue;
			}
			{// yzʂŃ`FbN
				float pt[3][2];
				pt[0][0] = v1.y;pt[0][1] = v1.z;
				pt[1][0] = v2.y;pt[1][1] = v2.z;
				if( !CheckHitObjectCircle( pt[0], pt[1], radius ) )continue;
			}
		}
		//printf( "[%s]\n", pObject->ObjectName );
		
		bool hit_check = false;
		CVector3 ret_tmp;
		float d_tmp = FLT_MAX;
		{
			MeshVertex* pv_top = pObject->Vertex;
			int face_num = pObject->FaceNum;
			for( int fi = 0 ; fi < face_num ; fi++ ){
				int* pIndex = pObject->Face[fi].VertexIndex;
				GLenum primitive_type = (GLenum)GetObjectFacePrimitiveType( i, fi );
				if( primitive_type ==  GL_TRIANGLES ){
					for( int fi = 0 ; fi < face_num ; fi++ ){
						int index_num = pObject->Face[fi].VertexIndexNum;
						for( int ii = 0 ; ii < index_num ; ii+=3 ){
							//printf( "Begin CheckCrossPoint\n" );
							int idx;
							MeshVertex* pv;
							CVector3 v[3];
							idx = pIndex[ii+0];
							pv = &pv_top[idx];
							v[0] = CVECTOR3( pv->x, pv->y, pv->z );
							idx = pIndex[ii+1];
							pv = &pv_top[idx];
							v[1] = CVECTOR3( pv->x, pv->y, pv->z );
							idx = pIndex[ii+2];
							pv = &pv_top[idx];
							v[2] = CVECTOR3( pv->x, pv->y, pv->z );
							CVector3 ret;
							//printf( "CheckCrossPoint\n" );
							if( CheckCrossPoint( v, p_local, &ret ) ){
								CVector3 vv = p_local[0] - ret;
								float dd = vv.GetLength();
								if( dd < d_tmp ){
									ret_tmp = ret;
									d_tmp = dd;
									hit_check = true;
								}
							}
						}
					}
				}
			}
		}
		
		if( hit_check ){
			if( d_tmp < dist ){
				*out = mtx_obj * ret_tmp;
				dist = d_tmp;
				Hit = true;
			}
		}
	}
	
	if( Hit ){
		*out = mtx_root * (*out);
	}
	return Hit;
}


bool CMesh::CheckHitBall( CVector3* p1, CVector3* p2 )
{
	if( !mObject )return false;
	
	bool Hit = false;
	CMatrix mtx_root = mMatrix;
	CMatrix mtx_root_inv = mtx_root;
	mtx_root_inv.Inverse();
	// MeshWnɕϊ
	CVector3 p1_root = mtx_root_inv * (*p1);
	CVector3 p2_root = mtx_root_inv * (*p2);
	
	float dist = FLT_MAX;

	for( int i = 0 ; i <  mInfo.ObjectNum ; i++ ){
		MeshObject* pObject = &mObject[i];
		//
		CMatrix mtx_obj = pObject->Matrix;
		CMatrix mtx_obj_inv = mtx_obj;
		mtx_obj_inv.Inverse();
		// ObjectWnɕϊ
		CVector3 p_local[2];
		p_local[0] = mtx_obj_inv * p1_root;
		p_local[1] = mtx_obj_inv * p2_root;
		
		// ܂̂Ɛ̌
		{
			CVector3 center = pObject->CenterPos;
			CVector3 v1 = p_local[0] - center;
			CVector3 v2 = p_local[1] - center;
			float radius = pObject->Radius;
			{// xyʂŃ`FbN
				float pt[3][2];
				pt[0][0] = v1.x;pt[0][1] = v1.y;
				pt[1][0] = v2.x;pt[1][1] = v2.y;
				if( !CheckHitObjectCircle( pt[0], pt[1], radius ) )continue;
			}
			{// xzʂŃ`FbN
				float pt[3][2];
				pt[0][0] = v1.x;pt[0][1] = v1.z;
				pt[1][0] = v2.x;pt[1][1] = v2.z;
				if( !CheckHitObjectCircle( pt[0], pt[1], radius ) )continue;
			}
			{// yzʂŃ`FbN
				float pt[3][2];
				pt[0][0] = v1.y;pt[0][1] = v1.z;
				pt[1][0] = v2.y;pt[1][1] = v2.z;
				if( !CheckHitObjectCircle( pt[0], pt[1], radius ) )continue;
			}
			
			return true;
		}
	}
	
	return false;
}



