////////////////////////////////////////////////////////////////////////////////
// ConvMesh.cpp
///////////////////////////////////////////////////////////////////////////////

#include "../Mesh/ConvMesh.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <float.h>
#include "../Matrix/CVector3.h"



bool ConvObject();
bool ConvMaterial();
int CreateVertexList( MeshVertex* vertex, int* inex, int* material, CMQOObjectData* srcObject);
bool SetVertex( CMQOObjectData* srcObject, int ObjectId );
bool MargeNormal( MeshVertex* vertex, int num );
void Cross3f( float* out, float* src1, float* src2 );
void Normal3f( float* v );
void Sub3f( float* target, float* src );
void CreateNormal(  float* out, float* src1, float* src2, float* src3 );

static CMQOData* pMQO;
static CMesh*    pMesh;
static float     Scale;

bool ConvMesh( CMQOData* in, CMesh* out, float scale )
{
	pMQO = in;
	pMesh = out;
	Scale = scale;

	if( !ConvMaterial() )return false;
	if( !ConvObject() )return false;

	return true;
}

bool ConvMaterial()
{
	int material_num = pMQO->GetMaterialNum();
	if( !pMesh->CreateMaterial( material_num ) )return false;

	for( int i = 0 ; i < material_num ; i++ ){
		const MQOMaterialData* pMqoMaterial = pMQO->GetMaterial( i );
		if( !pMqoMaterial )break;

		MeshMaterial MeshMaterial;

		strcpy( MeshMaterial.MaterilaName, pMqoMaterial->MaterilaName );
		strcpy( MeshMaterial.TextureName, pMqoMaterial->TextureName );

		for( int j = 0 ; j < 3 ; j++ ){
			MeshMaterial.Diffuse[j] = pMqoMaterial->Color[j];
			MeshMaterial.Ambient[j] = pMqoMaterial->Color[j] * pMqoMaterial->Ambient;
			MeshMaterial.Emissive[j] = pMqoMaterial->Color[j] * pMqoMaterial->Emissive;
			MeshMaterial.Speculer[j] = pMqoMaterial->Speculer;//pMqoMaterial->Color[j] * pMqoMaterial->Speculer;
		}
		MeshMaterial.Diffuse[3] = 1.0f;
		MeshMaterial.Ambient[3] = 1.0f;
		MeshMaterial.Emissive[3] = 1.0f;
		MeshMaterial.Speculer[3] = 1.0f;

		MeshMaterial.Power = pMqoMaterial->Power;

		pMesh->SetMaterial( i, &MeshMaterial );

	}

	return true;
}

bool ConvObject()
{
	int object_num = pMQO->GetObjectNum();
	if( !pMesh->CreateObject( object_num ) )return false;

	for( int i = 0 ; i < object_num ; i++ ){
		CMQOObjectData* pMqoObject = pMQO->GetObject( i );

		const MQOObjectInfo* pInfo = pMqoObject->GetInfo();
		//MeshObject MeshObject;
		//strcpy( MeshObject.ObjectName, pInfo->ObjectName );
		//printf( "%s\n", MeshObject.ObjectName );
		SetVertex( pMqoObject, i );
		pMesh->SetObjectName( i, pInfo->ObjectName );
	}

	return true;
}

bool SetVertex( CMQOObjectData* srcObject, int ObjectId )
{
	int src_vertex_num = srcObject->GetFaceNum();
	if( src_vertex_num == 0 ){
		if( !pMesh->CreateObjectVertex( ObjectId, 0 ) )return false;
		if( !pMesh->CreateObjectFace( ObjectId, 0 )  )return false;
		return true;
	}
	
	try {
		MeshVertex* src_vertex = new MeshVertex [src_vertex_num*6];
		if( !src_vertex )throw "";
		int*       src_vertex_inex    = new int [src_vertex_num*6];
		if( !src_vertex_inex )throw "";
		int*       src_vertex_material = new int [src_vertex_num*6];
		if( !src_vertex_material )throw "";

		src_vertex_num= CreateVertexList( src_vertex, src_vertex_inex, src_vertex_material, srcObject );
		if( src_vertex_num < 0  )return false;
		{
			float fmax = FLT_MAX;
			float fmin = FLT_MIN;
			CVector3 max = CVECTOR3( fmin, fmin, fmin );
			CVector3 min = CVECTOR3( fmax, fmax, fmax );
			MeshVertex* pv = src_vertex;
			for( int i = 0 ; i < src_vertex_num ; i++, pv++ ){
				if( pv->x > max.x )max.x = pv->x;
				if( pv->y > max.y )max.y = pv->y;
				if( pv->z > max.z )max.z = pv->z;
				if( pv->x < min.x )min.x = pv->x;
				if( pv->y < min.y )min.y = pv->y;
				if( pv->z < min.z )min.z = pv->z;
			}
			CVector3 center = (max*0.5f) + (min*0.5f);
			CVector3 v = max - center;
			v *= 1.1f;
			pMesh->SetObjectCenter( ObjectId, &center, v.GetLength() );
			//printf( " %.2f %.2f %.2f:%.2f\n", center.x, center.y, center.z, v.GetLength() );
		}
		int max_face_num = pMQO->GetMaterialNum();
		int* use_face_cnt = new int [max_face_num];
		if( !use_face_cnt )throw "";
		int face_num = 0;
		for( int i = 0 ; i < max_face_num ; i++ ){
			use_face_cnt[i] = 0;
			for( int j = 0 ; j < src_vertex_num ; j++ ){
				if( src_vertex_material[j] == i ){
					use_face_cnt[i]++;
				}
			}
			if( use_face_cnt[i] > 0 )face_num++;
		}
		if( !pMesh->CreateObjectVertex( ObjectId, src_vertex_num )  )return false;
		for( int i = 0 ; i < src_vertex_num ; i++ ){
			pMesh->SetObjectVertex( ObjectId, i, &src_vertex[i] );
		}

		if( !pMesh->CreateObjectFace( ObjectId, face_num ) )return false;
		
		int cnt = 0;
		for( int i = 0 ; i < face_num ; i++ ){
			
			for( ; cnt < max_face_num ; cnt++ ){
				if( use_face_cnt[cnt] > 0 )break;
			}
			if( !pMesh->CreateObjectFaceVertexIndex( ObjectId, i, use_face_cnt[cnt] ) )return false;
			pMesh->SetObjectFaceMaterial( ObjectId, i, cnt );

			int j = 0;
			for( int k = 0 ; k < src_vertex_num ; k++ ){
				if( src_vertex_material[k] == cnt ){
					if( src_vertex_inex[k] >= src_vertex_num ){
						int v_max = v_max;
					}
					pMesh->SetObjectFaceVertexIndex( ObjectId, i, j, &src_vertex_inex[k] );
					j++;
				}
			}
			if( j != use_face_cnt[cnt] ){
				j = j;
			}

			cnt++;
		}

		delete [] use_face_cnt;
		delete [] src_vertex;
		delete [] src_vertex_inex;
		delete [] src_vertex_material;
	}
	catch( ... ){
		src_vertex_num = src_vertex_num;
	}

	{
		int n = pMesh->GetObjectFaceNum( ObjectId );
		int v_max = pMesh->GetObjectVertexNum(ObjectId);
		for( int i = 0 ; i < n ; i++ ){
			const int* pIndex = pMesh->GetObjectFaceVertexIndex( ObjectId, i );
			int i_max = pMesh->GetObjectFaceVertexIndexNum( ObjectId, i );
			for( int j = 0 ; j < i_max ; j++ ){
				if( pIndex[j] >= v_max ){
					v_max = v_max;
				}
			}
		}
	}
	return true;
}

int CreateVertexList( MeshVertex* vertex, int* inex, int* material, CMQOObjectData* srcObject)
{
	int use_num = 0;
	int src_vertex_num = srcObject->GetFaceNum();
	const MQOObjectVertex* pV = srcObject->GetVertex();

	const MQOObjectFace* pFaceTop = srcObject->GetFace();
	for( int i = 0 ; i < src_vertex_num ; i++ ){
		int n = pFaceTop[i].VertexNum;
		if( n == 3){
			MeshVertex tmpV[3];
			for( int j = 0 ; j < 3 ; j++ ){
				tmpV[j].x = pV[pFaceTop[i].Index[j]].x * Scale;
				tmpV[j].y = pV[pFaceTop[i].Index[j]].y * Scale;
				tmpV[j].z = pV[pFaceTop[i].Index[j]].z * Scale;
				tmpV[j].u = pFaceTop[i].TexCood[j*2];
				tmpV[j].v = pFaceTop[i].TexCood[j*2+1];
				tmpV[j].nx = 0.0f;
				tmpV[j].ny = 0.0f;
				tmpV[j].nz = 0.0f;
				tmpV[j].Color[0] = 255;tmpV[j].Color[1] = 255;
				tmpV[j].Color[2] = 255;tmpV[j].Color[3] = 255;
			}
			
			float v_pos[3][3];
			for( int j = 0 ; j < 3 ; j++ ){
				memcpy( &vertex[use_num+j], &tmpV[j], sizeof(MeshVertex) );
				inex[use_num+j] = use_num+(2-j);
				material[use_num+j] = pFaceTop[i].Material;
				v_pos[j][0] = tmpV[j].x;
				v_pos[j][1] = tmpV[j].y;
				v_pos[j][2] = tmpV[j].z;
			}
			
			float Normal[3];
			CreateNormal( Normal, v_pos[0], v_pos[1], v_pos[2] );
			for( int k = 0 ; k < 3 ; k++ ){
				vertex[use_num+k].nx = -Normal[0];
				vertex[use_num+k].ny = -Normal[1];
				vertex[use_num+k].nz = -Normal[2];
			}
			
			use_num += 3;
		}else if( n == 4 ){
			MeshVertex tmpV[4];
			for( int j = 0 ; j < n ; j++ ){
				tmpV[j].x = pV[pFaceTop[i].Index[j]].x * Scale;
				tmpV[j].y = pV[pFaceTop[i].Index[j]].y * Scale;
				tmpV[j].z = pV[pFaceTop[i].Index[j]].z * Scale;
				tmpV[j].u = pFaceTop[i].TexCood[j*2];
				tmpV[j].v = pFaceTop[i].TexCood[j*2+1];
				tmpV[j].nx = 0.0f;
				tmpV[j].ny = 0.0f;
				tmpV[j].nz = 0.0f;
				tmpV[j].Color[0] = 255;tmpV[j].Color[1] = 255;
				tmpV[j].Color[2] = 255;tmpV[j].Color[3] = 255;
			}

//			int index_table[2][3] ={{0, 1, 3},{1, 2, 3}};
			int index_table[2][3] ={{3, 1, 0},{3, 2, 1}};
			for( int j = 0 ; j < 2 ; j++ ){
				for( int k = 0 ; k < 3 ; k++ ){
					int s_index = index_table[j][k];
					memcpy( &vertex[use_num+k], &tmpV[s_index], sizeof(MeshVertex) );
					inex[use_num+k] = use_num+k;
					material[use_num+k] = pFaceTop[i].Material;
					
				}

				
				float v_pos[3][3];
				for( int k = 0 ; k < 3 ; k++ ){
					int s_index = index_table[j][k];
					v_pos[k][0] =tmpV[s_index].x;
					v_pos[k][1] =tmpV[s_index].y;
					v_pos[k][2] =tmpV[s_index].z;
				}
				float Normal[3];
				CreateNormal( Normal, v_pos[0], v_pos[1], v_pos[2] );
				for( int k = 0 ; k < 3 ; k++ ){
					vertex[use_num+k].nx = Normal[0];
					vertex[use_num+k].ny = Normal[1];
					vertex[use_num+k].nz = Normal[2];
				}

				use_num += 3;
			}
		}
	}

	MargeNormal( vertex, use_num );
	return use_num;
}


void CreateNormal(  float* out, float* src1, float* src2, float* src3 )
{
	Sub3f( src1, src3 );
	Sub3f( src2, src3 );

	Normal3f( src1 );
	Normal3f( src2 );

	Cross3f( out, src1, src2 );
	Normal3f( out );
}

void Sub3f( float* target, float* src )
{
	target[0] -= src[0];
	target[1] -= src[1];
	target[2] -= src[2];
}
void Cross3f( float* out, float* src1, float* src2 )
{
	out[0] = src1[1] * src2[2] - src1[2] * src2[1];
	out[1] = src1[2] * src2[0] - src1[0] * src2[2];
	out[2] = src1[0] * src2[1] - src1[1] * src2[0];
//	A ~ B = (AyBz - AzBy, AzBx - AxBz, AxBy - AyBx)
}

void Normal3f( float* v )
{
	float d = v[0] * v[0] + v[1] * v[1] + v[2] * v[2];
	d = sqrt( d );
	v[0] /= d;
	v[1] /= d;
	v[2] /= d;
}

bool VertexCmp( MeshVertex* v1, MeshVertex* v2 )
{
	/*
	if( v1->x != v2->x )return false;
	if( v1->y != v2->y )return false;
	if( v1->z != v2->z )return false;
	*/
	if( v1->x  < (v2->x-0.001) )return false;
	if( v1->x  > (v2->x+0.001) )return false;
	if( v1->y  < (v2->y-0.001) )return false;
	if( v1->y  > (v2->y+0.001) )return false;
	if( v1->z  < (v2->z-0.001) )return false;
	if( v1->z  > (v2->z+0.001) )return false;

	return true;
}
bool MargeNormal( MeshVertex* vertex, int num )
{
	bool* bMarged = new bool [num];
	if( !bMarged ){
		return false;
	}
	int* SIndex   = new int [num];
	if( !SIndex ){
		return false;
	}
	for( int i = 0 ; i < num ; i++ ){
		bMarged[i] = false;
		SIndex[i]  = -1;
	}

	float limit_r = (180.0f / 180.0f) * 3.1415926535f;
	float limit_value = cos( limit_r );
	for( int i = 0 ; i < num ; i++ ){
		if( bMarged[i] )continue;
		SIndex[i] = i;
		CVector3 vec1 = CVector3( vertex[i].nx, vertex[i].ny, vertex[i].nz);
		vec1.Normalize();

		CVector3 sum = vec1;
		int      cnt = 1;
		for( int j = i + 1 ; j < num ; j++ ){
			if( bMarged[j] )continue;

			if( !VertexCmp( &vertex[i], &vertex[j] ) )continue;
			CVector3 vec2 = CVector3( vertex[j].nx, vertex[j].ny, vertex[j].nz);
			vec2.Normalize();

			float cs = vec1.InnerProduct( vec2 );
			if( cs < limit_value )continue;
			sum += vec2;
			cnt++;
			SIndex[j] = i;
		}

		CVector3 nml = sum / float(cnt);
		nml.Normalize();

		int mg_num = 0;
		for( int j = i ; j < num ; j++ ){
			if( SIndex[j] == i ){
				vertex[j].nx = nml.x;
				vertex[j].ny = nml.y;
				vertex[j].nz = nml.z;

				bMarged[j] = true;
				mg_num++;
			}
		}
		mg_num = mg_num;
	}

	delete [] SIndex;
	delete [] bMarged;
	return true;
}


