﻿// Copyright (C) 2003 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2005-2008, 2010, 2013, 2014 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: PriceUpdateDialog.cs 503 2014-02-10 06:10:07Z panacoran $

using System;
using System.ComponentModel;
using System.Windows.Forms;
using Protra.Lib;
using Protra.Lib.Data;
using Protra.Lib.Dialogs;
using Protra.Lib.Update;

namespace Protra.Dialogs
{
    /// <summary>
    /// 株価データを更新するダイアログ
    /// </summary>
    public partial class PriceUpdateDialog : Form
    {
        private readonly PriceDataUpdator _updator;
        private DateTime _beginDate;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PriceUpdateDialog()
        {
            InitializeComponent();
            _updator = PriceDataUpdator.Create();
        }

        private void PriceUpdateDialog_Load(object sender, EventArgs e)
        {
            // ロードされると同時に更新を開始します
            var maxDate = PriceData.MaxDate;
            if (maxDate != DateTime.MinValue)
                _beginDate = maxDate.AddDays(1);
            else
            {
                // データが存在しない場合には更新開始日時を入力させる。
                var dateDialog = new InputDateDialog
                {
                    Message = "データが存在しません。\n指定した日付以降のデータを取得します。",
                    Value = _updator.DataSince
                };
                dateDialog.ShowDialog(Owner); // thisはまだ表示されていない。
                if (dateDialog.DialogResult != DialogResult.OK)
                {
                    Close();
                    return;
                }
                _beginDate = dateDialog.Value;
            }
            backgroundWorkerUpdate.RunWorkerAsync(_beginDate);
        }

        private void backgroundWorkerUpdate_DoWork(object sender, DoWorkEventArgs e)
        {
            _updator.Update(sender as BackgroundWorker, e);
        }

        private void backgroundWorkerUpdate_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            labelInfo.Text = "更新しています... " + (string)e.UserState;
            progressBar.Value = e.ProgressPercentage;
        }

        private void buttonCancel_Click(object sender, EventArgs e)
        {
            if (!backgroundWorkerUpdate.IsBusy)
                return;
            backgroundWorkerUpdate.CancelAsync();
            labelInfo.Text = "中断しています...";
            Cursor = Cursors.WaitCursor;
            buttonCancel.Enabled = false;
            DialogResult = DialogResult.None;
        }

        private void backgroundWorkerUpdate_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            if (e.Error != null)
            {
                using (new CenteredDialogHelper())
                    MessageBox.Show(this, e.Error.Message, "エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
                labelInfo.Text = "更新中にエラーが発生したため継続できません。";
                buttonCancel.Text = "OK";
                return;
            }
            if (e.Cancelled)
            {
                labelInfo.Text = "中断しました。";
                Cursor = Cursors.Arrow;
                buttonCancel.Enabled = true;
                buttonCancel.Text = "OK";
            }
            else
            {
                labelInfo.Text = "更新が完了しました。";
                Cursor = Cursors.Arrow; // 中断が押されても完了した場合。
                buttonCancel.Enabled = true; // 同上。
                buttonCancel.Text = "OK";
            }
        }
    }
}