﻿// Copyright (C) 2003, 2005 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2008, 2010 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: Interpreter.cs 444 2013-03-11 04:41:16Z panacoran $

using System.Collections.Generic;

namespace Protra.Lib.Lang
{
    /// <summary>
    /// プログラムの実行処理を担うクラス。
    /// </summary>
    public class Interpreter
    {
        /// <summary>
        /// プログラムのノード
        /// </summary>
        readonly Node node = new ProgramNode();
        /// <summary>
        /// リソース
        /// </summary>
        private readonly Resource resource;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.ParseException">
        /// プログラムの構文解析中にエラーが発生した場合にthrowされる。
        /// </exception>
        /// <param name="file">プログラムファイルのパス</param>
        public Interpreter(string file)
        {
            Node.Scanner = new Scanner(file);
            // 関数の外のローカル変数の記号表を用意する。
            Node.LvtStack.Push(new Dictionary<string, int>());
            try
            {
                node = new ProgramNode();
                node.Parse();
            }
            catch (ParseException exc)
            {
                Token t = exc.Token;
                throw new ParseException(
                    "parse error in " + t.Filename + ":" + t.LineNo + ": " + exc.Message, null);
            }
            finally
            {
                Node.Scanner.Close();
            }
            resource = new Resource();
        }

        /// <summary>
        /// プログラムを実行する。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされる。
        /// </exception>
        /// <returns>プログラムの戻り値。</returns>
        public Value Execute()
        {
            resource.Stack.Clear();
            resource.FunctionTable.Clear();
            try
            {
                return node.Execute(resource, 0, null);
            }
            catch (RuntimeException e)
            {
                Token token = e.Token;
                throw new RuntimeException
                    ("runtime error in " + token.Filename + ":" + token.LineNo + ": " + e.Message, null);
            }
        }

        /// <summary>
        /// グローバル変数のテーブルを取得または設定する。
        /// </summary>
        public Dictionary<string, Value> GlobalVariableTable
        {
            get { return resource.GlobalVariableTable; }
            set { resource.GlobalVariableTable = value; }
        }

        /// <summary>
        /// 組み込み関数を処理するクラスのインスタンスを取得または設定する。
        /// </summary>
        public Builtins.BasicBuiltins Builtins
        {
            get { return resource.Builtins; }
            set { resource.Builtins = value; }
        }
    }
}
