﻿// Copyright (C) 2003 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2004, 2005 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: BrandList.cs 306 2010-03-19 13:21:22Z panacoran $

using System;
using System.Collections;

namespace Protra.Lib.Db
{
	/// <summary>
	/// BrandListテーブルのレコードを表すと同時に銘柄リストの操作を提供する。
	/// </summary>
	public class BrandList: IRecord
	{
		private int id;
		private string name;

		/// <summary>
		/// 既定のコンストラクタ
		/// </summary>
		public BrandList() {}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public BrandList(int id, string name)
		{
			this.id = id;
			this.name = name;
		}

		/// <summary>
		/// 銘柄リストIDを取得する。
		/// </summary>
		public int Id
		{
			get { return id; }
		}

		/// <summary>
		/// 名前を取得あるいは変更する。
		/// </summary>
		public string Name
		{
			get { return name; }
			set { 
				BrandListTable.ChangeBrandListName(id, value);
				name = value;
			}
		}

		/// <summary>
		/// 銘柄リストを取得する。
		/// </summary>
		public Brand[] Brands
		{
			get {
				return BrandList_BrandTable.GetBrands(id);
			}
		}

		/// <summary>
		/// 銘柄をリストに追加する。
		/// </summary>
		/// <param name="brandId">BrandId</param>
		public void Add(int brandId)
		{
			BrandList_BrandTable.Add(id, brandId);
		}

		/// <summary>
		/// 銘柄リストを削除する。
		/// </summary>
		public void Clear()
		{
			BrandList_BrandTable.Delete(id);
		}

		/// <summary>
		/// 銘柄リストそのものを削除する。
		/// </summary>
		public void Delete()
		{
			BrandList_BrandTable.Delete(id);
			BrandListTable.Delete(id);
		}

		/// <summary>
		/// データベースのレコードをオブジェクトに変換する。
		/// </summary>
		/// <param name="values">ネイティブ形式の値の配列</param>
		/// <returns>変換後のBrandListオブジェクトを返す。</returns>
		public Object ToObject(Object[] values)
		{
			return new BrandList((int)values[0], // @Id
								 (string)values[1]); // @Name
		}

		/// <summary>
		/// インスタンスを文字列表現に変換する。
		/// </summary>
		/// <returns>文字列表現</returns>
		public override string ToString()
		{
			return string.Format(
				"BrandList[Id={0}, Name={1}]", id, name);
		}
	}

	/// <summary>
	/// BrandList_Brandテーブルのレコードを表す。
	/// </summary>
	public class BrandList_Brand: IRecord
	{
		private int brandListId;
		private int brandId;

		/// <summary>
		/// 既定のコンストラクタ
		/// </summary>
		public BrandList_Brand() {}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public BrandList_Brand(int brandListId, int brandId)
		{
			this.brandListId = brandListId;
			this.brandId = brandId;
		}

		/// <summary>
		/// 銘柄リストIDを取得する。
		/// </summary>
		public int BrandListId
		{
			get { return brandListId; }
		}

		/// <summary>
		/// 銘柄IDを取得する。
		/// </summary>
		public int BrandId
		{
			get { return brandId; }
		}

		/// <summary>
		/// データベースのレコードをオブジェクトに変換する。
		/// </summary>
		/// <param name="values">ネイティブ形式の値の配列</param>
		/// <returns>変換後のBrandListオブジェクトを返す。</returns>
		public Object ToObject(Object[] values)
		{
			return new BrandList_Brand((int)values[0], // @BrandListId
									   (int)values[1]); // @BrandId
		}

		/// <summary>
		/// インスタンスを文字列表現に変換する。
		/// </summary>
		/// <returns>文字列表現</returns>
		public override string ToString()
		{
			return string.Format(
				"BrandList_Brand[BrandListId={0}, BrandId={1}]",
				brandListId, brandId);
		}
	}

	/// <summary>
	/// BrandListテーブルを操作する。
	/// </summary>
	public class BrandListTable
	{
		private static Connection conn;
		private static Connection Conn
		{
			get 
			{
				if (conn != null)
					return conn;
				conn = new Connection("protra");
				return conn;
			}
		}

		/// <summary>
		/// 指定された銘柄リストIDのレコードを取得する。
		/// </summary>
		/// <param name="id">銘柄リストID</param>
		/// <returns>BrandListオブジェクトを返す。レコードが存在しない場合にはnullを返す。</returns>
		public static BrandList GetRecord(int id) 
		{
			string sql = string.Format("SELECT * FROM BrandList WHERE @Id={0}", id);
			ArrayList result = Conn.Query(sql, new BrandList());
			return result.Count == 0 ? null : (BrandList)result[0];
		}

		/// <summary>
		/// すべてのレコードを取得する。
		/// </summary>
		/// <returns>BrandListオブジェクトの配列を返す。</returns>
		public static BrandList[] GetAllRecords()
		{
			string sql = "SELECT * FROM BrandList ORDER BY @Id";
			ArrayList result = Conn.Query(sql, new BrandList());
			return (BrandList[])result.ToArray(typeof(BrandList));
		}

		/// <summary>
		/// レコードを追加する。
		/// </summary>
		/// <param name="name">名前</param>
		/// <returns>追加されたレコードを返す</returns>
		public static BrandList Add(string name)
		{
			string sql = string.Format("INSERT INTO BrandList(@Name) VALUES('{0}')", name);
			Conn.Execute(sql);
			BrandList[] all = GetAllRecords();
			return all[all.Length - 1];
		}

		/// <summary>
		/// 銘柄リスト名を変更する。
		/// </summary>
		/// <param name="id">銘柄リストID</param>
		/// <param name="name">新しい名前</param>
		public static void ChangeBrandListName(int id, string name)
		{
			string sql = string.Format(
				"UPDATE BrandList SET @Name='{0}' WHERE @Id={1}", name, id);
			Conn.Execute(sql);
		}

		/// <summary>
		/// 指定された銘柄リストIDのデータを削除する。
		/// </summary>
		public static void Delete(int id)
		{
			string sql = string.Format("DELETE FROM BrandList WHERE @Id={0}", id);
			Conn.Execute(sql);
		}

		/// <summary>
		/// 全データを削除する。
		/// </summary>
		public static void Delete()
		{
			string sql = "DELETE FROM BrandList";
			Conn.Execute(sql);
		}
	}

	/// <summary>
	/// BrandList_Brandテーブルを操作する。
	/// </summary>
	public class BrandList_BrandTable
	{
		private static Connection conn;
		private static Connection Conn
		{
			get 
			{
				if (conn != null)
					return conn;
				conn = new Connection("protra");
				return conn;
			}
		}

		/// <summary>
		/// 指定された銘柄リストIDと対応付けられた銘柄IDを持つBrandテーブルのレコードを取得する。
		/// </summary>
		/// <param name="brandListId">銘柄リストID</param>
		/// <returns>Brandオブジェクトの配列を返す。</returns>
		public static Brand[] GetBrands(int brandListId) 
		{
			string sql = string.Format(
				"SELECT @Id, @Code, @MarketId, @Name " +
				"FROM BrandList_Brand, Brand " +
				"WHERE @BrandListId={0} AND @BrandId=@Id " +
				"ORDER BY @Code, @MarketId", brandListId);
			ArrayList result = Conn.Query(sql, new Brand());
			return (Brand[])result.ToArray(typeof(Brand));
		}

		/// <summary>
		/// レコードを追加する。
		/// </summary>
		/// <param name="brandListId">銘柄リストID</param>
		/// <param name="brandId">銘柄ID</param>
		public static BrandList_Brand Add(int brandListId, int brandId)
		{
			string sql = string.Format(
				"INSERT INTO BrandList_Brand(@BrandListId, @BrandId) " +
				"VALUES({0}, '{1}')", brandListId, brandId);
			Conn.Execute(sql);
			BrandList_Brand record =
				new BrandList_Brand(brandListId, brandId);
			return record;
		}

		/// <summary>
		/// 指定された銘柄リストIDのレコードを削除する。
		/// </summary>
		/// <param name="brandListId">銘柄リストId</param>
		public static void Delete(int brandListId)
		{
			string sql = string.Format(
				"DELETE FROM BrandList_Brand WHERE @BrandListId={0}",
				brandListId);
			Conn.Execute(sql);
		}

		/// <summary>
		/// 全データを削除する。
		/// </summary>
		public static void Delete()
		{
			string sql = "DELETE FROM BrandList_Brand";
			Conn.Execute(sql);
		}
	}
}
