﻿// Copyright (C) 2003 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2004, 2005, 2007 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: Connection.cs 415 2011-03-23 06:00:35Z panacoran $

using System;
using System.Collections;
using System.Data;
using System.Data.OleDb;
using System.IO;
using System.Windows.Forms;
using Protra.Lib.Dialogs;

namespace Protra.Lib.Db
{
	/// <summary>
	/// レコードをオブジェクトに変換するメソッドを定義する。
	/// </summary>
	public interface IRecord
	{
		/// <summary>
		/// データベースのレコードをオブジェクトに変換する。
		/// </summary>
		/// <param name="record">ネイティブ形式の値の配列</param>
		/// <returns>変換後のオブジェクトを返す。</returns>
		Object ToObject(Object[] record);
	}

	/// <summary>
	/// OLE DB経由でAccessデータベースを操作するクラス。
	/// DBへの接続をプロセス全体で共有する。
	/// </summary>
	public class Connection
	{
		/// <summary>
		/// 例外を投げるかどうかを取得または設定する。
		/// </summary>
		public static bool ThrowException { set; get; }

		/// <summary>
		/// OleDbErrorの内容を表示する。
		/// </summary>
		/// <param name="e">OleDbExceptionのインスタンス。</param>
		/// <param name="desc">エラーの原因の記述。</param>
		public static void ShowOleDbError(OleDbException e, string desc)
		{
			if (ThrowException)
				throw e;
			string msg = "データベースのアクセスでエラーが発生しました:\n";
			if (desc != null)
				msg += desc + "\n";
			for (int i = 0; i < e.Errors.Count; i++)
			{
				msg += "Index #" + i + "\n" +
					"Message: " + e.Errors[i].Message + "\n" +
					"NativeError: " + e.Errors[i].NativeError + "\n" +
					"Source: " + e.Errors[i].Source + "\n" +
					"SQLState: " + e.Errors[i].SQLState;
			}
            using (var dialog = new ApplicationError())
            {
                dialog.ErrorMessage = msg;
                dialog.Mode = ApplicationError.ErrorType.Fatal;
                dialog.ShowDialog();
            }
            Environment.Exit(1);
		}

		private static OleDbConnection conn;

		/// <summary>
		/// データベースに接続する。
		/// </summary>
		public static void Open()
		{
			string target = "Provider=Microsoft.JET.OLEDB.4.0;" +
					"data source=" + Path.Combine(Global.DirData, "protra.mdb");
			conn = new OleDbConnection(target);
			try
			{
				conn.Open();
			}
			catch (OleDbException e)
			{
				ShowOleDbError(e, target);
			}
		}

		/// <summary>
		/// クエリの実行結果をオブジェクトのリストで返す。
		/// </summary>
		/// <param name="sql">SQL文を指定する。</param>
		/// <param name="cvtr">変換メソッドを実装したクラスのインスタンスを指定する。</param>
		/// <returns>cvtrによって変換されたオブジェクトのリストを返す。</returns>
		public static ArrayList Query(string sql, IRecord cvtr)
		{
			ArrayList list = new ArrayList();
			OleDbCommand comm = new OleDbCommand(sql, conn);
			OleDbDataReader reader = null;
			lock (conn)
			{
				try
				{
					reader = comm.ExecuteReader();
				}
				catch (OleDbException e)
				{
					ShowOleDbError(e, sql);
				}
				try
				{
					while (reader.Read())
					{
						Object[] values = new Object[reader.FieldCount];
						reader.GetValues(values);
						list.Add(cvtr.ToObject(values));
					}
				}
				finally
				{
					reader.Close();
				}
			}
			return list;
		}

		/// <summary>
		/// クエリの結果の最初のレコードの最初のフィールドを返す。
		/// </summary>
		/// <param name="sql">SQL文を指定する。</param>
		/// <returns>最初のレコードの最初のフィールド。</returns>
		public static Object QueryScalar(string sql)
		{
			OleDbCommand comm = new OleDbCommand(sql, conn);
			lock (conn)
			{
				try 
				{
					return comm.ExecuteScalar();
				}
				catch (OleDbException e) 
				{
					ShowOleDbError(e, sql);
				}
				return null;
			}
		}

		/// <summary>
		/// SQL文を実行して影響を受けた行の数を返す。
		/// </summary>
		/// <param name="sql">SQL文を指定する。</param>
		/// <returns>影響を受けた行の数を返す。</returns>
		public static int Execute(string sql)
		{
			OleDbCommand comm = new OleDbCommand(sql, conn);
			lock (conn)
			{
				try
				{
					return  comm.ExecuteNonQuery();
				}
				catch (OleDbException e)
				{
					ShowOleDbError(e, sql);
				}
				return 0;
			}
		}

		/// <summary>
		/// データベースへの接続を閉じる。
		/// </summary>
		public static void Close()
		{
			conn.Close();
		}
	}
}
