/*
 *  Controller.cpp
 */

#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>

#include "Controller.h"
#include "Bridge.h"
#include "debug.h"


namespace PTx
{
namespace PTxD
{

void *Controller_run(void *arg)
{
    ((Controller *)arg)->run();
    return NULL;
}

Controller::Controller()
{
    // initialize
#if 0
    pthread_mutexattr_t mutexattr;
    pthread_mutexattr_init(&mutexattr);
    pthread_mutexattr_settype(&mutexattr, PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init(&_lock, &mutexattr);
    pthread_mutexattr_destroy(&mutexattr);
#else
    pthread_mutex_init(&_lock, NULL);
#endif
    pthread_cond_init(&_cond, NULL);

    _props = NULL;
    for (int i = 0; i < MAX_TUNER_COUNT * MAX_DEV_COUNT; ++i)
    {
        _pathsForRecording[i] = NULL;
        _fdsForRecording[i] = -1;
    }
}

Controller::~Controller()
{
    pthread_mutex_destroy(&_lock);
    pthread_cond_destroy(&_cond);
    if (_props != NULL)
    {
        // M:release
        delete _props;
    }
    for (int i = 0; i < MAX_TUNER_COUNT * MAX_DEV_COUNT; ++i)
    {
        if (_pathsForRecording[i] != NULL)
        {
            free(_pathsForRecording[i]);
        }
        if (_fdsForRecording[i] >= 0)
        {
            close(_fdsForRecording[i]);
        }
    }
}

bool Controller::init_properties(const char *path)
{
    DebugLog2("Controller::%s(): %s\n", __FUNCTION__, path);

    // M:alloc
    _props = new Properties();
    if (!_props->init(path))
    {
        //
        DebugLog1("can't open properties file: %s\n", path);
        return false;
    }

    // HTTP Port
    if (_props->integerForKey(kPTxD_KEY_HTTP_PORT) == 0)
    {
        _props->setIntegerForKey(50080, kPTxD_KEY_HTTP_PORT);
    }

    // HC Port
    if (_props->integerForKey(kPTxD_KEY_HC_PORT) == 0)
    {
        _props->setIntegerForKey(50080, kPTxD_KEY_HC_PORT);
    }

    // Root Path
    char *rootPath = _props->stringForKey(kPTxD_KEY_ROOT_PATH);
    if (rootPath == NULL)
    {
        _props->setStringForKey("/Users/Shared/PTxD", kPTxD_KEY_ROOT_PATH);
    }
    else
    {
        free(rootPath);
    }

    // root path check

    return true;
}

bool Controller::init_tuners()
{
    DebugLog2("Controller::%s()\n", __FUNCTION__);

    // assertion on

    // scan
    _tuner_count = Tuner::scan(_tuners);
    if (_tuner_count <= 0)
    {
        DebugLog1("tuner not exists.\n");
        return false;
    }

    bool needs_update = false;

    // M:alloc
    Dictionary *tunersInfo = _props->dictionaryForKey(kPTxD_KEY_TUNERS);
    if (tunersInfo == NULL)
    {
        tunersInfo = new Dictionary();
        needs_update = true;
    }
    for (int i = 0; i < _tuner_count; ++i)
    {
        // M:alloc
        Dictionary *tunerInfo = tunersInfo->dictionaryForKey(_tuners[i]->name());
        if (tunerInfo == NULL)
        {
            tunerInfo = new Dictionary();
            needs_update = true;
        }

        if (!tunerInfo->boolForKey(kPTxD_KEY_INITIALIZED))
        {
            DebugLog2("channel scan for Tuner %d\n", i);

            needs_update = true;

            ISDB isdb = _tuners[i]->isdb();
            tunerInfo->setIntegerForKey(isdb, kPTxD_KEY_TYPE);
            DebugLog2("Tuner type is %s.\n", isdb == ISDB_S ? "ISDB-S" : "ISDB-T");

            // M:alloc
            Dictionary *channelsInfo = new Dictionary();

            int currentChannel = -1;
            for (int ch = 0; ch <= ((isdb == ISDB_S) ? TunerMaxChannel_ISDB_S : TunerMaxChannel_ISDB_T); ++ch)
            {
                // M:alloc
                Dictionary *channelInfo = new Dictionary();

                if (_tuners[i]->setChannel(ch))
                {
                    DebugLog2("success: set channel: %d\n", ch);
                    currentChannel = ch;

                    // M:alloc
                    Dictionary *stationInfo = _tuners[i]->stationInfo();
                    if (stationInfo != NULL)
                    {
                        stationInfo->dump();
                        Dictionary *self = stationInfo->dictionaryForKey(MPEG2_TS_SDT_SELF);
                        if (self != NULL)
                        {
                            char *name = self->stringForKey(MPEG2_TS_SDT_NAME);
                            if (name != NULL)
                            {
                                channelInfo->setStringForKey(name, MPEG2_TS_SDT_NAME);
                                free(name);
                            }
                            Dictionary *services = self->dictionaryForKey(MPEG2_TS_SDT_SERVICES);
                            if (services != NULL)
                            {
                                channelInfo->setDictionaryForKey(services, MPEG2_TS_SDT_SERVICES);
                                delete services;
                            }
                            delete self;
                        }

                        // M:release
                        delete stationInfo;
                    }
                }
                else
                {
                    DebugLog3("failed: set channel: %d\n", ch);
                }

                char channelID[64];
                if (isdb == ISDB_S)
                {
                    if (ch < 12)
                    {
                        sprintf(channelID, "BS%02d", 1 + 2 * ch);
                    }
                    else if (ch < 24)
                    {
                        sprintf(channelID, "ND%02d", 2 + 2 * (ch - 12));
                    }
                    else
                    {
                        sprintf(channelID, "ND%02d", 1 + 2 * (ch -24));
                    }
                }
                else
                {
                    static uint TABLE[][3] =
                    {
                        {   2,  0,   3 },
                        {  12,  1,  22 },
                        {  21,  0,  12 },
                        {  62,  1,  63 },
                        { 112,  0,  62 }
                    };

                    uint i;
                    for (i = 0; i < sizeof(TABLE)/sizeof(*TABLE); ++i)
                    {
                        if (ch <= TABLE[i][0])
                        {
                            sprintf(channelID, "%s%d", TABLE[i][1] ? "C" : "", ch + TABLE[i][2] - TABLE[i][0]);
                            break;
                        }
                    }
                }

                channelInfo->setStringForKey(channelID, kPTxD_KEY_CHANNEL_ID);

                char key[16];
                sprintf(key, "%03d", ch);
                channelsInfo->setDictionaryForKey(channelInfo, key);

                // M:release
                delete channelInfo;
            }

            tunerInfo->setDictionaryForKey(channelsInfo, kPTxD_KEY_CHANNELS);
            tunerInfo->setIntegerForKey(currentChannel, kPTxD_KEY_CURRENT_CHANNEL);
            tunerInfo->setBoolForKey(true, kPTxD_KEY_INITIALIZED);

            // M:release
            delete channelsInfo;
        }

        tunersInfo->setDictionaryForKey(tunerInfo, _tuners[i]->name());

        // M:release
        delete tunerInfo;
    }
    if (needs_update)
    {
        _props->setDictionaryForKey(tunersInfo, kPTxD_KEY_TUNERS);
    }

    // M:release
    delete tunersInfo;

    // assetion off

    return true;
}

bool Controller::init_httpd()
{
    DebugLog2("Controller::%s()\n", __FUNCTION__);

    if (pthread_mutex_lock(&_lock) == 0)
    {
        _status = ST_IDLE;

        pthread_t pid;
        pthread_create(&pid, NULL, Controller_run, this);
        pthread_detach(pid);

        while (_status == ST_IDLE)
        {
            pthread_cond_wait(&_cond, &_lock);
        }

        pthread_mutex_unlock(&_lock);
    }

    if (_status != ST_RUN)
    {
        DebugLog3("start httpd: ng.\n");
        return false;
    }

    return true;
}

bool Controller::start(const char *path)
{
    DebugLog0("start...\n");

    //
    if (!init_properties(path))
    {
        return false;
    }

    //
    if (!init_tuners())
    {
        return false;
    }

    //
    if (!init_httpd())
    {
        return false;
    }

    //
    return true;
}

void Controller::stop()
{
    // stop Tuner

    // stop C/S interface
    if (pthread_mutex_lock(&_lock) == 0)
    {
        _status = ST_STOP;
        pthread_mutex_unlock(&_lock);
    }
    DebugLog0("stop...\n");
}

void Controller::waitFinished()
{
    if (pthread_mutex_lock(&_lock) == 0)
    {
        while (_status != ST_DONE)
        {
            pthread_cond_wait(&_cond, &_lock);
        }
        pthread_mutex_unlock(&_lock);
    }
    DebugLog0("finished.\n");
}

//
// httpd & tick
//
void Controller::run()
{
    while (true)
    {
        if ((_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
        {
            DebugLog3("Can't create socket: %s\n", strerror(errno));
            break;
        }

        int reuse = 1;
        if (setsockopt(_fd, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(int)) < 0)
        {
            DebugLog3("Can't set sockopt5: %s\n", strerror(errno));
            close(_fd);
            break;
        }

        struct sockaddr_in own_addr;
        bzero((char *)&own_addr, sizeof(own_addr));
        own_addr.sin_family = AF_INET;
        own_addr.sin_port = htons(_props->integerForKey(kPTxD_KEY_HTTP_PORT));
        own_addr.sin_addr.s_addr = htonl(INADDR_ANY);
        if (bind(_fd, (struct sockaddr *)&own_addr, sizeof(own_addr)) < 0)
        {
            DebugLog3("Can't bind socket: %s\n", strerror(errno));
            close(_fd);
            break;
        }

        if (listen(_fd, 10) < 0)
        {
            DebugLog3("Can't listen socket: %s\n", strerror(errno));
            close(_fd);
            break;
        }

        if (pthread_mutex_lock(&_lock) == 0)
        {
            _status = ST_RUN;
            pthread_cond_signal(&_cond);
            pthread_mutex_unlock(&_lock);
        }

        DebugLog1("httpd started: port %d\n", _props->integerForKey(kPTxD_KEY_HTTP_PORT));

        bool done = false;
        while (!done)
        {
            fd_set fdset;
            FD_ZERO(&fdset);
            FD_SET(_fd, &fdset);
            struct timeval timeout = {0, 100000};

            if (select(_fd + 1, &fdset, nil, nil, &timeout) < 0)
            {
                DebugLog3("select error: %s\n", strerror(errno));
                break;
            }

            if (FD_ISSET(_fd, &fdset))
            {
                // session
                struct sockaddr_in acc_addr;
                uint sock_len = sizeof(acc_addr);
                int newfd;
                if ((newfd = accept(_fd, (struct sockaddr *)&acc_addr, &sock_len)) >= 0)
                {
                    if (acc_addr.sin_family == AF_INET)
                    {
                        char client[256];
                        sprintf(client, "%d.%d.%d.%d",
                                ((unsigned char *)&acc_addr.sin_addr)[0],
                                ((unsigned char *)&acc_addr.sin_addr)[1],
                                ((unsigned char *)&acc_addr.sin_addr)[2],
                                ((unsigned char *)&acc_addr.sin_addr)[3]);
                        request(newfd, client);
                    }

                    close(newfd);
                }
                else
                {
                    DebugLog3("accept error: %s\n", strerror(errno));
                }
            }
            else
            {
                tick();
            }

            if (pthread_mutex_lock(&_lock) == 0)
            {
                done = (_status != ST_RUN);
                pthread_mutex_unlock(&_lock);
            }
        }
        break;
    }

    if (pthread_mutex_lock(&_lock) == 0)
    {
        _status = ST_DONE;
        pthread_cond_signal(&_cond);
        pthread_mutex_unlock(&_lock);
    }
}

static uint32_t decimalStringToUINT32(const char *dec)
{
    uint32_t result = UINT32_MAX;
    if (strcmp(dec, "0") == 0)
    {
        result = 0;
    }
    else if (isdigit(dec[0]) && (dec[0] != '0'))
    {
        char *e = NULL;
        unsigned long t = strtoul(dec, &e, 10);
        if (*e == '\0')
        {
            result = t;
        }
    }
    return result;
}

void Controller::request(int fd, char *client)
{
    char line[256];
    int offset = 0;
    while (offset < sizeof(line) - 1)
    {
        ssize_t len = read(fd, &line[offset], 1);
        if (len == 0)
        {
            break;
        }
        if (offset > 0)
        {
            if ((line[offset - 1] == 0x0d) && (line[offset] == 0x0a))
            {
                ++offset;
                break;
            }
        }
        ++offset;
    }
    line[offset] = '\0';

    char *uri = strchr(line, ' ');
    if (uri != NULL)
    {
        *uri = '\0';
        ++uri;
        char *ver = strchr(uri, ' ');
        if (ver != NULL)
        {
            *ver = '\0';
            ++ver;
        }

        int rc = kPTxD_HTTP_STATUS_NOT_IMPLEMENTED;
        Dictionary *body = NULL;

        if (strcmp(line, "GET") == 0)
        {
            rc = kPTxD_HTTP_STATUS_BAD_REQUEST;
            if (strcmp(uri, "/") == 0)
            {
            }
            else if (strcmp(uri, "/" kPTxD_FILE_INFO) == 0)
            {
                // system info.
                DebugLog2("sys info.\n");
                body = new Dictionary();
                if (body != NULL)
                {
                    if (body->init())
                    {
                        // collect
                        body->setStringForKey(kPTxD_SYS_NAME,    kNetTunerKeySystemName);
                        body->setStringForKey(kPTxD_SYS_VERSION, kNetTunerKeySystemVersion);
                        body->setIntegerForKey(_tuner_count,     kNetTunerKeySystemNumberOfTuners);
                        char *path = _props->stringForKey(kPTxD_KEY_ROOT_PATH);
                        if (path != NULL)
                        {
                            if (strcmp(client, "127.0.0.1") == 0)
                            {
                                // local
                                char *url = (char *)malloc(strlen(path) + 8);
                                if (url != NULL)
                                {
                                    sprintf(url, "file://%s", path);
                                    body->setStringForKey(url, kNetTunerKeySystemRootURL);
                                    free(url);
                                }
                            }
                            else
                            {
                                // remote
                            }
                            free(path);
                        }
                        rc = kPTxD_HTTP_STATUS_OK;
                    }
                }
            }
            else if (strcmp(uri, "/" kPTxD_FILE_STATUS) == 0)
            {
                printf("status\n");
            }
            else if (strcmp(uri, kPTxD_FILE_SHUTDOWN) == 0)
            {
                printf("shutdown\n");
            }
            else if (strcmp(uri, kPTxD_FILE_REBOOT) == 0)
            {
                printf("reboot\n");
            }
            else if (strcmp(uri, kPTxD_FILE_SLEEP) == 0)
            {
                printf("sleep\n");
            }
            else
            {
                char *p = strchr(&uri[1], '/');
                if (p != NULL)
                {
                    *p = '\0';
                    uint32_t tuner = UINT32_MAX;
                    tuner = decimalStringToUINT32(&uri[1]);
                    *p = '/';
                    if (tuner != UINT32_MAX)
                    {
                        request(fd, client, tuner, p, &rc, &body);
                    }
                }
            }
        }

        if (rc > 0)
        {
            const char *phrase;
            switch (rc)
            {
                case kPTxD_HTTP_STATUS_OK:
                    phrase = "OK";
                    break;
                case kPTxD_HTTP_STATUS_BAD_REQUEST:
                    phrase = "Bad Request";
                    break;
                case kPTxD_HTTP_STATUS_NOT_FOUND:
                    phrase = "Not Found";
                    break;
                case kPTxD_HTTP_STATUS_NOT_ACCEPTABLE:
                    phrase = "Not Acceptable";
                    break;
                case kPTxD_HTTP_STATUS_NOT_IMPLEMENTED:
                default:
                    phrase = "Not Implemented";
                    break;
            }

            // Status-Line
            char response[256];
            sprintf(response, "HTTP/1.1 %03d %s\r\n", rc, phrase);
            write(fd, response, strlen(response));

            // CRLF
            sprintf(response, "\r\n");
            write(fd, response, strlen(response));

            if (body == NULL)
            {
                sprintf(response, "<html><head><title>%s</title></head><body>%s</body></html>", phrase, phrase);
                write(fd, response, strlen(response));
            }
            else
            {
                body->writeXmlData(fd);
            }
        }
        if (body != NULL)
        {
            delete body;
        }
    }
}

void Controller::request(int fd, char *client, uint32_t tuner, char *path, int *status, Dictionary **body)
{
    printf("request from: %s, tuner: %d, path: %s\n", client, tuner, path);
    int sc = kPTxD_HTTP_STATUS_BAD_REQUEST;
    if (tuner < _tuner_count)
    {
        if (strcmp(path, "/" kPTxD_FILE_INFO) == 0)
        {
            DebugLog2("tuner info.\n");
            if (body != NULL)
            {
                *body = new Dictionary();
                if (*body != NULL)
                {
                    (*body)->setStringForKey(_tuners[tuner]->name(), kNetTunerKeyTunerName);
                    Dictionary *dict = _props->dictionaryForKey(kPTxD_KEY_TUNERS);
                    if (dict != NULL)
                    {
                        Dictionary *tunerInfo = dict->dictionaryForKey(_tuners[tuner]->name());
                        if (tunerInfo != NULL)
                        {
                            Dictionary *channels = tunerInfo->dictionaryForKey(kPTxD_KEY_CHANNELS);
                            if (channels != NULL)
                            {
                                (*body)->setDictionaryForKey(channels, kNetTunerKeyTunerChannels);
                                delete channels;
                            }
                            int currentChannel = tunerInfo->integerForKey(kPTxD_KEY_CURRENT_CHANNEL);
                            if (currentChannel >= 0)
                            {
                                (*body)->setIntegerForKey(currentChannel, kNetTunerKeyTunerCurrentChannel);
                            }
                            switch (tunerInfo->integerForKey(kPTxD_KEY_TYPE))
                            {
                                case ISDB_S:
                                    (*body)->setStringForKey("ISDB-S", kNetTunerKeyTunerType);
                                    break;
                                case ISDB_T:
                                    (*body)->setStringForKey("ISDB-T", kNetTunerKeyTunerType);
                                    break;
                                default:
                                    (*body)->setStringForKey("Unknown", kNetTunerKeyTunerType);
                                    break;
                            }
                            delete tunerInfo;

                            sc = kPTxD_HTTP_STATUS_OK;
                        }
                        delete dict;
                    }
                }
            }
        }
        else if (strcmp(path, "/" kPTxD_FILE_STATUS) == 0)
        {
        }
        else if (strncmp(path, "/" kPTxD_FILE_CH "=", strlen(kPTxD_FILE_CH) + 2) == 0)
        {
            uint32_t ccc = decimalStringToUINT32(&path[strlen(kPTxD_FILE_CH) + 2]);

            // ccc is valid number?
            if (ccc != UINT32_MAX)
            {
                DebugLog2("valid number: %d\n", ccc);

                //
                sc = kPTxD_HTTP_STATUS_NOT_ACCEPTABLE;

                // tuner is not recording?
                if (_pathsForRecording[tuner] == NULL)
                {
                    DebugLog2("tuner is not recording.\n");

                    // tuner is not locked?
                    if (!_tuners[tuner]->isLocked(client))
                    {
                        DebugLog2("tuner is not locked.\n");

                        // set channel
                        if (_tuners[tuner]->setChannel(ccc))
                        {
                            DebugLog2("setChannel() ok.\n");
                            sc = kPTxD_HTTP_STATUS_OK;
                        }
                        else
                        {
                            DebugLog2("setChannel() ok.\n");
                        }
                    }
                    else
                    {
                        DebugLog3("tuner is locked.\n");
                    }
                }
                else
                {
                    DebugLog3("tuner is recording.\n");
                }
            }
            else
            {
                DebugLog3("invalid number: %d\n", ccc);
            }
        }
        else if (strncmp(path, "/" kPTxD_FILE_STREAMING "=", strlen(kPTxD_FILE_STREAMING) + 2) == 0)
        {
//   /n/streaming=on|off[&sid=xx]&dst=url
        }
        else if (strncmp(path, "/" kPTxD_FILE_RECORDING "=", strlen(kPTxD_FILE_RECORDING) + 2) == 0)
        {
            char *p = &path[strlen(kPTxD_FILE_RECORDING) + 2];
            if (strncmp(p, "on", 2) == 0)
            {
                if (strcmp(p, "on") == 0)
                {
                    if (startRecording(tuner, NULL))
                    {
                        sc = kPTxD_HTTP_STATUS_OK;
                    }
                    else
                    {
                        sc = kPTxD_HTTP_STATUS_NOT_ACCEPTABLE;
                    }
                }
                else if ((strncmp(p, "on&name=", 8) == 0) && (strchr(&p[3], '&') == NULL))
                {
                    printf("&p[8]: %s\n", &p[8]);
                    char *utf8str = strdupWithFromToEncoding(&p[8], "ASCII", "UTF-8");
                    if (utf8str != NULL)
                    {
                        char *name = strdupWithReplacingPercentEscapesUsingEncoding(utf8str, "UTF-8");
                        if (startRecording(tuner, name))
                        {
                            sc = kPTxD_HTTP_STATUS_OK;
                        }
                        else
                        {
                            sc = kPTxD_HTTP_STATUS_NOT_ACCEPTABLE;
                        }
                        if (name != NULL)
                        {
                            free(name);
                        }
                        free(utf8str);
                    }
                }
            }
            else if (strcmp(p, "off") == 0)
            {
                // tuner is recording?
                if (_pathsForRecording[tuner] != NULL)
                {
                    stopRecording(tuner);
                    sc = kPTxD_HTTP_STATUS_OK;
                }
                else
                {
                    sc = kPTxD_HTTP_STATUS_NOT_ACCEPTABLE;
                }
            }
            else
            {
                DebugLog3("parameter error.\n");
            }
        }
        else if (strncmp(path, "/" kPTxD_FILE_RESERVE "=", strlen(kPTxD_FILE_RESERVE) + 2) == 0)
        {
//   /n/reserve=yyyymmddHHMM_HHMM_CCC[&name=xxxx]
        }
        else if (strncmp(path, "/" kPTxD_FILE_LOCK "=", strlen(kPTxD_FILE_LOCK) + 2) == 0)
        {
            char *onoff = &path[strlen(kPTxD_FILE_LOCK) + 2];
            if (strcmp(onoff, "on") == 0)
            {
                // lock
                if (_tuners[tuner]->lock(client))
                {
                    DebugLog2("lock ok.\n");
                    sc = kPTxD_HTTP_STATUS_OK;
                }
                else
                {
                    DebugLog3("lock ng.\n");
                    sc = kPTxD_HTTP_STATUS_NOT_ACCEPTABLE;
                }
            }
            else if (strcmp(onoff, "off") == 0)
            {
                // unlock
                if (_tuners[tuner]->unlock(client))
                {
                    DebugLog2("unlock ok.\n");
                    sc = kPTxD_HTTP_STATUS_OK;
                }
                else
                {
                    DebugLog3("unlock ng.\n");
                    sc = kPTxD_HTTP_STATUS_NOT_ACCEPTABLE;
                }
            }
            else
            {
                DebugLog3("parameter error.\n");
            }
        }
    }
    if (status != NULL)
    {
        *status = sc;
    }
}

static char *strcatWithRealloc(char *str1, const char *str2)
{
    char *result = (char *)realloc(str1, strlen(str1) + strlen(str2) + 1);
    if (result != NULL)
    {
        strcat(result, str2);
    }
    return result;
}

bool Controller::startRecording(uint32_t tuner, char *name)
{
    DebugLog2("Controller::startRecording(tuner=%d, name=%s)\n", tuner, name);

    bool result = false;

    if (tuner < _tuner_count)
    {
        char *pathForRecording = _props->stringForKey(kPTxD_KEY_ROOT_PATH);

        while ((_pathsForRecording[tuner] == NULL) && (pathForRecording != NULL))
        {
            struct stat statOfRootPath;
            if (stat(pathForRecording, &statOfRootPath) != 0)
            {
                DebugLog3("stat error: %s\n", strerror(errno));
                break;
            }

            // root path is directory?
            if ((statOfRootPath.st_mode & S_IFDIR) != S_IFDIR)
            {
                DebugLog3("%s is not directory.\n", pathForRecording);
                break;
            }

            time_t now_secs = time(NULL);
            struct tm *now = localtime(&now_secs);
            if (now == NULL)
            {
                DebugLog3("localtime error.");
                break;
            }

            //
            char year[8];
            if (pathForRecording[strlen(pathForRecording) - 1] == '/')
            {
                sprintf(year, "%04d/", now->tm_year + 1900);
            }
            else
            {
                sprintf(year, "/%04d/", now->tm_year + 1900);
            }
            char *p = strcatWithRealloc(pathForRecording, year);
            if (p == NULL)
            {
                DebugLog3("internal error.\n");
                break;
            }
            pathForRecording = p;

            struct stat statOfYearPath;
            if (stat(pathForRecording, &statOfYearPath) != 0)
            {
                if (errno == ENOENT)
                {
                    printf("mkdir %s\n", pathForRecording);
                    if (mkdir(pathForRecording, 0755) != 0)
                    {
                        DebugLog3("mkdir error: %s\n", strerror(errno));
                        break;
                    }
                    if (chown(pathForRecording, statOfRootPath.st_uid, statOfRootPath.st_gid) != 0)
                    {
                        DebugLog3("chown error: %s\n", strerror(errno));
                    }
                }
                else
                {
                    DebugLog3("stat error: %s\n", strerror(errno));
                    break;
                }
            }
            else if ((statOfYearPath.st_mode & S_IFDIR) != S_IFDIR)
            {
                DebugLog3("%s is not directory.\n", pathForRecording);
                break;
            }

            //
            char month[8];
            sprintf(month, "%02d/", now->tm_mon + 1);
            p = strcatWithRealloc(pathForRecording, month);
            if (p == NULL)
            {
                DebugLog3("internal error.\n");
                break;
            }
            pathForRecording = p;

            struct stat statOfMonthPath;
            if (stat(pathForRecording, &statOfMonthPath) != 0)
            {
                if (errno == ENOENT)
                {
                    printf("mkdir %s\n", pathForRecording);
                    if (mkdir(pathForRecording, 0755) != 0)
                    {
                        DebugLog3("mkdir error: %s\n", strerror(errno));
                        break;
                    }
                    if (chown(pathForRecording, statOfRootPath.st_uid, statOfRootPath.st_gid) != 0)
                    {
                        DebugLog3("chown error: %s\n", strerror(errno));
                    }
                }
                else
                {
                    DebugLog3("stat error: %s\n", strerror(errno));
                    break;
                }
            }
            else if ((statOfMonthPath.st_mode & S_IFDIR) != S_IFDIR)
            {
                DebugLog3("%s is not directory.\n", pathForRecording);
                break;
            }

            //
            char file[256];
            sprintf(file, "%02d%02d_%03d_%02d%02d_",
                    now->tm_mon + 1, now->tm_mday,
                    _tuners[tuner]->channel(), now->tm_hour, now->tm_min);
            
            p = strcatWithRealloc(pathForRecording, file);
            if (p == NULL)
            {
                DebugLog3("internal error.\n");
                break;
            }
            printf("a: %s\n", pathForRecording);
            printf("b: %s\n", p);
            pathForRecording = p;

            p = strcatWithRealloc(pathForRecording, (name != NULL ? name : _tuners[tuner]->name()));
            if (p == NULL)
            {
                DebugLog3("internal error.\n");
                break;
            }
            pathForRecording = p;

            p = strcatWithRealloc(pathForRecording, ".recording");
            if (p == NULL)
            {
                DebugLog3("internal error.\n");
                break;
            }
            pathForRecording = p;

            _fdsForRecording[tuner] = open(pathForRecording, O_CREAT|O_WRONLY, 0644);
            if (_fdsForRecording[tuner] < 0)
            {
                DebugLog3("open error.\n");
                break;
            }

            if (!_tuners[tuner]->setRecording(_fdsForRecording[tuner]))
            {
                DebugLog3("setRecording error.\n");
                close(_fdsForRecording[tuner]);
                _fdsForRecording[tuner] = -1;
                break;
            }

            printf("pathForRecording: %s\n", pathForRecording);

            // success
            _pathsForRecording[tuner] = pathForRecording;
            pathForRecording = NULL;

            result = true;
            break;
        }
        if (pathForRecording != NULL)
        {
            free(pathForRecording);
        }
    }
    return result;
}

void Controller::stopRecording(uint32_t tuner)
{
    DebugLog2("stopRecording(tuner=%d)\n", tuner);
    if (tuner < _tuner_count)
    {
        // stop recording
        _tuners[tuner]->setRecording(-1);

        // close fd for recording
        close(_fdsForRecording[tuner]);
        _fdsForRecording[tuner] = -1;

        // rename
        char *newname = strdup(_pathsForRecording[tuner]);
        if (newname != NULL)
        {
            char *ext = strrchr(newname, '.');
            if (ext != NULL)
            {
                sprintf(ext, ".ts");
                if (rename(_pathsForRecording[tuner], newname) != 0)
                {
                    DebugLog1("error: rename \"%s\" to \"%s\".\n", _pathsForRecording[tuner], newname);
                }
            }
            free(newname);
        }

        // 
        free(_pathsForRecording[tuner]);
        _pathsForRecording[tuner] = NULL;
    }
}

void Controller::tick()
{
    static time_t sec_ = -1;
    time_t sec = time(NULL);
    if (sec == sec_)
    {
        return;
    }
    sec_ = sec;

//    DebugLog3("tick\n");


}

} // PTxD
} // PTx

