/*
 *  Sample.m
 */

#include <termios.h>

#import <Cocoa/Cocoa.h>

#import "PT1.h"
#import "SampleBuffer.h"

static struct termios t_orig;

void begin_getch(void)
{
    struct termios t;

    tcgetattr(0, &t);
    t_orig = t;
    t.c_lflag &= ~(ICANON | ECHO);
    t.c_cc[VMIN] = 1;
    tcsetattr(0, TCSADRAIN, &t);
}

char getch(void)
{
    char c;
    int status = read(0, &c, sizeof(char));
    return status ? c : (char)status;
}
  
void end_getch(void)
{
    tcsetattr(0, TCSADRAIN, &t_orig);
}

int getNumber(int max)
{
    if (max > 9)
    {
        max = 9;
    }

    char c;
    while (YES)
    {
        fflush(stdout);
        c = getch();
        if ((c >= '0') && (c <= ('0' + max)))
        {
            break;
        }
    }
    printf("\n");
    return (c - '0');
}

void setLnbPower(PT1 *tuner)
{
    LnbPower lnbPower;
    if ([tuner getLnbPower:&lnbPower] != kStatusSuccess)
    {
        printf("getLnbPower NG.\n");
        return;
    }
    printf("\n[LNB 電源設定] (現在の設定値:%d)\n", lnbPower);
    printf("0: オフ\n");
    printf("1: +15V\n");
    printf("2: +11V\n");
    printf("3: キャンセル\n");
    printf(">");
    int num = getNumber(3);
    if (num >= 3) return;

    if ([tuner setLnbPower:(LnbPower)num] != kStatusSuccess)
    {
        printf("setLnbPower NG.\n");
    }
}

void GetChannelNameS(uint channel, bool *bs, uint *number)
{
    if ((bs == NULL) || (number == NULL))
    {
        exit(0);
    }

    if (channel < 12)
    {
        *bs = true;
        *number = 1 + 2 * channel;
    }
    else if (channel < 24)
    {
        channel -= 12;

        *bs = false;
        *number = 2 + 2 * channel;
    }
    else
    {
        channel -= 24;

        *bs = false;
        *number = 1 + 2 * channel;
    }
}

void GetChannelNameT(uint channel, bool *catv, uint *number)
{
    if ((catv == NULL) || (number == NULL))
    {
        exit(0);
    }

    static uint TABLE[][3] =
    {
        {   2,  0,   3 },
        {  12,  1,  22 },
        {  21,  0,  12 },
        {  62,  1,  63 },
        { 112,  0,  62 }
    };

    uint i;
    for (i = 0; i < sizeof(TABLE)/sizeof(*TABLE); ++i)
    {
        if (channel <= TABLE[i][0])
        {
            *catv = TABLE[i][1] ? true : false;
            *number = channel + TABLE[i][2] - TABLE[i][0];
            break;
        }
    }
}

void PrintFrequencyOffset(PT1 *pt1, uint tuner, ISDB isdb)
{
    int offset[2];

    [pt1 getFrequencyOffset:tuner isdb:isdb clock:&offset[0] carrier:&offset[1]];

    bool minus[2];

    uint k;
    for (k = 0; k < 2; ++k)
    {
        minus[k] = (offset[k] < 0);
        if (minus[k])
        {
            offset[k] *= -1;
        }
    }

    printf("       ");
    printf("クロック周波数誤差:%s%d.%02dppm", minus[0] ? "-" : "+", offset[0] / 100, offset[0] % 100);
    printf(" / ");

    char *si;
    if (isdb == ISDB_S)
    {
        offset[1] /= 1000;
        si = "k";
    }
    else
    {
        si = "";
    }
    printf("キャリア周波数誤差:%s%d%sHz" , minus[1] ? "-" : "+", offset[1], si);
    printf("\n");
}

int ScanS(PT1 *pt1, uint tuner, uint startChannel, uint endChannel)
{
    int status = kStatusSuccess;

    uint i;
    for (i = startChannel; i <= endChannel; ++i)
    {
        bool bs;
        uint number;
        GetChannelNameS(i, &bs, &number);
        printf("(S%3d) チャンネル:%s%02d", i, bs ? "BS" : "ND", number);
		  
        status = [pt1 setFrequency:tuner isdb:ISDB_S channel:i offset:0];
        if (status != kStatusSuccess)
        {
            printf("Device::SetFrequency() %d", status);
            return status;
        }

        TmccS tmcc;
        memset(&tmcc, 0, sizeof(tmcc));

        uint j;
        for (j = 0; j < 6; ++j)
        {
            status = [pt1 getTmccS:tuner tmcc:&tmcc];
            if (status == kStatusSuccess)
            {
                break;
            }
            usleep((50)*1000);
        }
        bool tmccIsValid = (status == kStatusSuccess);

        uint32_t cn, currentAgc, maxAgc;
        status = [pt1 getCnAgc:tuner isdb:ISDB_S cn100:&cn currentAgc:&currentAgc maxAgc:&maxAgc];
        if (status != kStatusSuccess)
        {
            printf("Device::GetCnAgc() %d", status);
            return status;
        }

        printf(" CN:%02d.%02ddB AGC:%03d/%03d", cn / 100, cn % 100, currentAgc, maxAgc);

        if (tmccIsValid) {
            printf("\n");
        } else {
            printf(" TMCC 受信不可\n");
        }
	
        if (tmccIsValid) {
            uint j;
            for (j = 0; j < 8; ++j)
            {
                uint id = tmcc.Id[j];
                if (id != 0xffff)
                {
                    printf("       ");
                    printf("TS-ID %04x", id);
                    [pt1 setIdS:tuner tsid:id];

                    uint k;
                    for (k = 0; k < 5; ++k)
                    {
                        uint currentId;
                        [pt1 getIdS:tuner tsid:&currentId];
                        if (currentId == id) break;
                        usleep((10)*1000);
                    }
                    LayerS layer;
                    [pt1 getLayerS:tuner layer:&layer];
                    uint i;
                    for (i = 0; i < 2; ++i)
                    {
                        printf(" %s", (i == 0) ? "低階層" : "高階層");
                        printf(" モード:%d スロット数:%02d", layer.Mode[i], layer.Count[i]);
                    }
                    printf("\n");
                }
            }
            PrintFrequencyOffset(pt1, tuner, ISDB_S);
        }
    }

    return status;
}

int ScanT(PT1 *pt1, uint tuner, uint startChannel, uint endChannel)
{
    int status = kStatusSuccess;

    uint i;
    for (i = startChannel; i <= endChannel; ++i)
    {
        bool catv;
        uint number;
        bool tmccIsValid;
        TmccT tmcc;

        GetChannelNameT(i, &catv, &number);
        printf("(T%3d) チャンネル:%s%2d", i, catv ? "C" : " ", number);


/*
        if (startChannel == endChannel)
        {
*/
            status = [pt1 setFrequency:tuner isdb:ISDB_T channel:i offset:0];
            if (status != kStatusSuccess)
            {
                printf("Device::SetFrequency %d", status);
                return status;
            }
            memset(&tmcc, 0, sizeof(tmcc));

            int j;
            for (j = 0; j < 6; ++j)
            {
                status = [pt1 getTmccT:tuner tmcc:&tmcc];
                tmccIsValid = (status == kStatusSuccess);
                if (tmccIsValid) break;
            }
/*
        }
        else
        {
            int offset;
            for (offset = -15; offset < 16; ++offset)
            {
                status = [pt1 setFrequency:tuner isdb:ISDB_T channel:i offset:offset];
                if (status != kStatusSuccess)
                {
                    printf("Device::SetFrequency %d", status);
                    return status;
                }
                memset(&tmcc, 0, sizeof(tmcc));

                status = [pt1 getTmccT:tuner tmcc:&tmcc];
                tmccIsValid = (status == kStatusSuccess);
                if (tmccIsValid)
                {
                    break;
                }
            }
        }
*/

        uint cn, currentAgc, maxAgc;
        status = [pt1 getCnAgc:tuner isdb:ISDB_T cn100:&cn currentAgc:&currentAgc maxAgc:&maxAgc];
        if (status != kStatusSuccess)
        {
            printf("Device::GetCnAgc() %d", status);
            return status;
        }

        BOOL locked[3];
        status = [pt1 getLockedT:tuner locked:locked];
        if (status != kStatusSuccess)
        {
            printf("Device::GetLockedT() %d", status);
            return status;
        }

        printf(" CN:%02d.%02ddB AGC:%03d/%03d ロック:%d%d%d",
            cn / 100, cn % 100,
            currentAgc, maxAgc,
            locked[0], locked[1], locked[2]);

        if (tmccIsValid)
        {
            printf(" システム識別:%d", tmcc.System);
            printf("\n");
            uint j;
            for (j = 0; j < 3; ++j)
            {
                printf("      ");
                printf(" %c階層", L'A'+j);

                printf(" 変調:");
                uint mode = tmcc.Mode[j];
                if (mode != 7)
                {
                    printf("%d", mode);
                }
                else
                {
                    printf("-");
                }

                printf(" 符号化率:");
                uint rate = tmcc.Rate[j];
                if (mode != 7)
                {
                    printf("%d", rate);
                }
                else
                {
                    printf("-");
                }

                printf(" インターリーブ長:");
                uint interleave = tmcc.Interleave[j];
                if (mode != 7)
                {
                    printf("%d", interleave);
                }
                else
                {
                    printf("-");
                }

                printf(" セグメント数:");
                uint segment = tmcc.Segment[j];
                if (segment != 15)
                {
                    printf("%02d", segment);
                }
                else
                {
                    printf("--");
                }

                printf("\n");
            }
            PrintFrequencyOffset(pt1, tuner, ISDB_T);
        }
        else
        {
            printf(" TMCC 受信不可\n");
        }
    }

    return status;
}

void scanChannel(PT1 *tuner)
{
    printf("0: チューナー番号0\n");
    printf("1: チューナー番号1\n");
    printf("2: キャンセル\n");
    printf(">");
    int num1 = getNumber(2);
    if (num1 >= 2) return;

    printf("0: ISDB-S\n");
    printf("1: ISDB-T\n");
    printf("2: キャンセル\n");
    printf(">");
    int num2 = getNumber(2);
    if (num2 >= 2) return;

    switch (num2)
    {
    case 0:
        ScanS(tuner, num1, 0,  23);
        break;

    case 1:
        ScanT(tuner, num1, 0, 112);
        break;
    }
}

void setChannel(PT1 *tuner)
{
    printf("0: チューナー番号0\n");
    printf("1: チューナー番号1\n");
    printf("2: キャンセル\n");
    printf(">");
    int num1 = getNumber(2);
    if (num1 == 2) return;

    printf("0: ISDB-S\n");
    printf("1: ISDB-T\n");
    printf("2: キャンセル\n");
    printf(">");
    int num2 = getNumber(2);
    if (num2 == 2) return;

    uint max = (num2 == ISDB_S) ? 35 : 112;

    printf("チャンネル番号を入力してください。(範囲:0～%d)\n", max);
    printf(">");

    uint channel;
    scanf("%d", &channel);
    if (max < channel) {
        channel = max;
    }

    switch (num2)
    {
    case 0:
        ScanS(tuner, num1, channel, channel);
        break;
    case 1:
        ScanT(tuner, num1, channel, channel);
        break;
    }
}

void capture(PT1 *tuner, bool transferEnable)
{
    static SampleBuffer *_buffer = NULL;
    if (_buffer == NULL)
    {
        _buffer = [[SampleBuffer alloc] initWithPT1:tuner];
    }

    [_buffer transfer:transferEnable];
}

void device_menu(PT1 *tuner)
{
    if ([tuner open] != kStatusSuccess)
    {
        return;
    }

    if ([tuner setTunerPowerReset:kTunerPowerOnResetEnable] != kStatusSuccess)
    {
        printf("Error: setTunerPowerReset:TUNER_POWER_ON_RESET_ENABLE\n");
        return;
    }
    usleep(20 * 1000);

    if ([tuner setTunerPowerReset:kTunerPowerOnResetDisable] != kStatusSuccess)
    {
        printf("Error: setTunerPowerReset:TUNER_POWER_ON_RESET_DISABLE\n");
        return;
    }
    usleep(1 * 1000);

    uint i;
    for (i = 0; i < 2; ++i)
    {
        if ([tuner initTuner:i] != kStatusSuccess)
        {
            printf("Error: initTuner:%d\n", i);
            return;
        }
    }

    for (i = 0; i < 2; ++i)
    {
        uint j;
        for (j = 0; j < 2; ++j)
        {
            if ([tuner setTunerSleep:i isdb:(ISDB)j sleep:false] != kStatusSuccess)
            {
                printf("Error: setTunerSleep:%d isdb:%d sleep:%d\n", i, j, false);
                return;
			}
        }
    }
    
    for (i = 0; i < 2; ++i)
    {
        uint j;
        for (j = 0; j < 2; ++j)
        {
            if ([tuner setStreamEnable:i isdb:(ISDB)j enable:true] != kStatusSuccess)
            {
                printf("Error: setStreamEnable:%d isdb:%d enable:%d\n", i, j, true);
                return;
            }
        }
    }

    // SetStreamGray() で設定した値は TS パケットの 189 バイト目の下位 3 ビットに反映されます。
    // このサンプルでは積極的に使っていません。
    for (i = 0; i < 2; ++i)
    {
        uint j;
        for (j = 0; j < 2; ++j)
        {
            if ([tuner setStreamGray:i isdb:(ISDB)j gray:(4 + i * 2 + j)] != kStatusSuccess)
            {
                printf("Error: setStreamGray:%d isdb:%d gray:%d\n", i, j, 4 + i * 2 + j);
                return;
            }
        }
    }

    bool done = false;
    while (!done)
    {
        BOOL transferEnable;
        [tuner getTransferEnable:&transferEnable];

        printf("\n[デバイスメニュー (1/3ページ)]\n");
        printf("0: LNB 電源設定\n");
        printf("1: チャンネルスキャン\n");
        printf("2: チャンネル設定\n");
        printf("3: ISDB-S TS-ID設定\n");
        printf("4: エラーレート・エラーカウンタ取得\n");
        printf("5: キャプチャ%s\n", (transferEnable ? "停止" : "開始"));
        printf("6: → 2ページ目\n");
        printf("7: → 3ページ目（工場出荷時検査項目）\n");
        printf("8: デバイスを閉じる\n");
        printf(">");
        fflush(stdout);

        int num = getNumber(8);
        switch (num)
        {
        case 0:
            setLnbPower(tuner);
            break;

        case 1:
            scanChannel(tuner);
            break;

        case 2:
            setChannel(tuner);
            break;

        case 5:
            capture(tuner, !transferEnable);
            break;

        case 8:
            done = true;
            break;
        }
    }
    [tuner close];
}

#if 0
    LnbPower lnbPower;
    BOOL enable;

    uint32_t counter;
    if ([tuner getPciClockCounter:&counter] != kStatusSuccess)
    {
        NSLog(@"getPciClockCounter NG.");
    }
    else
    {
        NSLog(@"getPciClockCounter OK. counter = 0x%08x", counter);
    }

    uint8_t latencyTimer;
    if ([tuner getPciLatencyTimer:&latencyTimer] != kStatusSuccess)
    {
        NSLog(@"getPciLatencyTimer NG.");
    }
    else
    {
        NSLog(@"getPciLatencyTimer OK. latencyTimer = 0x%02X", latencyTimer);
    }

    if ([tuner setPciLatencyTimer:0x88] != kStatusSuccess)
    {
        NSLog(@"setPciLatencyTimer NG.");
    }
    else
    {
        NSLog(@"setPciLatencyTimer OK.");
    }

    if ([tuner getLnbPower:&lnbPower] != kStatusSuccess)
    {
        NSLog(@"getLnbPower NG.");
    }
    else
    {
        NSLog(@"getLnbPower OK. lnbPower = 0x%08X", lnbPower);
    }

    if ([tuner setLnbPower:(LnbPower)0x11] != kStatusSuccess)
    {
        NSLog(@"setLnbPower NG.");
    }
    else
    {
        NSLog(@"setLnbPower OK.");
    }

    if ([tuner getLnbPowerWhenClose:&lnbPower] != kStatusSuccess)
    {
        NSLog(@"getLnbPowerWhenClose NG.");
    }
    else
    {
        NSLog(@"getLnbPowerWhenClose OK. lnbPower = 0x%08X", lnbPower);
    }

    if ([tuner setLnbPowerWhenClose:(LnbPower)0x22] != kStatusSuccess)
    {
        NSLog(@"setLnbPowerWhenClose NG.");
    }
    else
    {
        NSLog(@"setLnbPowerWhenClose OK.");
    }

    TunerPowerReset tunerPowerReset;
    if ([tuner getTunerPowerReset:&tunerPowerReset] != kStatusSuccess)
    {
        NSLog(@"getTunerPowerReset NG.");
    }
    else
    {
        NSLog(@"getTunerPowerReset OK. tunerPowerReset = 0x%08X", tunerPowerReset);
    }

    if ([tuner setTunerPowerReset:(TunerPowerReset)0x11] != kStatusSuccess)
    {
        NSLog(@"setTunerPowerReset NG.");
    }
    else
    {
        NSLog(@"setTunerPowerReset OK.");
    }

    if ([tuner initTuner:0x12345678] != kStatusSuccess)
    {
        NSLog(@"initTuner NG.");
    }
    else
    {
        NSLog(@"initTuner OK.");
    }

    BOOL sleep;
    if ([tuner getTunerSleep:0x12345678 isdb:(ISDB)0x33 sleep:&sleep] != kStatusSuccess)
    {
        NSLog(@"getTunerSleep NG.");
    }
    else
    {
        NSLog(@"getTunerSleep OK. sleep = 0x%08x", sleep);
    }

    if ([tuner setTunerSleep:0x12345678 isdb:(ISDB)0x22 sleep:NO] != kStatusSuccess)
    {
        NSLog(@"setTunerSleep NG.");
    }
    else
    {
        NSLog(@"setTunerSleep OK.");
    }

    uint32_t channel;
    int32_t offset;
    if ([tuner getFrequency:0x12345678 isdb:(ISDB)0x55 channel:&channel offset:&offset] != kStatusSuccess)
    {
        NSLog(@"getFrequency NG.");
    }
    else
    {
        NSLog(@"getFrequency OK. channel = 0x%08x, offset = %d", channel, offset);
    }

    if ([tuner setFrequency:0x12345678 isdb:(ISDB)0x44 channel:0x23456789 offset:0x3456789a] != kStatusSuccess)
    {
        NSLog(@"setFrequency NG.");
    }
    else
    {
        NSLog(@"setFrequency OK.");
    }

    int32_t clock;
    int32_t carrier;
    if ([tuner getFrequencyOffset:0x12345678 isdb:(ISDB)0x66 clock:&clock carrier:&carrier] != kStatusSuccess)
    {
        NSLog(@"getFrequencyOffset NG.");
    }
    else
    {
        NSLog(@"getFrequencyOffset OK. clock = %d, carrier = %d", clock, carrier);
    }

    uint32_t cn100;
    uint32_t currentAgc;
    uint32_t maxAgc;
    if ([tuner getCnAgc:0x12345678 isdb:(ISDB)0x77 cn100:&cn100 currentAgc:&currentAgc maxAgc:&maxAgc] != kStatusSuccess)
    {
        NSLog(@"getCnAgc NG.");
    }
    else
    {
        NSLog(@"getCnAgc OK. cn100 = 0x%08x, currnetAgc = 0x%08x, maxAgc = 0x%08x", cn100, currentAgc, maxAgc);
    }

    uint32_t tsid;
    if ([tuner getIdS:0x12345678 tsid:&tsid] != kStatusSuccess)
    {
        NSLog(@"getIdS NG.");
    }
    else
    {
        NSLog(@"getIdS OK. tsid = 0x%08x", tsid);
    }

    if ([tuner setIdS:0x12345678 tsid:0x23456789] != kStatusSuccess)
    {
        NSLog(@"setIdS NG.");
    }
    else
    {
        NSLog(@"setIdS OK.");
    }

    ErrorRate errorRate;
    if ([tuner getCorrectedErrorRate:0x12345678 isdb:(ISDB)0x11 layerIndex:(LayerIndex)0x22 errorRate:&errorRate] != kStatusSuccess)
    {
        NSLog(@"getCorrectedErrorRate NG.");
    }
    else
    {
        NSLog(@"getCorrectedErrorRate OK. errorRate->Denominator = 0x%08x, errorRate->Numerator = 0x%08x",
                errorRate.Denominator, errorRate.Numerator);
    }

    if ([tuner resetCorrectedErrorCount:0x12344567 isdb:(ISDB)0x22] != kStatusSuccess)
    {
        NSLog(@"resetCorrectedErrorCount NG.");
    }
    else
    {
        NSLog(@"resetCorrectedErrorCount OK.");
    }

    uint32_t count;
    if ([tuner getErrorCount:0x12345678 isdb:(ISDB)0x33 count:&count] != kStatusSuccess)
    {
        NSLog(@"getErrorCount NG.");
    }
    else
    {
        NSLog(@"getErrorCount OK. count = 0x%08x", count);
    }

    TmccS tmccS;
    if ([tuner getTmccS:0x12345678 tmcc:&tmccS] != kStatusSuccess)
    {
        NSLog(@"getTmccS NG.");
    }
    else
    {
        NSLog(@"getTmccS OK.");

        NSLog(@"tmcc->Indicator = 0x%08x", tmccS.Indicator);
        int i;
        for (i = 0; i < 4; ++i)
        {
            NSLog(@"tmcc->Mode[%d] = 0x%08x", i, tmccS.Mode[i]);
            NSLog(@"tmcc->Slot[%d] = 0x%08x", i, tmccS.Slot[i]);
        }
        for (i = 0; i < 8; ++i)
        {
            NSLog(@"tmcc->Id[%d] = 0x%08x", i, tmccS.Id[i]);
        }
        NSLog(@"tmcc->Emergency = 0x%08x", tmccS.Emergency);
        NSLog(@"tmcc->UpLink = 0x%08x", tmccS.UpLink);
        NSLog(@"tmcc->ExtFlag = 0x%08x", tmccS.ExtFlag);
        NSLog(@"tmcc->ExtData[0] = 0x%08x", tmccS.ExtData[0]);
        NSLog(@"tmcc->ExtData[1] = 0x%08x", tmccS.ExtData[1]);
    }

    LayerS layer;
    if ([tuner getLayerS:0x12345678 layer:&layer] != kStatusSuccess)
    {
        NSLog(@"getLayerS NG.");
    }
    else
    {
        NSLog(@"getLayerS OK.");
        int i;
        for (i = 0; i < LAYER_COUNT_S; ++i)
        {
            NSLog(@"layer->Mode[%d] = 0x%08x", i, layer.Mode[i]);
            NSLog(@"layer->Count[%d] = 0x%08x", i, layer.Count[i]);
        }
    }

    TmccT tmccT;
    if ([tuner getTmccT:0x12345678 tmcc:&tmccT] != kStatusSuccess)
    {
        NSLog(@"getTmccT NG.");
    }
    else
    {
        NSLog(@"getTmccT OK.");

        NSLog(@"tmcc->System = 0x%08x", tmccT.System);
        NSLog(@"tmcc->Indicator = 0x%08x", tmccT.Indicator);
        NSLog(@"tmcc->Emergency = 0x%08x", tmccT.Emergency);
        NSLog(@"tmcc->Partial = 0x%08x", tmccT.Partial);
        int i;
        for (i = 0; i < LAYER_COUNT_T; ++i)
        {
            NSLog(@"tmcc->Mode[%d] = 0x%08x", i, tmccT.Mode[i]);
            NSLog(@"tmcc->Rate[%d] = 0x%08x", i, tmccT.Rate[i]);
            NSLog(@"tmcc->Interleave[%d] = 0x%08x", i, tmccT.Interleave[i]);
            NSLog(@"tmcc->Segment[%d] = 0x%08x", i, tmccT.Segment[i]);
        }
        NSLog(@"tmcc->Phase = 0x%08x", tmccT.Phase);
        NSLog(@"tmcc->Reserved = 0x%08x", tmccT.Reserved);
    }

    BOOL locked[LAYER_COUNT_T];
    if ([tuner getLockedT:0x12345678 locked:locked] != kStatusSuccess)
    {
        NSLog(@"getLockedT NG.");
    }
    else
    {
        NSLog(@"getLockedT OK.");
        int i;
        for (i = 0; i < LAYER_COUNT_T; ++i)
        {
            NSLog(@"locked[%d] = 0x%08x", i, locked[i]);
        }
    }

    LayerMask layerMask;
    if ([tuner getLayerEnable:0x12345678 isdb:(ISDB)0x22 layerMask:&layerMask] != kStatusSuccess)
    {
        NSLog(@"getLayerEnable NG.");
    }
    else
    {
        NSLog(@"getLayerEnable OK. layerMask = 0x%08x", layerMask);
    }

    if ([tuner setLayerEnable:0x12345678 isdb:(ISDB)0x11 layerMask:(LayerMask)0x22] != kStatusSuccess)
    {
        NSLog(@"setLayerEnable NG.");
    }
    else
    {
        NSLog(@"setLayerEnable OK.");
    }

#if 0
- (int)setBufferInfo:(pconst BufferInfo *)bufferInfo;
- (int)getBufferInfo:(BufferInfo *)bufferInfo;
- (int)getBufferPtr:(uint32_t)index ptr:(void **)ptr;
#endif

    if ([tuner resetTransferCounter] != kStatusSuccess)
    {
        NSLog(@"resetTransferCounter NG.");
    }
    else
    {
        NSLog(@"resetTransferCounter OK.");
    }

    if ([tuner incrementTransferCounter] != kStatusSuccess)
    {
        NSLog(@"incrementTransferCounter NG.");
    }
    else
    {
        NSLog(@"incrementTransferCounter OK.");
    }

    if ([tuner getStreamEnable:0x12345678 isdb:(ISDB)0x44 enabled:&enable] != kStatusSuccess)
    {
        NSLog(@"getStreamEnable NG.");
    }
    else
    {
        NSLog(@"getStreamEnable OK. enable = 0x%08x", enable);
    }

    if ([tuner setStreamEnable:0x12345678 isdb:(ISDB)0x33 enable:YES] != kStatusSuccess)
    {
        NSLog(@"setStreamEnable NG.");
    }
    else
    {
        NSLog(@"setStreamEnable OK.");
    }

    uint32_t gray;
    if ([tuner getStreamGray:0x12345678 isdb:(ISDB)0x66 gray:&gray] != kStatusSuccess)
    {
        NSLog(@"getStreamGray NG.");
    }
    else
    {
        NSLog(@"getStreamGray OK. gray = 0x%08x", gray);
    }

    if ([tuner setStreamGray:0x12345678 isdb:(ISDB)0x55 gray:0x23456789] != kStatusSuccess)
    {
        NSLog(@"setStreamGray NG.");
    }
    else
    {
        NSLog(@"setStreamGray OK.");
    }

    if ([tuner getTransferEnable:&enable] != kStatusSuccess)
    {
        NSLog(@"getTransferEnable NG.");
    }
    else
    {
        NSLog(@"getTransferEnable OK. enable = 0x%08x", enable);
    }

    if ([tuner setTransferEnable:NO] != kStatusSuccess)
    {
        NSLog(@"setTransferEnable NG.");
    }
    else
    {
        NSLog(@"setTransferEnable OK.");
    }

    TransferInfo transferInfo;
    if ([tuner getTransferInfo:&transferInfo] != kStatusSuccess)
    {
        NSLog(@"getTransferInfo NG.");
    }
    else
    {
        NSLog(@"getTransferInfo OK.");

        NSLog(@"transferInfo->BufferOverflow = 0x%08x", transferInfo.BufferOverflow);
        NSLog(@"transferInfo->TransferCounter0 = 0x%08x", transferInfo.TransferCounter0);
        NSLog(@"transferInfo->TransferCounter1 = 0x%08x", transferInfo.TransferCounter1);
    }
    exit(0);
#endif

/*
 *
 */
int main(int argc, char *argv[])
{
    NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];

    begin_getch();

    NSArray *tuners = [PT1 scan];
    if (tuners)
    {
        bool done = false;
        while (!done)
        {
            printf("\n[デバイス選択メニュー]\n");
            printf("   Bus Dev Fun Cnt\n");

            int i;
            for (i = 0; i < [tuners count]; ++i)
            {
                // デバイス情報取得
                DeviceInfo info;
                PT1 *tuner = [tuners objectAtIndex:i];
                if ([tuner getDeviceInfo:&info])
                {
                    printf("%d: %3d %3d %3d %3d\n", i, info.Bus, info.Slot, info.Function, info.BadBitCount);
                }
                else
                {
                    printf("getDeviceInfo error.\n");
                }
            }
            
            printf("q: 終了\n");
            printf(">");
            fflush(stdout);

            char c = getch();
            switch (c)
            {
            default:
                if ((c >= '0') && (c <= '9'))
                {
                    if ((c - '0') < [tuners count])
                    {
                        device_menu([tuners objectAtIndex:(c - '0')]);
                    }
                }
                break;
            case 'q':
                done = true;
                break;
            }
        }
    }
    else
    {
		NSLog(@"No matching drivers found.");
    }

    end_getch();

    [pool release];

    return 0;
}
