// QPin.h
// 2009/05/29

#pragma once

// QPinImpl
template<class T>
class ATL_NO_VTABLE QPinImpl : public IPin {

	QPinHost* m_pHost;

	IBaseFilter* m_pFilter; /* weak */

	ATL::CStringW m_Id;

	INT32       m_cMediaTypes;
	QMediaType* m_MediaTypes;

	ATL::CComPtr<IPin> m_ConnectedTo;

	ATL::CComPtr<IMemInputPin>  m_MemInputPin;
	ATL::CComPtr<IMemAllocator> m_MemAllocator;

	QMediaType m_ConnectionMediaType;

public:

	QPinImpl() : m_pFilter(0), m_cMediaTypes(0), m_MediaTypes(0)
	{
	}

	~QPinImpl()
	{
		delete[] m_MediaTypes;
	}

	void Init(
		QPinHost*    host,
		IBaseFilter* p,
		LPCWSTR      id)
	{
		m_pHost   = host;
		m_pFilter = p;
		m_Id      = id;
	}

	/* */

	QMediaType* MediaTypes()
	{
		return m_MediaTypes;
	}

	void SetupMediaTypes(INT32 c)
	{
		m_cMediaTypes = c;
		m_MediaTypes  = new QMediaType[c];
	}

	/* */

	IMemInputPin* GetMemInputPin()
	{
		return m_MemInputPin;
	}

	IMemAllocator* GetMemAllocator()
	{
		return m_MemAllocator;
	}

	/* IPin */

	STDMETHOD(Connect)(
		IPin*                pReceivePin,
		const AM_MEDIA_TYPE* pmt)
	{
		ATLTRACE("QPinImpl::Connect()\n");

		ATL::CComCritSecLock<ATL::CComCriticalSection> lock(m_pHost->FilterCSec());

		if (pReceivePin == 0) {
			return E_INVALIDARG;
		}

		if (m_ConnectedTo != 0) {
			return VFW_E_ALREADY_CONNECTED;
		}

		if (m_pHost->IsRunning()) {
			return VFW_E_NOT_STOPPED;
		}

		const AM_MEDIA_TYPE* target = 0;

		for (INT32 i = 0; i < m_cMediaTypes; i++) {
			const AM_MEDIA_TYPE* mt = m_MediaTypes + i;

			if (pmt != 0) {
				if (mt->majortype != pmt->majortype ||
					mt->subtype   != pmt->subtype) {
					continue;
				}

				if (pmt->formattype != GUID_NULL) {
					if (mt->formattype != pmt->formattype) {
						continue;
					}

					if (mt->cbFormat != pmt->cbFormat) {
						continue;
					}

					if (memcmp(mt->pbFormat, pmt->pbFormat, mt->cbFormat) != 0) {
						continue;
					}
				}
			}

			HRESULT hRslt = pReceivePin->QueryAccept(mt);
			if (hRslt == S_OK) {
				target = mt;
				break;
			}
		}

		if (target == 0) {
			return VFW_E_TYPE_NOT_ACCEPTED;
		}

		HRESULT hRslt = pReceivePin->ReceiveConnection(
			this,
			target);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		m_ConnectionMediaType.SafeRelease();
		hRslt = pReceivePin->ConnectionMediaType(&m_ConnectionMediaType);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		ATL::CComQIPtr<IMemInputPin> mem = pReceivePin;
		if (mem == 0) {
			return VFW_E_TYPE_NOT_ACCEPTED;
		}

		ATL::CComPtr<IMemAllocator> alloc;
		hRslt = mem->GetAllocator(&alloc);
		if (FAILED(hRslt)) {
			return VFW_E_TYPE_NOT_ACCEPTED;
		}

		T* pThis = static_cast<T*>(this);
		hRslt = pThis->SetupAllocator(
			alloc,
			target);
		if (FAILED(hRslt)) {
			return VFW_E_TYPE_NOT_ACCEPTED;
		}

		hRslt = mem->NotifyAllocator(alloc, FALSE);
		if (FAILED(hRslt)) {
			return VFW_E_TYPE_NOT_ACCEPTED;
		}

		m_MemInputPin  = mem;
		m_MemAllocator = alloc;

		m_ConnectedTo = pReceivePin;

		return S_OK;
	}

	STDMETHOD(ReceiveConnection)(
		IPin*                /* pConnector */,
		const AM_MEDIA_TYPE* /* pmt */)
	{
		ATLTRACE("QPinImpl::ReceiveConnection()\n");

		return E_NOTIMPL;
	}

	STDMETHOD(Disconnect)(void)
	{
		ATLTRACE("QPinImpl::Disconnect()\n");

		ATL::CComCritSecLock<ATL::CComCriticalSection> lock(m_pHost->FilterCSec());

		if (m_pHost->IsRunning()) {
			return VFW_E_NOT_STOPPED;
		}

		if (m_ConnectedTo == 0) {
			return S_FALSE;
		}

		m_MemAllocator.Release();
		m_MemInputPin. Release();

		m_ConnectedTo.Release();

		return S_OK;
	}

	STDMETHOD(ConnectedTo)(
		IPin** pPin)
	{
		ATLTRACE("QPinImpl::ConnectedTo()\n");

		ATL::CComCritSecLock<ATL::CComCriticalSection> lock(m_pHost->FilterCSec());

		if (m_ConnectedTo == 0) {
			if (pPin != 0) {
				*pPin = 0;
			}

			return VFW_E_NOT_CONNECTED;
		}

		if (pPin == 0) {
			return E_INVALIDARG;
		}

		*pPin = m_ConnectedTo;

		if (*pPin != 0) {
			(*pPin)->AddRef();
		}

		return S_OK;
	}

	STDMETHOD(ConnectionMediaType)(
		AM_MEDIA_TYPE* pmt)
	{
		ATLTRACE("QPinImpl::ConnectionMediaType()\n");

		ATL::CComCritSecLock<ATL::CComCriticalSection> lock(m_pHost->FilterCSec());

		if (pmt == 0) {
			return E_INVALIDARG;
		}

		if (!QMediaType::Copy(pmt, &m_ConnectionMediaType)) {
			return E_OUTOFMEMORY;
		}

		return S_OK;
	}

	/* */

	STDMETHOD(QueryPinInfo)(
		PIN_INFO* pInfo)
	{
		ATLTRACE("QPinImpl::QueryPinInfo()\n");

		if (pInfo == 0) {
			return E_INVALIDARG;
		}

		pInfo->pFilter = m_pFilter;
		pInfo->dir     = PINDIR_OUTPUT;

		SIZE_T len = m_Id.GetLength();
		if (len > 127) {
			len = 127;
		}
		wcsncpy_s(pInfo->achName, 128, m_Id, len);

		if (m_pFilter != 0) {
			m_pFilter->AddRef();
		}

		return S_OK;
	}

	STDMETHOD(QueryDirection)(
		PIN_DIRECTION* pPinDir)
	{
		ATLTRACE("QPinImpl::QueryDirection()\n");

		if (pPinDir == 0) {
			return E_INVALIDARG;
		}

		*pPinDir = PINDIR_OUTPUT;

		return S_OK;
	}

	STDMETHOD(QueryId)(
		LPWSTR* Id)
	{
		ATLTRACE("QPinImpl::QueryId()\n");

		if (Id == 0) {
			return E_INVALIDARG;
		}

		LPCWSTR id = m_Id;
		SIZE_T  cb = (m_Id.GetLength() + 1) * sizeof(WCHAR);

		*Id = static_cast<LPWSTR>(CoTaskMemAlloc(cb));
		if (*Id == 0) {
			return E_OUTOFMEMORY;
		}

		memcpy(*Id, id, cb);

		return S_OK;
	}

	/* */

	STDMETHOD(QueryAccept)(
		const AM_MEDIA_TYPE* pmt)
	{
		ATLTRACE("QPinImpl::QueryAccept()\n");

		ATL::CComCritSecLock<ATL::CComCriticalSection> lock(m_pHost->FilterCSec());

		if (pmt == 0) {
			return E_INVALIDARG;
		}

		T* pThis = static_cast<T*>(this);
		HRESULT hRslt = pThis->CheckAccept(pmt);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		m_ConnectionMediaType.Assign(pmt);

		return S_OK;
	}

	STDMETHOD(EnumMediaTypes)(
		IEnumMediaTypes** ppEnum)
	{
		ATLTRACE("QPinImpl::EnumMediaTypes()\n");

		if (ppEnum == 0) {
			return E_INVALIDARG;
		}

		HRESULT hRslt = QEnumMediaTypes::Create(m_MediaTypes, m_MediaTypes + m_cMediaTypes, this, ppEnum);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		return S_OK;
	}

	/* */

	STDMETHOD(QueryInternalConnections)(
		IPin** apPin,
		ULONG* nPin)
	{
		ATLTRACE("QPinImpl::QueryInternalConnections()\n");

		if (apPin != 0 && nPin != 0) {
			if (*nPin > 0) {
				memset(apPin, 0, *nPin * sizeof(IPin*));
			}
			*nPin = 0;
		}

		return E_NOTIMPL;
	}

	/* */

	STDMETHOD(EndOfStream)(void)
	{
		ATLTRACE("QPinImpl::EndOfStream()\n");

		return E_NOTIMPL;
	}

	STDMETHOD(BeginFlush)(void)
	{
		ATLTRACE("QPinImpl::BeginFlush()\n");

		return E_NOTIMPL;
	}

	STDMETHOD(EndFlush)(void)
	{
		ATLTRACE("QPinImpl::EndFlush()\n");

		return E_NOTIMPL;
	}

	STDMETHOD(NewSegment)( 
		REFERENCE_TIME /* tStart */,
		REFERENCE_TIME /* tStop  */,
		double         /* dRate  */)
	{
		ATLTRACE("QPinImpl::NewSegment()\n");

		return E_NOTIMPL;
	}

	/* */

}; // QPinImpl

