/* FrameDecoder.c */
/* 2009/06/29     */

#include "StdAfx.h"

#include "FrameDecoder.h"

#include "FrameReconstructor.h"

/* */

#pragma warning(disable : 4799)

/* */

static BOOL Dequantize_MakeMatrix(
	DequantizeMatrix_t*      m,
	const DequantizeTable_t* t,
	INT32                    index)
{
	INT32 i, j, p, q;

	for (i = 0; i < 2; i++) {
		for (p = 0; p < 3; p++) {
			INT32 ss, qs;
			INT32 si, sj;

			const UINT8* mi;
			const UINT8* mj;

			INT16* mat;

			INT32 s = 0;
			for (q = 0; q < t->Count[i][p]; q++) {
				s += t->Size[i][p][q];
				if (s >= index) {
					break;
				}
			}

			ss = t->Size[i][p][q];
			qs = s - ss;
			si = s  - index;
			sj = qs - index;

			mi = t->Matrix[t->Base[i][p][q    ]];
			mj = t->Matrix[t->Base[i][p][q + 1]];

			mat = m->Matrix[i][p];

			for (j = 0; j < 64; j++) {
				INT32 coeff = (2 * si * mi[j] - 2 * sj * mj[j] + ss) / (2 * ss);

				INT32 qmin = 8 << ((j == 0) ? i + 1 : i);
				INT32 qscl = (j == 0) ? t->DCScale[index] : t->ACScale[index];

				INT32 v = ((qscl * coeff) / 100) * 4;

				mat[j] = (INT16)((v < qmin) ? qmin : ((v > 4096) ? 4096 : v));
			}
		}
	}

	return TRUE;
}

static void Filter_Setup(
	LoopFilter_t*        t,
	const FilterTable_t* l,
	INT32                q)
{
	INT32 x;
	INT32 lim = l->Limit[q];

	INT16* d = t->Delta;

	memset(t->Delta, 0, sizeof(t->Delta));

	t->Limit = lim;

	for (x = 0; x < lim; x++) {
		INT32 i = lim + x;

		if (127 - i >= 0) {
			d[127 - i] = x - lim;
		}

		d[127 - x] = -x;
		d[127 + x] =  x;

		if (127 + i < 256) {
			d[127 + i] = lim - x;
		}
	}
}

/* */

static void RunLength_Start(
	RunLength_t* t,
	INT32        x)
{
	t->Bit = !x; /* invert later */
	t->Run = 0;
}

static const UINT8 HLONG[64] = {
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,
	0x34,0x34,0x34,0x34,0x45,0x45,0x56,0x66
};

static const INT8 HLONG_BASE[7] = { 1, 2, 4, 6, 10, 18, 34 };
static const INT8 HLONG_BITS[7] = { 0, 1, 1, 2,  3,  4, 12 };

static const UINT8 HSHORT[32] = {
	0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,
	0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,
	0x23,0x23,0x23,0x23,
	0x34,0x34,0x45,0x55
};

static const INT8 HSHORT_BASE[6] = { 1, 3, 5, 7, 11, 15 };
static const INT8 HSHORT_BITS[6] = { 1, 1, 1, 2,  2,  4 };

#define RL_LONG_FULL_RUN 4129

#define RL_LONG_DECODE_RUN(X) \
	{ INT32 token; LOAD_BITS token = HLONG[GET_BITS_I(6)]; RETIRE_BITS(token & 0xf); (X) = HLONG_BASE[token >> 4]; \
	if (HLONG_BITS[token >> 4] > 0) { INT32 x; FETCH_BITS(x, HLONG_BITS[token >> 4]) (X) += x; } }

#define RL_LONG_DECODE \
	if (r.Run == 0) { RL_LONG_DECODE_RUN(r.Run) r.Bit = !(r.Bit); } \
	(r.Run)--;

#define RL_SHORT_DECODE \
	if (r.Run == 0) { INT32 token, x; LOAD_BITS token = HSHORT[GET_BITS_I(5)]; RETIRE_BITS(token & 0xf); r.Run = HSHORT_BASE[token >> 4]; \
		FETCH_BITS(x, HSHORT_BITS[token >> 4]) r.Run += x; r.Bit = !(r.Bit); }                                                            \
	(r.Run)--;

/* */

static const UINT8 M_MODE[8][8] = {
	{ 0, 0, 0, 0, 0, 0, 0, 0 },
	{ 3, 4, 2, 0, 1, 5, 6, 7 },
	{ 3, 4, 0, 2, 1, 5, 6, 7 },
	{ 3, 2, 4, 0, 1, 5, 6, 7 },
	{ 3, 2, 0, 4, 1, 5, 6, 7 },
	{ 0, 3, 4, 2, 1, 5, 6, 7 },
	{ 0, 5, 3, 4, 2, 1, 6, 7 },
	{ 0, 1, 2, 3, 4, 5, 6, 7 }
};

static const UINT8 H_MODE[128] = {
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,
	0x34,0x34,0x34,0x34,0x34,0x34,0x34,0x34,0x45,0x45,0x45,0x45,0x56,0x56,0x67,0x77
};

/* */

static const INT8 MV_VAL[0x100] = {
	   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
	   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
	   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
	   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
	  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
	  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
	   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,
	  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,
	   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,
	  -3,  -3,  -3,  -3,  -3,  -3,  -3,  -3,  -3,  -3,  -3,  -3,  -3,  -3,  -3,  -3,
	   4,   4,   4,   4,  -4,  -4,  -4,  -4,
	   5,   5,   5,   5,  -5,  -5,  -5,  -5,
	   6,   6,   6,   6,  -6,  -6,  -6,  -6,
	   7,   7,   7,   7,  -7,  -7,  -7,  -7,
	   8,   8,  -8,  -8,
	   9,   9,  -9,  -9,
	  10,  10, -10, -10,
	  11,  11, -11, -11,
	  12,  12, -12, -12,
	  13,  13, -13, -13,
	  14,  14, -14, -14,
	  15,  15, -15, -15,
	  16, -16,
	  17, -17,
	  18, -18,
	  19, -19,
	  20, -20,
	  21, -21,
	  22, -22,
	  23, -23,
	  24, -24,
	  25, -25,
	  26, -26,
	  27, -27,
	  28, -28,
	  29, -29,
	  30, -30,
	  31, -31
};

static const INT8 MV_LEN[0x100] = {
	3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
	3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
	3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
	3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
	3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
	3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
	4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
	4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
	4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
	4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
	6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
	6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8
};

/* */

static INT32 DecodeMV0(MotionVector_t* mv, INT32 bits)
{
	INT32 s;
	INT32 b0, b1;

	b0 = (bits >> 8) & 0xff;
	mv->X = MV_VAL[b0];
	s  = MV_LEN[b0];

	b1 = (bits >> (8 - s)) & 0xff;
	mv->Y = MV_VAL[b1];
	s += MV_LEN[b1];

	return s;
}

static INT32 DecodeMV1(MotionVector_t* mv, INT32 bits)
{
	INT32 b0 = (bits >> 10) & 0x3f;
	INT32 b1 = (bits >>  4) & 0x3f;

	INT8 v[2];

	v[0] = b0 >> 1;
	v[1] = -v[0];
	mv->X = v[b0 & 1];

	v[0] = b1 >> 1;
	v[1] = -v[0];
	mv->Y = v[b1 & 1];

	return 12;
}

/* */

static const INT32 CMV[2] = { 2, 1 };

/* */

struct DCTCoefficientsContext {

	INT8*  Run;
	INT16* Coeff;

	INT32 EOB_Run;

	INT32* BlocksCoded;

}; /* DCTCoefficientsContext */

typedef struct DCTCoefficientsContext DCTCoefficientsContext_t;

/* */

static const INT8 EOB_BITS_LEN[7] = { 0, 0, 0, 2, 3,  4, 12 };
static const INT8 EOB_RUN_BASE[7] = { 1, 2, 3, 4, 8, 16,  0 };

static const INT8 COEFF_SIGN[2] = { 1, -1 };

static const INT8 COEFF_BITS_LEN[32 - 7] = {
	0, 0, 0, 0, 0, 0,
	1, 1, 1, 1,
	2, 3, 4, 5, 6, 10,
	1, 1, 1, 1, 1, 1, 1,
	2, 2
};

static const INT8 COEFF_BASE[32 - 7] = {
	0, 0, 1, -1, 2, -2,
	3, 4, 5, 6,
	7, 9, 13, 21, 37, 69,
	1, 1, 1, 1, 1, 1, 1,
	2, 2
};

static const INT8 RUN_BITS_LEN[32 - 7] = {
	3, 6,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0,
	2, 3, 0, 1
};

static const INT8 RUN_BASE[32 - 7] = {
	0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	1, 2, 3, 4, 5,
	6, 10, 1, 2
};

/* */

static const INT8 COEFFS[5] = { 1, 5, 9, 13, 36 };

/* */

static void DecodeDCCoefficients(FrameDecoder_t* t)
{
	INT32 i;

	INT32 eob = 0;

	const UINT8* c = t->BCoded;

	INT16* dc = t->DC;

	for (i = 0; i < 3; i++) {
		const INT8*  br = t->BRun  [i][0];
		const INT16* bc = t->BCoeff[i][0];

		const UINT8* end = c + t->Index->BC[i];

		const UINT16* bi = t->Index->BIndex[i];

		for (; ; ) {
			INT16 coeff = 0;

			for (; c < end && *c == 0; c++) {
				dc[*(bi++)] = NOT_CODED;
			}

			if (c >= end) {
				break;
			}

			if (eob == 0) {
				INT8  rr = *(br++);
				INT16 cc = *(bc++);
				if (rr == 0) {
					coeff = cc;
				} else if (rr < 0) {
					eob   = cc - 1;
				}

			} else {
				eob--;
			}

			dc[*(bi++)] = coeff;

			c++;
		}

		dc += t->Index->BC[i];
	}
}

/* */

static const INT32 DCP_T[8] = {
	1, 0, 1, 1,
	1, 2, 2, 1
};

/* */

static void UndoDCPrediction(
	FrameDecoder_t* t)
{
	INT32 i;

	INT16* DC = t->DC;

	const UINT8* mode = t->BMode;

	for (i = 0; i < 3; i++) {
		INT32 bx = t->Index->BX[i];
		INT32 by = t->Index->BY[i];

		INT16 last[3] = { 0 };

		INT32 v[4] = { 0 };

		INT32 idx = 0, ni;

		INT32 y;

		for (ni = idx + bx; idx < ni; idx++) {
			if (DC[idx] != NOT_CODED) {
				INT32 type = DCP_T[mode[idx]];
				last[type] = (DC[idx] += last[type]);
			}
		}

		for (y = 1; y < by; y++) {
			ni = idx + bx;

			if (DC[idx] != NOT_CODED) {
				INT32 type = DCP_T[mode[idx]];

				INT32 i2 = idx - bx;
				INT32 i3 = idx - bx + 1;

				if        (DC[i2] != NOT_CODED && DCP_T[mode[i2]] == type) {
					last[type] = (DC[idx] += DC[i2]);
				} else if (DC[i3] != NOT_CODED && DCP_T[mode[i3]] == type) {
					last[type] = (DC[idx] += DC[i3]);
				} else {
					last[type] = (DC[idx] += last[type]);
				}
			}

			idx++;

			for (; idx < ni; idx++) {
				if (DC[idx] != NOT_CODED) {
					INT32 pred;

					INT32 t0 = 0;
					INT32 type = DCP_T[mode[idx]];

					INT32 i0 = idx - 1;
					INT32 i1 = idx - bx - 1;
					INT32 i2 = idx - bx;

					v[0] = DC[i0];
					v[1] = DC[i1];
					v[2] = DC[i2];

					if (v[0] != NOT_CODED && DCP_T[mode[i0]] == type) {
						t0 += 1;
					}
					if (v[1] != NOT_CODED && DCP_T[mode[i1]] == type) {
						t0 += 2;
					}
					if (v[2] != NOT_CODED && DCP_T[mode[i2]] == type) {
						t0 += 4;
					}

					if (idx < ni - 1) {
						INT32 i3 = idx - bx + 1;
						v[3] = DC[i3];
						if (v[3] != NOT_CODED && DCP_T[mode[i3]] == type) {
							t0 += 8;
						}
					}

					switch (t0) {
					case  0:
						pred = last[type];
						break;

					case  1:
					case  3:
						pred = v[0];
						break;

					case  2:
						pred = v[1];
						break;

					case  4:
					case  6:
					case 12:
						pred = v[2];
						break;

					case  8:
						pred = v[3];
						break;

					case  5:
						pred = (v[0] + v[2]) / 2;
						break;

					case  9:
					case 11:
					case 13:
						pred = (75 * v[0] + 53 * v[3]) / 128;
						break;

					case 10:
						pred = (v[1] + v[3]) / 2;
						break;

					case 14:
						pred = (3 * (v[1] + v[3]) + 10 * v[2]) / 16;
						break;

					case  7:
					case 15:
						pred = (29 * (v[0] + v[2]) - 26 * v[1]) / 32;
						if        ((UINT32)(pred - v[2] + 128) > 256) { /* D */
							pred = v[2];
						} else if ((UINT32)(pred - v[0] + 128) > 256) { /* L */
							pred = v[0];
						} else if ((UINT32)(pred - v[1] + 128) > 256) { /* DL */
							pred = v[1];
						}
						break;

					default:
						break;
					}

					last[type] = (DC[idx] += pred);
				}
			}
		}

		DC   += t->Index->BC[i];
		mode += t->Index->BC[i];
	}
}

/* */

#define ARCH_C

#include "QTheoraArch.h"

#define FrameHeader_Decode FrameHeader_Decode_C

#define FrameDecoder_DecodeCodedBlockFlag        FrameDecoder_DecodeCodedBlockFlag_C
#define FrameDecoder_DecodeMacroBlockCodingModes FrameDecoder_DecodeMacroBlockCodingModes_C
#define FrameDecoder_DecodeMotionVectors         FrameDecoder_DecodeMotionVectors_C

#define FrameDecoder_BlockLevelQIDecode FrameDecoder_BlockLevelQIDecode_C

#define FrameDecoder_DecodeBlocks          FrameDecoder_DecodeBlocks_C
#define FrameDecoder_DecodeDCTCoefficients FrameDecoder_DecodeDCTCoefficients_C

#define FrameDecoder_Decode FrameDecoder_Decode_C

#include "FrameDecoder_Impl.h"

#undef ARCH_C

/* */

#define ARCH_X86

#include "QTheoraArch.h"

#define FrameHeader_Decode FrameHeader_Decode_X86

#define FrameDecoder_DecodeCodedBlockFlag        FrameDecoder_DecodeCodedBlockFlag_X86
#define FrameDecoder_DecodeMacroBlockCodingModes FrameDecoder_DecodeMacroBlockCodingModes_X86
#define FrameDecoder_DecodeMotionVectors         FrameDecoder_DecodeMotionVectors_X86

#define FrameDecoder_BlockLevelQIDecode FrameDecoder_BlockLevelQIDecode_X86

#define FrameDecoder_DecodeBlocks          FrameDecoder_DecodeBlocks_X86
#define FrameDecoder_DecodeDCTCoefficients FrameDecoder_DecodeDCTCoefficients_X86

#define FrameDecoder_Decode FrameDecoder_Decode_X86

#include "FrameDecoder_Impl.h"

#undef ARCH_X86

/* */

#define ARCH_MMX

#include "QTheoraArch.h"

#define FrameHeader_Decode FrameHeader_Decode_MMX

#define FrameDecoder_DecodeCodedBlockFlag        FrameDecoder_DecodeCodedBlockFlag_MMX
#define FrameDecoder_DecodeMacroBlockCodingModes FrameDecoder_DecodeMacroBlockCodingModes_MMX
#define FrameDecoder_DecodeMotionVectors         FrameDecoder_DecodeMotionVectors_MMX

#define FrameDecoder_BlockLevelQIDecode FrameDecoder_BlockLevelQIDecode_MMX

#define FrameDecoder_DecodeBlocks          FrameDecoder_DecodeBlocks_MMX
#define FrameDecoder_DecodeDCTCoefficients FrameDecoder_DecodeDCTCoefficients_MMX

#define FrameDecoder_Decode FrameDecoder_Decode_MMX

#include "FrameDecoder_Impl.h"

#undef ARCH_MMX

/* */

BOOL QT_FrameDecoder_Setup(
	FrameDecoder_t*      t,
	const BlockIndex_t*  index,
	const SetupHeader_t* setup,
	MemoryPool_t*        pool)
{
	extern BOOL g_QT_Enable_X86;
	extern BOOL g_QT_Enable_MMX;
	extern BOOL g_QT_Enable_SSE2;

	INT32 i, j;

	/* */

	memset(t, 0, sizeof(FrameDecoder_t));

	t->Index = index;
	t->Setup = setup;

	/* */

	for (i = 0; i < 3; i++) {
		Plane_t* r = t->Plane + i * 3;

		INT32 cb = index->MX * 16 * index->MY * 16;

		UINT8* p = (UINT8*)QT_MemoryPool_Allocate(pool, sizeof(UINT8) * cb);
		if (p == NULL) {
			return FALSE;
		}

		r->Plane = p;
		r->Pitch = index->MX * 16;
		r->CX    = index->MX * 16;
		r->CY    = index->MY * 16;

		for (j = 1; j < 3; j++) {
			p = (UINT8*)QT_MemoryPool_Allocate(pool, sizeof(UINT8) * cb / 4);
			if (p == NULL) {
				return FALSE;
			}

			r[j].Plane = p;
			r[j].Pitch = index->MX * 8;
			r[j].CX    = index->MX * 8;
			r[j].CY    = index->MY * 8;
		}
	}

	for (i = 0; i < 3; i++) {
		t->Frame[i] = t->Plane + i * 3;
	}

	/* */

	if (g_QT_Enable_SSE2 || g_QT_Enable_MMX) {
		t->Reconstructor = (FrameReconstructor_SSE2_t*)QT_MemoryPool_Allocate(pool, sizeof(FrameReconstructor_SSE2_t));
		if (t->Reconstructor == NULL) {
			return FALSE;
		}
	}

	/* */

	t->QIndex[0] = -1;
	t->QIndex[1] = -1;
	t->QIndex[2] = -1;

	t->QCount = -1;

	/* */

	t->SBCoded = (INT8*)QT_MemoryPool_Allocate(pool, sizeof(INT8) * index->SBlocks);
	if (t->SBCoded == NULL) {
		return FALSE;
	}

	t->BCoded = (UINT8*)QT_MemoryPool_Allocate(pool, sizeof(UINT8) * index->Blocks);
	if (t->BCoded == NULL) {
		return FALSE;
	}

	t->BQI = (UINT8*)QT_MemoryPool_Allocate(pool, sizeof(UINT8) * index->Blocks);
	if (t->BQI == NULL) {
		return FALSE;
	}

	t->MBMode = (UINT8*)QT_MemoryPool_Allocate(pool, sizeof(UINT8) * index->MC);
	if (t->MBMode == NULL) {
		return FALSE;
	}

	t->BMode = (UINT8*)QT_MemoryPool_Allocate(pool, sizeof(UINT8) * index->Blocks);
	if (t->BMode == NULL) {
		return FALSE;
	}

	t->MV = (MotionVector_t*)QT_MemoryPool_Allocate(pool, sizeof(MotionVector_t) * index->BC[0]);
	if (t->MV == NULL) {
		return FALSE;
	}

	t->MVC = (MotionVector_t*)QT_MemoryPool_Allocate(pool, sizeof(MotionVector_t) * index->BC[1]);
	if (t->MVC == NULL) {
		return FALSE;
	}

	t->DCTRun = (INT8*)QT_MemoryPool_Allocate(pool, sizeof(INT8) * index->Blocks * 64);
	if (t->DCTRun == NULL) {
		return FALSE;
	}

	t->DCTCoeff = (INT16*)QT_MemoryPool_Allocate(pool, sizeof(INT16) * index->Blocks * 64);
	if (t->DCTCoeff == NULL) {
		return FALSE;
	}

	t->DC = (INT16*)QT_MemoryPool_Allocate(pool, sizeof(INT16) * index->Blocks);
	if (t->DC == NULL) {
		return FALSE;
	}

	/* */

	if (g_QT_Enable_SSE2 || g_QT_Enable_MMX) {
		t->Decode = FrameDecoder_Decode_MMX;
	} else if (g_QT_Enable_X86) {
		t->Decode = FrameDecoder_Decode_X86;
	} else {
		t->Decode = FrameDecoder_Decode_C;
	}

	/* */

	if (g_QT_Enable_SSE2) {
		t->UpdateDequantizeMatrix = QT_UpdateDequantizeMatrix_SSE2;
	} else if (g_QT_Enable_MMX) {
		t->UpdateDequantizeMatrix = QT_UpdateDequantizeMatrix_MMX;
	} else {
		t->UpdateDequantizeMatrix = NULL;
	}

	if (g_QT_Enable_SSE2) {
		t->Reconstruct = QT_ReconstructFrame_SSE2;
	} else if (g_QT_Enable_MMX) {
		t->Reconstruct = QT_ReconstructFrame_MMX;
	} else {
		t->Reconstruct = QT_ReconstructFrame;
	}

	/* */

	return TRUE;
}

/* */

BOOL QT_FrameDecoder_DecodeFrame(
	FrameDecoder_t* t,
	const VOID*     p,
	SIZE_T          size)
{
	extern BOOL g_QT_Available_MMX;

	BOOL b = t->Decode(t, p, size);

	if (g_QT_Available_MMX) {
		_mm_empty();
	}

	return b;
}

/* */

